//////////////////////////////////////////////////////////////////////////////
//Original By http://s2works.homeunix.net/note/inifile.html
////////////////////////

#ifndef __INIFILE_H__
#define __INIFILE_H__

#include <fstream>
#include <sstream>
#include <string>
#include <map>
#include <boost/lexical_cast.hpp>

namespace s2lib
{

class inifile
{
private:
	typedef std::string key_type;
	typedef std::string value_type;
	typedef std::map<key_type, value_type> record_type;
	typedef std::string section_type;
	typedef std::map<section_type, record_type> ini_type;

	std::string m_file;
	ini_type m_ini;

public:
	inifile() {}

	inifile(const std::string &file)
	{
		load(file);
	}

	~inifile() {}

	bool load(const std::string &file)
	{
		if (file.empty())
		{
			return false;
		}

		std::ifstream fs(file.c_str());
		if (fs.fail())
		{
			return false;
		}

		m_file = file;

		m_ini.clear();

		record_type dummy;
		std::string buf;
		section_type section;
		char c;

		while (!fs.eof())
		{
			c = fs.get();
			if(fs.fail())
			{
				continue;
			}

			if (c == '\r' || c == '\n')
			{
				if (buf.empty())
				{
					continue;
				}

				if (buf.at(0) != ';')
				{
					long len = (long)buf.length();

					if (len > 2 && buf.at(0) == '[' && buf.at(len - 1) == ']')
					{
						section = buf.substr(1, len - 2);
						if (!section.empty())
						{
							if (m_ini.count(section) == 0)
							{
								m_ini[section] = dummy;
							}
						}
					}
					else
					{
						long pos = static_cast<long>(buf.find('='));
						key_type key;
						value_type value;

						if (pos != std::string::npos && pos > 0)
						{
							key = buf.substr(0, pos);
							value = buf.substr(pos + 1, len - pos - 1);

							if (!section.empty() && !key.empty())
							{
								m_ini[section][key] = value;
							}
						}
					}
				}
				buf.erase();
			}
			else
			{
				buf += c;
			}
		}
		return true;
	}

	bool save()
	{
		return save(m_file);
	}

	bool save(const std::string &file)
	{
		if (file.empty())
		{
			return false;
		}

		std::ofstream fs(file.c_str());
		if (fs.fail())
		{
			return false;
		}

		ini_type::iterator itm;
		ini_type::iterator itmend = m_ini.end();

		for (itm = m_ini.begin(); itm != itmend; itm++)
		{
			fs << "[" << itm->first.c_str() << "]" << std::endl;

			record_type::iterator its;
			record_type::iterator itsend = itm->second.end();

			for (its = itm->second.begin(); its != itsend; its++)
			{
				fs << its->first.c_str() << "=" << its->second.c_str() << std::endl;
			}
			fs << std::endl;
		}

		return true;
	}

	template<typename T> bool get(const section_type &section, const key_type &key, T &value)
	{
		if (section.empty() || key.empty())
		{
			return false;
		}

		if (m_ini.count(section) == 0 || m_ini[section].count(key) == 0)
		{
			return false;
		}

		try
		{
			value = boost::lexical_cast<T>(m_ini[section][key]);
		}
		catch (boost::bad_lexical_cast &)
		{
			return false;
		}

		return true;
	}

	template<typename T> bool set(const section_type &section, const key_type &key, const T &value)
	{
		if (section.empty() || key.empty())
		{
			return false;
		}

		try
		{
			m_ini[section][key] = boost::lexical_cast<value_type>(value);
		}
		catch (boost::bad_lexical_cast &)
		{
			return false;
		}

		return true;
	}

	bool exists(const section_type &section)
	{
		if (section.empty())
		{
			return false;
		}

		if (m_ini.count(section) == 0)
		{
			return false;
		}

		return true;
	}

	bool exists(const section_type &section, const key_type &key)
	{
		if (section.empty() || key.empty())
		{
			return false;
		}

		if (m_ini.count(section) == 0 || m_ini[section].count(key) == 0)
		{
			return false;
		}

		return true;
	}

	bool remove(const section_type &section)
	{
		if (section.empty())
		{
			return false;
		}

		if (m_ini.count(section) == 0)
		{
			return false;
		}

		if (m_ini.erase(section) > 0)
		{
			return true;
		}

		return false;
	}

	bool remove(const section_type &section, const key_type &key)
	{
		if (section.empty() || key.empty())
		{
			return false;
		}

		if (m_ini.count(section) == 0 || m_ini[section].count(key) == 0)
		{
			return false;
		}

		if (m_ini[section].erase(key) > 0)
		{
			return true;
		}

		return false;
	}
};

} // namespace s2lib

#endif // __INIFILE_H__

