﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Reflection;
using System.Text;
using System.Xml;
using System.Xml.Schema;
using System.Xml.Serialization;

namespace Kasuga
{
    [Serializable]
    public abstract class PluginWrapper<T> : IXmlSerializable where T : IPlugin
    {
        public PluginWrapper() { }

        public PluginWrapper(string baseName, T plugin)
        {
            try
            {
                BaseName = baseName;
                Plugin = plugin;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        [XmlIgnore]
        public string BaseName { get; set; }
        [XmlIgnore]
        public T Plugin { get; set; }

        private static Dictionary<string, XmlSerializer> _serializerCache =
            new Dictionary<string, XmlSerializer>();

        protected abstract T GetPluginByName(string name);

        public XmlSchema GetSchema()
        {
            try
            {
                return null;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return null;
            }
        }

        public void ReadXml(XmlReader reader)
        {
            try
            {
                reader.Read();
                if (reader.LocalName == "BaseName")
                {
                    BaseName = reader.ReadElementString("BaseName");
                    T basePlugin = GetPluginByName(BaseName);
                    Type type = basePlugin.GetType();
                    Plugin = (T)basePlugin.Clone();

                    if (reader.IsEmptyElement)
                    {
                        reader.Read();
                    }
                    else
                    {
                        reader.ReadStartElement("Overrides");
                        while (reader.NodeType != XmlNodeType.EndElement)
                        {
                            PropertyInfo property = type.GetProperty(reader.LocalName);
                            string key = property.DeclaringType.AssemblyQualifiedName + ", " + property.Name;
                            XmlSerializer serializer;
                            if (_serializerCache.ContainsKey(key))
                            {
                                serializer = _serializerCache[key];
                            }
                            else
                            {
                                serializer = new XmlSerializer(property.PropertyType, new XmlRootAttribute(property.Name));
                                _serializerCache.Add(key, serializer);
                            }
                            property.SetValue(Plugin, serializer.Deserialize(reader), null);
                        }
                        reader.ReadEndElement();
                    }
                }
                else if (reader.LocalName == "TypeName")
                {
                    BaseName = string.Empty;
                    Type type;
                    {
                        Func<AssemblyName, Assembly> assemblyResolver = (AssemblyName assemblyName) =>
                        {
                            string directoryPath =
                                Path.GetDirectoryName(
                                Assembly.GetExecutingAssembly().GetModules()[0].FullyQualifiedName)
                                + "\\Plugins";
                            if (!Directory.Exists(directoryPath))
                            {
                                throw new DirectoryNotFoundException();
                            }
                            string[] dlls = Directory.GetFiles(directoryPath, assemblyName.Name + ".dll", SearchOption.AllDirectories);
                            if (dlls.Length <= 0)
                            {
                                throw new DllNotFoundException();
                            }
                            return Assembly.LoadFrom(dlls[0]);
                        };
                        Func<Assembly, string, bool, Type> typeResolver = (Assembly assembly, string typeName, bool ignoreCase) =>
                        {
                            return assembly.GetType(typeName, true, ignoreCase);
                        };
                        type = Type.GetType(reader.ReadElementString("TypeName"), assemblyResolver, typeResolver, true);
                    }
                    XmlSerializer serializer = new XmlSerializer(type);
                    Plugin = (T)serializer.Deserialize(reader);
                }
                reader.Read();
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void WriteXml(XmlWriter writer)
        {
            try
            {
                T basePlugin = GetPluginByName(BaseName);
                Type type = Plugin.GetType();
                if (basePlugin != null && type == basePlugin.GetType())
                {
                    writer.WriteElementString("BaseName", BaseName);
                    writer.WriteStartElement("Overrides");
                    PropertyInfo[] properties = type.GetProperties();
                    foreach (PropertyInfo property in properties)
                    {
                        bool ignore;
                        {
                            XmlIgnoreAttribute attribute = (XmlIgnoreAttribute)
                                Attribute.GetCustomAttribute(property, typeof(XmlIgnoreAttribute));
                            ignore = attribute != null;
                        }
                        if (!ignore && property.CanRead && property.CanWrite
                            && !property.GetValue(Plugin, null).Equals(property.GetValue(basePlugin, null)))
                        {
                            string key = property.DeclaringType.AssemblyQualifiedName + ", " + property.Name;
                            XmlSerializer serializer;
                            if (_serializerCache.ContainsKey(key))
                            {
                                serializer = _serializerCache[key];
                            }
                            else
                            {
                                serializer = new XmlSerializer(property.PropertyType, new XmlRootAttribute(property.Name));
                                _serializerCache.Add(key, serializer);
                            }
                            serializer.Serialize(writer, property.GetValue(Plugin, null));
                        }
                    }
                    writer.WriteEndElement();
                }
                else
                {
                    writer.WriteElementString("TypeName", type.AssemblyQualifiedName);
                    XmlSerializer serializer = new XmlSerializer(type);
                    serializer.Serialize(writer, Plugin);
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }
    }
}
