﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Dynamic;
using System.IO;
using System.Reflection;

namespace Kasuga
{
    public static class PluginManager
    {
        static PluginManager()
        {
            FormatPlugins = new List<PluginTypeInfo>();
            WordArrangementPlugins = new List<PluginTypeInfo>();
            LineArrangementPlugins = new List<PluginTypeInfo>();
            PageArrangementPlugins = new List<PluginTypeInfo>();
        }

        public static List<PluginTypeInfo> FormatPlugins { get; set; }
        public static List<PluginTypeInfo> WordArrangementPlugins { get; set; }
        public static List<PluginTypeInfo> LineArrangementPlugins { get; set; }
        public static List<PluginTypeInfo> PageArrangementPlugins { get; set; }

        public static List<PluginTypeInfo> AllPlugins
        {
            get
            {
                try
                {
                    List<PluginTypeInfo> plugins = new List<PluginTypeInfo>();
                    plugins.AddRange(FormatPlugins);
                    plugins.AddRange(WordArrangementPlugins);
                    plugins.AddRange(LineArrangementPlugins);
                    plugins.AddRange(PageArrangementPlugins);
                    return plugins;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        public static void LoadPlugins()
        {
            try
            {
                FormatPlugins.Clear();
                WordArrangementPlugins.Clear();
                LineArrangementPlugins.Clear();
                PageArrangementPlugins.Clear();

                string directoryPath =
                    Path.GetDirectoryName(
                    Assembly.GetExecutingAssembly().GetModules()[0].FullyQualifiedName)
                    + "\\Plugins";
                if (!Directory.Exists(directoryPath))
                {
                    return;
                }
                string[] dlls = Directory.GetFiles(directoryPath, "*.dll", SearchOption.AllDirectories);
                foreach (string dll in dlls)
                {
                    Assembly assembly;
                    try
                    {
                        assembly = Assembly.LoadFrom(dll);
                    }
                    catch
                    {
                        continue;
                    }
                    foreach (Type type in assembly.GetTypes())
                    {
                        if (type.IsClass && type.IsPublic && !type.IsAbstract
                            && type.GetInterface(typeof(IPlugin).FullName) != null)
                        {
                            PluginTypeInfo plugin = new PluginTypeInfo();
                            plugin.Type = type;
                            {
                                DisplayNameAttribute attribute =
                                    (DisplayNameAttribute)Attribute.GetCustomAttribute(type, typeof(DisplayNameAttribute));
                                plugin.Name = attribute.DisplayName ?? type.FullName;
                            }
                            {
                                DescriptionAttribute attribute =
                                    (DescriptionAttribute)Attribute.GetCustomAttribute(type, typeof(DescriptionAttribute));
                                plugin.Description = attribute.Description ?? string.Empty;
                            }

                            if (type.GetInterface(typeof(IFormat).FullName) != null)
                            {
                                FormatPlugins.Add(plugin);
                                CatalogManager.AddStaticPlugin<IFormat>(type, CatalogManager.FormatCatalog);
                            }
                            if (type.GetInterface(typeof(IWordArrangement).FullName) != null)
                            {
                                WordArrangementPlugins.Add(plugin);
                                CatalogManager.AddStaticPlugin<IWordArrangement>(type, CatalogManager.WordArrangementCatalog);
                            }
                            if (type.GetInterface(typeof(ILineArrangement).FullName) != null)
                            {
                                LineArrangementPlugins.Add(plugin);
                                CatalogManager.AddStaticPlugin<ILineArrangement>(type, CatalogManager.LineArrangementCatalog);
                            }
                            if (type.GetInterface(typeof(IPageArrangement).FullName) != null)
                            {
                                PageArrangementPlugins.Add(plugin);
                                CatalogManager.AddStaticPlugin<IPageArrangement>(type, CatalogManager.PageArrangementCatalog);
                            }
                        }
                    }
                }

                {
                    List<Type> types = new List<Type>();
                    foreach (PluginTypeInfo plugin in FormatPlugins)
                    {
                        types.Add(plugin.Type);
                    }
                    FormatTypeListConverter.Types = types.ToArray();
                    FormatTypeListConverter.Plugins = FormatPlugins;
                }
                {
                    List<Type> types = new List<Type>();
                    foreach (PluginTypeInfo plugin in WordArrangementPlugins)
                    {
                        types.Add(plugin.Type);
                    }
                    WordArrangementTypeListConverter.Types = types.ToArray();
                    WordArrangementTypeListConverter.Plugins = WordArrangementPlugins;
                }
                {
                    List<Type> types = new List<Type>();
                    foreach (PluginTypeInfo plugin in LineArrangementPlugins)
                    {
                        types.Add(plugin.Type);
                    }
                    LineArrangementTypeListConverter.Types = types.ToArray();
                    LineArrangementTypeListConverter.Plugins = LineArrangementPlugins;
                }
                {
                    List<Type> types = new List<Type>();
                    foreach (PluginTypeInfo plugin in PageArrangementPlugins)
                    {
                        types.Add(plugin.Type);
                    }
                    PageArrangementTypeListConverter.Types = types.ToArray();
                    PageArrangementTypeListConverter.Plugins = PageArrangementPlugins;
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }
    }
}
