﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Reflection;
using System.Xml.Serialization;

namespace Kasuga
{
    [Serializable]
    [TypeConverter(typeof(KsgSubtitleElementConverter))]
    public class KsgPage : ISubtitleElement, IHasFormatDictionary, IHasWordArrangementDictionary, IHasLineArrangementDictionary, IHasPageArrangementDictionary
    {
        public KsgPage()
        {
            try
            {
                Lines = new List<KsgLine>();
                BasePageArrangementName = CatalogManager.PageArrangementCatalog[0].Name;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void Initialize(KsgPart parentPart)
        {
            try
            {
                _parentPart = parentPart;
                foreach (KsgLine line in Lines)
                {
                    line.Initialize(this);
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        private string _basePageArrangementName;

        [Category("ページ内要素の配置")]
        [DisplayName("ベース")]
        [XmlIgnore]
        public string BasePageArrangementName
        {
            get
            {
                try
                {
                    return _basePageArrangementName;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return string.Empty;
                }
            }
            set
            {
                try
                {
                    CatalogItem<IPageArrangement> item = CatalogManager.PageArrangementCatalog.Find((CatalogItem<IPageArrangement> listItem) =>
                    {
                        return listItem.Name == value;
                    });
                    if (item != null)
                    {
                        _basePageArrangementName = item.Name;
                        PageArrangement = (IPageArrangement)item.Plugin.Clone();
                    }
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public IPageArrangement BasePageArrangement
        {
            get
            {
                try
                {
                    return Elements.GetBasePageArrangement(this);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Category("ページ内要素の配置")]
        [DisplayName("タイプ")]
        [TypeConverter(typeof(PageArrangementTypeListConverter))]
        [XmlIgnore]
        public Type PageArrangementType
        {
            get
            {
                try
                {
                    return PageArrangement != null ? PageArrangement.GetType() : null;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public IPageArrangement PageArrangement { get; set; }

        [Browsable(false)]
        public PageArrangementWrapper PageArrangementWrapper
        {
            get
            {
                try
                {
                    return new PageArrangementWrapper(BasePageArrangementName, PageArrangement);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
            set
            {
                try
                {
                    PageArrangement = value.Plugin;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        [XmlIgnore]
        public Dictionary<string, object> PageArrangementDictionary { get; set; }
        [Browsable(false)]
        public List<KsgLine> Lines { get; set; }
        private KsgPart _parentPart;

        [Browsable(false)]
        [XmlIgnore]
        public KsgSubtitle ParentSubtitle
        {
            get
            {
                try
                {
                    return ParentPart.ParentSubtitle;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public KsgPart ParentPart
        {
            get
            {
                try
                {
                    return _parentPart;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public List<KsgWord> Words
        {
            get
            {
                try
                {
                    List<KsgWord> words = new List<KsgWord>();
                    foreach (KsgLine line in Lines)
                    {
                        words.AddRange(line.Words);
                    }
                    return words;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public List<KsgCharacter> MainTextCharacters
        {
            get
            {
                try
                {
                    List<KsgCharacter> characters = new List<KsgCharacter>();
                    foreach (KsgLine line in Lines)
                    {
                        characters.AddRange(line.MainTextCharacters);
                    }
                    return characters;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public List<KsgCharacter> RubyTextCharacters
        {
            get
            {
                try
                {
                    List<KsgCharacter> characters = new List<KsgCharacter>();
                    foreach (KsgLine line in Lines)
                    {
                        characters.AddRange(line.RubyTextCharacters);
                    }
                    return characters;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public List<KsgCharacter> Characters
        {
            get
            {
                try
                {
                    List<KsgCharacter> characters = new List<KsgCharacter>();
                    foreach (KsgLine line in Lines)
                    {
                        characters.AddRange(line.Characters);
                    }
                    return characters;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Category("行内要素の配置")]
        [DisplayName("ベース")]
        [XmlIgnore]
        public string BaseLineArrangementName
        {
            get
            {
                try
                {
                    return Elements.GetBaseLineArrangementName(Lines);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
            set
            {
                try
                {
                    Elements.SetBaseLineArrangementName(Lines, value);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public ILineArrangement BaseLineArrangement
        {
            get
            {
                try
                {
                    return Elements.GetBaseLineArrangement(this);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Category("行内要素の配置")]
        [DisplayName("タイプ")]
        [TypeConverter(typeof(LineArrangementTypeListConverter))]
        [XmlIgnore]
        public Type LineArrangementType
        {
            get
            {
                try
                {
                    return Elements.GetLineArrangementType(Lines);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [XmlIgnore]
        public Dictionary<string, object> LineArrangementDictionary { get; set; }

        [Category("単語内要素の配置")]
        [DisplayName("ベース")]
        [XmlIgnore]
        public string BaseWordArrangementName
        {
            get
            {
                try
                {
                    return Elements.GetBaseWordArrangementName(Words);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
            set
            {
                try
                {
                    Elements.SetBaseWordArrangementName(Words, value);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public IWordArrangement BaseWordArrangement
        {
            get
            {
                try
                {
                    return Elements.GetBaseWordArrangement(this);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Category("単語内要素の配置")]
        [DisplayName("タイプ")]
        [TypeConverter(typeof(WordArrangementTypeListConverter))]
        [XmlIgnore]
        public Type WordArrangementType
        {
            get
            {
                try
                {
                    return Elements.GetWordArrangementType(Words);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [XmlIgnore]
        public Dictionary<string, object> WordArrangementDictionary { get; set; }

        [Category("書式")]
        [DisplayName("ベース")]
        [XmlIgnore]
        public string BaseFormatName
        {
            get
            {
                try
                {
                    return Elements.GetBaseFormatName(Characters);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
            set
            {
                try
                {
                    Elements.SetBaseFormatName(Characters, value);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public IFormat BaseFormat
        {
            get
            {
                try
                {
                    return Elements.GetBaseFormat(this);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Category("書式")]
        [DisplayName("タイプ")]
        [TypeConverter(typeof(FormatTypeListConverter))]
        [XmlIgnore]
        public Type FormatType
        {
            get
            {
                try
                {
                    return Elements.GetFormatType(Characters);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [XmlIgnore]
        public Dictionary<string, object> FormatDictionary { get; set; }

        [Browsable(false)]
        [XmlIgnore]
        public GraphicsPath Path
        {
            get
            {
                try
                {
                    GraphicsPath path = new GraphicsPath();
                    foreach (KsgCharacter character in Characters)
                    {
                        GraphicsPath charactersPath = character.Path;
                        path.AddPath(charactersPath, false);
                        charactersPath.Dispose();
                    }
                    return path;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        public void PageArrange()
        {
            try
            {
                PageArrangement.Arrange(this);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void LineArrange()
        {
            try
            {
                foreach (KsgLine line in Lines)
                {
                    line.LineArrange();
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void WordArrange()
        {
            try
            {
                foreach (KsgWord word in Words)
                {
                    word.WordArrange();
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void DrawForPreview(bool isWiped)
        {
            try
            {
                SortedDictionary<int, Bitmap> bitmaps = new SortedDictionary<int, Bitmap>();
                foreach (KsgCharacter character in Characters)
                {
                    character.DrawForPreview(bitmaps, isWiped);
                }
                Graphics graphics = ParentSubtitle.Graphics;
                graphics.Clear(ParentSubtitle.BackColor);
                foreach (Bitmap bitmap in bitmaps.Values)
                {
                    graphics.DrawImage(bitmap, new Point(0, 0));
                    bitmap.Dispose();
                }
                graphics.Dispose();
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void ExportToAss(
            List<Ass.AssStyle> styles,
            List<Ass.AssEvent> events)
        {
            try
            {
                foreach (KsgCharacter character in Characters)
                {
                    character.ExportToAss(styles, events);
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void SetFormatProperty(string propertyName, object value, bool tooParent)
        {
            try
            {
                foreach (KsgLine line in Lines)
                {
                    line.SetFormatProperty(propertyName, value, false);
                }
                RefreshFormatProperty(false, tooParent);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void RefreshFormatProperty(bool tooChildren, bool tooParent)
        {
            try
            {
                if (tooChildren)
                {
                    foreach (KsgLine line in Lines)
                    {
                        line.RefreshFormatProperty(true, false);
                    }
                }
                FormatDictionary = Elements.GetFormat(Characters);
                if (tooParent && ParentPart != null)
                {
                    ParentPart.RefreshFormatProperty(false, true);
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void SetWordArrangementProperty(string propertyName, object value, bool tooParent)
        {
            try
            {
                foreach (KsgLine line in Lines)
                {
                    line.SetWordArrangementProperty(propertyName, value, false);
                }
                RefreshWordArrangementProperty(false, tooParent);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void RefreshWordArrangementProperty(bool tooChildren, bool tooParent)
        {
            try
            {
                if (tooChildren)
                {
                    foreach (KsgLine line in Lines)
                    {
                        line.RefreshWordArrangementProperty(true, false);
                    }
                }
                WordArrangementDictionary = Elements.GetWordArrangement(Words);
                if (tooParent && ParentPart != null)
                {
                    ParentPart.RefreshWordArrangementProperty(false, true);
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void SetLineArrangementProperty(string propertyName, object value, bool tooParent)
        {
            try
            {
                foreach (KsgLine line in Lines)
                {
                    line.SetLineArrangementProperty(propertyName, value, false);
                }
                RefreshLineArrangementProperty(false, tooParent);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void RefreshLineArrangementProperty(bool tooChildren, bool tooParent)
        {
            try
            {
                if (tooChildren)
                {
                    foreach (KsgLine line in Lines)
                    {
                        line.RefreshLineArrangementProperty(true, false);
                    }
                }
                LineArrangementDictionary = Elements.GetLineArrangement(Lines);
                if (tooParent && ParentPart != null)
                {
                    ParentPart.RefreshLineArrangementProperty(false, true);
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void SetPageArrangementProperty(string propertyName, object value, bool tooParent)
        {
            try
            {
                PropertyDescriptorCollection descriptors = TypeDescriptor.GetProperties(PageArrangementType);
                PropertyDescriptor descriptor = descriptors.Find(propertyName, true);
                if (descriptor != null && descriptor.Name == propertyName && !descriptor.IsReadOnly && value != null)
                {
                    descriptor.SetValue(PageArrangement, value);
                }
                RefreshPageArrangementProperty(false, tooParent);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void RefreshPageArrangementProperty(bool tooChildren, bool tooParent)
        {
            try
            {
                PropertyDescriptorCollection descriptors = TypeDescriptor.GetProperties(PageArrangementType);
                Dictionary<string, object> dictionary = new Dictionary<string, object>();
                foreach (PropertyDescriptor descriptor in descriptors)
                {
                    dictionary.Add(descriptor.Name, descriptor.GetValue(PageArrangement));
                }
                PageArrangementDictionary = dictionary;
                if (tooParent && ParentPart != null)
                {
                    ParentPart.RefreshPageArrangementProperty(false, true);
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }
    }
}
