﻿using System;
using System.Drawing;
using System.Reflection;
using System.Text;
using System.Text.RegularExpressions;

namespace Kasuga.Ass
{
    public class AssStyle
    {
        public AssStyle(
            string name,
            string fontName,
            float fontSize,
            Color primaryColour,
            Color secondaryColour,
            Color outlineColour,
            Color backColour,
            bool isBold,
            bool isItalic,
            bool hasUnderline,
            bool hasStrikeout,
            float scaleX,
            float scaleY,
            float spacing,
            float angle,
            AssBorderStyleKind borderStyle,
            float outline,
            float shadow,
            ContentAlignment alignment,
            int marginL,
            int marginR,
            int marginV,
            byte textEncoding)
        {
            try
            {
                Name = name;
                FontName = fontName;
                FontSize = fontSize;
                PrimaryColour = primaryColour;
                SecondaryColour = secondaryColour;
                OutlineColour = outlineColour;
                BackColour = backColour;
                IsBold = isBold;
                IsItalic = isItalic;
                HasUnderline = hasUnderline;
                HasStrikeout = hasStrikeout;
                ScaleX = scaleX;
                ScaleY = scaleY;
                Spacing = spacing;
                Angle = angle;
                BorderStyle = borderStyle;
                Outline = outline;
                Shadow = shadow;
                Alignment = alignment;
                MarginL = marginL;
                MarginR = marginR;
                MarginV = marginV;
                TextEncoding = textEncoding;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public string Name { get; set; }
        public string FontName { get; set; }
        public float FontSize { get; set; }
        public Color PrimaryColour { get; set; }
        public Color SecondaryColour { get; set; }
        public Color OutlineColour { get; set; }
        public Color BackColour { get; set; }
        public bool IsBold { get; set; }
        public bool IsItalic { get; set; }
        public bool HasUnderline { get; set; }
        public bool HasStrikeout { get; set; }
        public float ScaleX { get; set; }
        public float ScaleY { get; set; }
        public float Spacing { get; set; }
        public float Angle { get; set; }
        public AssBorderStyleKind BorderStyle { get; set; }
        public float Outline { get; set; }
        public float Shadow { get; set; }
        public ContentAlignment Alignment { get; set; }
        public int MarginL { get; set; }
        public int MarginR { get; set; }
        public int MarginV { get; set; }
        public byte TextEncoding { get; set; }

        private const string HEADER = "Style: ";

        public FontStyle FontStyle
        {
            get
            {
                try
                {
                    FontStyle style = 0;
                    if (IsBold) { style |= FontStyle.Bold; }
                    if (IsItalic) { style |= FontStyle.Italic; }
                    if (HasUnderline) { style |= FontStyle.Underline; }
                    if (HasStrikeout) { style |= FontStyle.Strikeout; }
                    return style;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return FontStyle.Regular;
                }
            }
            set
            {
                try
                {
                    IsBold = (value & FontStyle.Bold) > 0;
                    IsItalic = (value & FontStyle.Italic) > 0;
                    HasUnderline = (value & FontStyle.Underline) > 0;
                    HasStrikeout = (value & FontStyle.Strikeout) > 0;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        public Font Font
        {
            get
            {
                try
                {
                    string familyName;
                    bool isVertical;
                    if (FontName.Substring(0, 1) == "@")
                    {
                        familyName = FontName.Substring(1);
                        isVertical = true;
                    }
                    else
                    {
                        familyName = FontName;
                        isVertical = false;
                    }
                    return new Font(
                        familyName,
                        FontSize,
                        FontStyle,
                        GraphicsUnit.Pixel,
                        TextEncoding,
                        isVertical);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
            set
            {
                try
                {
                    FontName = value.FontFamily.Name;
                    if (value.GdiVerticalFont)
                    {
                        FontName = "@" + FontName;
                    }
                    FontSize = value.GetHeight();
                    FontStyle = value.Style;
                    TextEncoding = value.GdiCharSet;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        public int Bold
        {
            get
            {
                try
                {
                    return IsBold ? -1 : 0;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return 0;
                }
            }
            set
            {
                try
                {
                    IsBold = value != 0;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        public int Italic
        {
            get
            {
                try
                {
                    return IsItalic ? -1 : 0;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return 0;
                }
            }
            set
            {
                try
                {
                    IsItalic = value != 0;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        public int Underline
        {
            get
            {
                try
                {
                    return HasUnderline ? -1 : 0;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return 0;
                }
            }
            set
            {
                try
                {
                    HasUnderline = value != 0;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        public int Strikeout
        {
            get
            {
                try
                {
                    return HasStrikeout ? -1 : 0;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return 0;
                }
            }
            set
            {
                try
                {
                    HasStrikeout = value != 0;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        public string ToLine()
        {
            try
            {
                StringBuilder builder = new StringBuilder(HEADER);
                builder.Append(Name);
                builder.Append(",");
                builder.Append(FontName);
                builder.Append(",");
                builder.Append(FontSize);
                builder.Append(",");
                builder.Append(AssColorConverter.ConvertToAssAbgrString(PrimaryColour));
                builder.Append(",");
                builder.Append(AssColorConverter.ConvertToAssAbgrString(SecondaryColour));
                builder.Append(",");
                builder.Append(AssColorConverter.ConvertToAssAbgrString(OutlineColour));
                builder.Append(",");
                builder.Append(AssColorConverter.ConvertToAssAbgrString(BackColour));
                builder.Append(",");
                builder.Append(Bold);
                builder.Append(",");
                builder.Append(Italic);
                builder.Append(",");
                builder.Append(Underline);
                builder.Append(",");
                builder.Append(Strikeout);
                builder.Append(",");
                builder.Append((int)(ScaleX * 100));
                builder.Append(",");
                builder.Append((int)(ScaleY * 100));
                builder.Append(",");
                builder.Append(Spacing);
                builder.Append(",");
                builder.Append(Angle);
                builder.Append(",");
                builder.Append((int)BorderStyle);
                builder.Append(",");
                builder.Append(Outline);
                builder.Append(",");
                builder.Append(Shadow);
                builder.Append(",");
                {
                    int an = 0;
                    switch (Alignment)
                    {
                        case ContentAlignment.BottomLeft:
                            an = 1;
                            break;
                        case ContentAlignment.BottomCenter:
                            an = 2;
                            break;
                        case ContentAlignment.BottomRight:
                            an = 3;
                            break;
                        case ContentAlignment.MiddleLeft:
                            an = 4;
                            break;
                        case ContentAlignment.MiddleCenter:
                            an = 5;
                            break;
                        case ContentAlignment.MiddleRight:
                            an = 6;
                            break;
                        case ContentAlignment.TopLeft:
                            an = 7;
                            break;
                        case ContentAlignment.TopCenter:
                            an = 8;
                            break;
                        case ContentAlignment.TopRight:
                            an = 9;
                            break;
                    }
                    builder.Append(an);
                }
                builder.Append(",");
                builder.Append(MarginL);
                builder.Append(",");
                builder.Append(MarginR);
                builder.Append(",");
                builder.Append(MarginV);
                builder.Append(",");
                builder.Append(TextEncoding);
                return builder.ToString();
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return string.Empty;
            }
        }

        public bool EqualsExcludeName(AssStyle style)
        {
            try
            {
                return FontName == style.FontName
                    && FontSize == style.FontSize
                    && PrimaryColour == style.PrimaryColour
                    && SecondaryColour == style.SecondaryColour
                    && OutlineColour == style.OutlineColour
                    && BackColour == style.BackColour
                    && IsBold == style.IsBold
                    && IsItalic == style.IsItalic
                    && HasUnderline == style.HasUnderline
                    && HasStrikeout == style.HasStrikeout
                    && ScaleX == style.ScaleX
                    && ScaleY == style.ScaleY
                    && Spacing == style.Spacing
                    && Angle == style.Angle
                    && BorderStyle == style.BorderStyle
                    && Outline == style.Outline
                    && Shadow == style.Shadow
                    && Alignment == style.Alignment
                    && MarginL == style.MarginL
                    && MarginR == style.MarginR
                    && MarginV == style.MarginV
                    && TextEncoding == style.TextEncoding;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return false;
            }
        }
    }
}
