#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# This file is part of Karesansui Core.
#
# Copyright (C) 2010 HDE, Inc.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#

import sqlalchemy
from sqlalchemy.orm import mapper, clear_mappers, relation, backref
import karesansui
import karesansui.db.model
from karesansui.lib.const import DEFAULT_LANGS

def get_watch_table(metadata, now):
    """<comment-ja>
    監視(watch)のテーブル定義を返却します。
    @param metadata: MetaData
    @type metadata: sqlalchemy.schema.MetaData
    @param now: now
    @type now: Datatime
    @return: sqlalchemy.schema.Table
    </comment-ja>
    <comment-en>
    TODO: English Comment
    </comment-en>
    """
    return sqlalchemy.Table('watch', metadata,
                            sqlalchemy.Column('id', sqlalchemy.Integer,
                                              primary_key=True,
                                              autoincrement=True,
                                              ),
                            sqlalchemy.Column('machine_id', sqlalchemy.Integer,
                                              sqlalchemy.ForeignKey('machine.id'),
                                              ),
                            sqlalchemy.Column('created_user_id', sqlalchemy.Integer,
                                              sqlalchemy.ForeignKey('user.id'),
                                              ),
                            sqlalchemy.Column('modified_user_id', sqlalchemy.Integer,
                                              sqlalchemy.ForeignKey('user.id'),
                                              ),
                            sqlalchemy.Column('name', sqlalchemy.String(256),
                                              nullable=False,
                                              ),
                            sqlalchemy.Column('plugin', sqlalchemy.String(256),
                                              nullable=False,
                                              ),                                              
                            sqlalchemy.Column('plugin_selector', sqlalchemy.Text,
                                              nullable=False,
                                              ),
                            sqlalchemy.Column('karesansui_version', sqlalchemy.String(12),
                                              nullable=False,
                                              ),
                            sqlalchemy.Column('collectd_version', sqlalchemy.String(12),
                                              nullable=False,
                                              ),
                            sqlalchemy.Column('is_deleted', sqlalchemy.Boolean,
                                              default=False,
                                              ),
                            sqlalchemy.Column('created', sqlalchemy.DateTime,
                                              default=now,
                                              ),
                            sqlalchemy.Column('modified', sqlalchemy.DateTime,
                                              default=now,
                                              onupdate=now,
                                              ),
                            )

class Watch(karesansui.db.model.Model):
    """<comment-ja>
    watchテーブルモデルクラス
    </comment-ja>
    <comment-en>
    TODO: English Comment
    </comment-en>
    """
    def __init__(self, created_user, modified_user,
                 machine_id, name, plugin, plugin_selector,
                 karesansui_version, collectd_version,
                 is_deleted=False
                 ):
        """<comment-ja>
        @param created_user: 作成者
        @type created_user: User
        @param modified_user: 最終更新者
        @type modified_user: User
        @param machine_id:
        @type machine_id: machine ID
        @param plugin: 
        @type plugin: str
        @param plugin_selector: 
        @type plugin_selector: str(Text)
        @param karesansui_version: 作成したKaresansuiのバージョン
        @type karesansui_version: str
        @param collectd_version: 作成したCollectdのバージョン
        @type collectd_version: str
        </comment-ja>
        <comment-en>
        TODO: English Comment
        </comment-en>
        """
        self.created_user = created_user
        self.modified_user = modified_user
        self.machine_id = machine_id
        self.name = name
        self.plugin = plugin
        self.plugin_selector = plugin_selector
        self.karesansui_version = karesansui_version
        self.collectd_version = collectd_version
        self.is_deleted = is_deleted

    def get_json(self, languages):
        import web
        ret = {}
        ret["id"] = self.id
        ret["machine"] = self.machine.get_json(languages)
        ret["name"] = self.name
        ret["plugin"] = self.plugin
        ret["plugin_selector"] = self.plugin_selector
        ret["created_user"] = self.created_user.get_json(languages)
        ret["modified_user"] = self.modified_user.get_json(languages)
        ret["created"] = self.created.strftime(
            DEFAULT_LANGS[languages]['DATE_FORMAT'][1])
        ret["created_user_id"] = self.created_user_id
        ret["modified"] = self.modified.strftime(
            DEFAULT_LANGS[languages]['DATE_FORMAT'][1])
        ret["modified_user_id"] = self.modified_user_id
        ret["is_deleted"] = self.is_deleted
        return ret

    def __repr__(self):
        return "Watch<'%s, %s'>" % (self.name,
                                        self.plugin,
                                        )
        
def reload_mapper(metadata, now):
    """<comment-ja>
    Watch(Model)のマッパーをリロードします。
    @param metadata: リロードしたいMetaData
    @type metadata: sqlalchemy.schema.MetaData
    @param now: now
    @type now: Datatime
    </comment-ja>
    <comment-en>
    TODO: English Comment
    </comment-en>
    """
    t_watch = get_watch_table(metadata, now)
    t_user = metadata.tables['user']
    t_machine = metadata.tables['machine']
    
    mapper(Watch, t_watch, properties={
        'created_user' : relation(karesansui.db.model.user.User,
                                  primaryjoin=t_watch.c.created_user_id==t_user.c.id),
        'modified_user' : relation(karesansui.db.model.user.User,
                                  primaryjoin=t_watch.c.modified_user_id==t_user.c.id),
        'machine' : relation(karesansui.db.model.machine.Machine,
                             primaryjoin=t_watch.c.machine_id==t_machine.c.id),
        })

if __name__ == '__main__':
    import sqlalchemy.orm
    bind_name = 'sqlite:///:memory:'
    engine = sqlalchemy.create_engine(bind_name,
                                      encoding="utf-8",
                                      convert_unicode=True,
                                      #assert_unicode='warn', # DEBUG
                                      echo=True,
                                      echo_pool=False
                                      )
    
    metadata = sqlalchemy.MetaData(bind=engine)

    # relation
    karesansui.db.model.user.reload_mapper(metadata)

    reload_mapper(metadata)
    import pdb; pdb.set_trace()
    metadata.drop_all()
    metadata.create_all()
    Session = sqlalchemy.orm.sessionmaker(bind=engine, autoflush=False)
    session = Session()
    print ""
    # INSERT
    # SELECT One
    # UPDATE
    # DELETE

