#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# This file is part of Karesansui Core.
#
# Copyright (C) 2010 HDE, Inc.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#

"""
collectdからデータを受け取り、アラートを出すかどうか判断する

アラートあり：メール送信用、シェル実行用のモジュールへデータを投げる
TODO: karesansui画面にも通知する

アラートなし：そのまま終了(リターンいらない)


Example: collectd.d/python.conf
----- snip ----- snip ----- snip -----
# "LoadPlugin python" だけだと、システムのpythonライブラリが使用できない
# 下記のように"<LoadPlugin python>" でGlobalsパラメータの設定が必要
<LoadPlugin python>
        Globals true
</LoadPlugin>

<Plugin python>
        ModulePath "/opt/karesansui/lib/python/karesansui/lib/collectd"
        Encoding utf-8
        LogTraces true
        #Interactive true
        Import "notification"

        <Module "notification">
                CountupDBPath "/var/tmp/notify_count.db"
                Logfile "/var/log/hde-collectd/notification.log"
        </Module>
</Plugin>
----- snip ----- snip ----- snip -----

Example: collectd.d/filter.conf
----- snip ----- snip ----- snip -----
# Messageオプションの値をdictの文字列にすることで、pythonスクリプト側で
# exec()の展開をすると各種データが取得できるようになる。
LoadPlugin match_regex
LoadPlugin match_value
LoadPlugin target_notification

PostCacheChain     PostTestChain
<Chain "PostTestChain">
    Target "write"
    <Rule "memory_cached_exceeded">
        <Target "notification">
            #Message            "Oops, the %{plugin} %{type_instance} memory_size is currently %{ds:value}!"
            Message            "{'host':'%{host}','plugin':'%{plugin}','plugin_instance':'%{plugin_instance}','type':'%{type}','type_instance':'%{type_instance}','value':'%{ds:value}','extra':'{}'}"
            Severity           "WARNING"
        </Target>
        <Match "regex">
            TypeInstance       "^cached$"
            Plugin             "^memory$"
        </Match>
        <Match "value">
            Satisfy            "Any"
            Min                47000000
        </Match>
    </Rule>
</Chain>
----- snip ----- snip ----- snip -----

Example: collectd.d/threshold.conf
----- snip ----- snip ----- snip -----
# 閾値を越えるとnotificationプラグインには、以下のメッセージが送られる。
# Host %{host}, plugin %{plugin} type %{type} (instance {%type_instance}):
# Data source "{%ds.name}" is currently {%ds.value}.
# 1. That is (below|above) the (failure|warning) threshold of ({%threshold_min}|{%thresold_max})%?.
# 2. That is within the (failure|warning) region of {%threshold_min}%? and {%threshold_max}%?.
<Threshold>
    <Plugin "df">
        <Type "df">
            DataSource "used"
            WarningMax 10
            FailureMax 20
            Percentage true
            Persist true
        </Type>
    </Plugin>
    <Plugin "load">
        <Type "load">
            DataSource "shortterm"
            WarningMax    0.01
            FailureMax    0.40
            Persist true
        </Type>
    </Plugin>
</Threshold>
----- snip ----- snip ----- snip -----
"""

import collectd
import sys, time, os.path, re

search_paths = ["/opt/karesansui/lib/python",]
for _path in search_paths:
    if os.path.exists(_path):
        sys.path.insert(0, os.path.join(_path))

from karesansui.lib.const import COUNTUP_DATABASE_PATH

from karesansui.lib.collectd.action.mail   import send_mail
from karesansui.lib.collectd.action.script import exec_script

"""
notifyオブジェクト(printの結果)
インスタンス名はプラグインによっては無い場合がある

type=%s
type_instance=%s
plugin=%s
plugin_instance=%s
host=%s
message=%s
severity=%i
time=%lu
"""

countup_db_path = COUNTUP_DATABASE_PATH
logfile         = "/dev/null"

def config(cfg):
    global countup_db_path
    global logfile

    for child in cfg.children:
        if child.key == "CountupDBPath":
            collectd.debug( "[config] config arg set key %s: %s" % ( child.key, child.values[0] ) )
            countup_db_path = child.values[0]

        if child.key == "Logfile":
            collectd.debug( "[config] config arg set key %s: %s" % ( child.key, child.values[0] ) )
            logfile = child.values[0]

def init():
    global countup_db_path
    global logfile

    if not os.path.exists(countup_db_path):
        collectd.error( "Can't find CountupDBPath at %s, disabling plugin" % (countup_db_path))

def notification(notify=None, data=None):
    global countup_db_path
    global logfile

    # メッセージを分解して値を取得

    # karesansui DBから該当するプラグインのフィルタリング条件を取得
    #  無い場合は終了

    # 回数DBに呼び出されたことを記録

    # 何回連続しているか調べる
    #  連続していない場合は古いデータを削除
    #  !! 連続していない場合は、そもそもこのルーチンに入らないので不可能と思う

    # 連続している回数が許容範囲内であるかチェック
    #  許容範囲内もしくはアラートが連続している場合は終了
    #  !! 連続していない場合は、そもそもこのルーチンに入らないので不可能と思う

    # !! メモ書き !!
    # むしろ、何秒間の間に何回のカウントされたかをみる方法がよいと思われる
    #  inspired by iptables's hitcount and seconds option.
    #  例えば、監視間隔が30秒であったときに 5分間に5回カウントされたら
    #  アクションを起こすなど。。
    #  また、一回アクションを起こしたら、一定期間は同一アラートでアクションを
    #  起こさない設定なども必要と思われる。

    # アラート発生の場合、アラートの種類を取得

    # アラート呼び出し

    data_type     = notify.type
    data_type_instance = notify.type_instance
    data_plugin   = notify.plugin
    data_host     = notify.host
    data_message  = notify.message
    data_time     = notify.time

    ts_percentage = None

    # messageを展開、ds.xxxxをdata_valueとして取り出す
    data_value = data_message
    if data_message[0:2] == "{'":
        try:
            exec("data_value = %s['value']" % data_message)
        except:
            pass

    else:
        regex  = "^Host (?P<host>.+), plugin (?P<plugin>.+) type (?P<type>.+) \(instance (?P<type_instance>.+)\): "
        regex += "Data source \"(?P<ds_name>.+)\" is currently (?P<ds_value>.+)\. "
        regex += "That is (below|above) the (failure|warning) threshold of (?P<ts_value>.+)\.$"

        m = re.match(regex,data_message)
        if m:
            data_value    = m.group('ds_value')
            ts_percentage = m.group('ts_value')

    from karesansui.lib.utils import float_from_string
    _ret = float_from_string(data_value)
    if _ret is not False:
        data_value = _ret

    fp = open(logfile,"a")
    fp.write("countup_db_path: %s\n" % (countup_db_path))
    fp.write("data           : %s\n" % (data))
    fp.write("type           : %s\n" % (notify.type))
    fp.write("type_instance  : %s\n" % (notify.type_instance))
    fp.write("plugin         : %s\n" % (notify.plugin))
    fp.write("host           : %s\n" % (notify.host))
    fp.write("message        : %s\n" % (notify.message))
    fp.write("time           : %d\n" % (notify.time))
    fp.write("data_value     : %s\n" % (data_value))
    fp.write("ts_percentage  : %s\n" % (ts_percentage))
    #fp.write("interval       : %d\n" % (notify.interval))
    fp.close()

    """
    collectd.Values(type='cpu',type_instance='steal',
                    plugin='cpu',plugin_instance='0',
                    host='host.example.com',
                    message='Oops, the cpu steal cpu is currently 0!',
                    time=%lu,interval=%i)
    """

collectd.register_config(config)
collectd.register_init(init)
collectd.register_notification(notification)
