# -*- coding: utf-8 -*-
#
# This file is part of Karesansui.
#
# Copyright (C) 2009-2010 HDE, Inc.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#

import os.path
import re

import web
import simplejson as json

import karesansui
from karesansui.gadget.guest import regist_guest
from karesansui.lib.rest import Rest, auth

from karesansui.lib.utils import chk_create_disk, json_dumps, is_param, base64_encode
from karesansui.lib.const import \
    VIRT_COMMAND_EXPORT_GUEST, \
    VIRT_DOMAINS_DIR, \
    ID_MIN_LENGTH, ID_MAX_LENGTH, \
    NOTE_TITLE_MIN_LENGTH, NOTE_TITLE_MAX_LENGTH

from karesansui.lib.virt.virt import KaresansuiVirtConnection

from karesansui.db.access.machine import findbyguest1
from karesansui.db.access.machine2jobgroup import new as m2j_new
from karesansui.db.access._2pysilhouette import save_job_collaboration

from pysilhouette.command import dict2command
from karesansui.db.model._2pysilhouette import Job, JobGroup

from karesansui.lib.checker import Checker, \
    CHECK_EMPTY, CHECK_VALID, CHECK_LENGTH, CHECK_ONLYSPACE, \
    CHECK_MIN, CHECK_MAX

def validates_guest_export(obj):
    checker = Checker()
    check = True

    _ = obj._
    checker.errors = []

    if not is_param(obj.input, 'export_title'):
        check = False
        checker.add_error(_('Parameter export_title does not exist.'))
    else:
        check = checker.check_string(
                    _('Title'),
                    obj.input.export_title,
                    CHECK_LENGTH | CHECK_ONLYSPACE,
                    None,
                    min = NOTE_TITLE_MIN_LENGTH,
                    max = NOTE_TITLE_MAX_LENGTH,
                ) and check

    obj.view.alert = checker.errors
    return check


def validates_src_id(obj):
    """<comment-ja>
    ゲストOSエクスポート対象のチェッカー
    
    @param obj: karesansui.lib.rest.Rest オブジェクト
    @type obj: karesansui.lib.rest.Rest
    @return: check
    @rtype: bool
    </comment-ja>
    <comment-en>
    TODO: English Comment
    </comment-en>
    """
    checker = Checker()
    check = True

    _ = obj._
    checker.errors = []

    if not is_param(obj.input, 'src_id'):
        check = False
        checker.add_error(_('"%s" is required.') % _('Copy Source'))
    else:
        check = checker.check_number(_('Copy Source'),
                                     obj.input.src_id,
                                     CHECK_EMPTY | CHECK_VALID | CHECK_MIN | CHECK_MAX,
                                     ID_MIN_LENGTH,
                                     ID_MAX_LENGTH
                                     ) and check
        
        obj.view.alert = checker.errors
    return check

class GuestExport(Rest):

    @auth
    def _GET(self, *param, **params):

        host_id = self.chk_hostby1(param)
        if host_id is None: return web.notfound()

        if not validates_src_id(self):
            return web.badrequest(self.view.alert)

        src_id = self.input.src_id
        if self.is_mode_input() is False:
            return web.nomethod()

        self.view.src_id = src_id

        model = findbyguest1(self.orm, src_id)
        if not model:
            return web.badrequest()

        kvc = KaresansuiVirtConnection()
        try:

            domname = kvc.uuid_to_domname(model.uniq_key)
            #if not domname: return web.conflict(web.ctx.path)

            virt = kvc.search_kvg_guests(domname)[0]
            self.view.domain_src_name = virt.get_domain_name()

            active_pools   = kvc.list_active_storage_pool()
            inactive_pools = kvc.list_inactive_storage_pool()

            self.view.pools = sorted(active_pools + inactive_pools)
        finally:
            kvc.close()

        return True


    @auth
    def _POST(self, *param, **params):
        host_id = self.chk_hostby1(param)
        if host_id is None: return web.notfound()
        
        if not validates_guest_export(self):
            return web.badrequest(self.view.alert)
            
        if not validates_src_id(self):
            return web.badrequest(self.view.alert)

        model = findbyguest1(self.orm, self.input.src_id)
        if not model:
            return web.badrequest()
    
        kvc = KaresansuiVirtConnection()
        try:
            domname = kvc.uuid_to_domname(model.uniq_key)
            if not domname: return web.conflict(web.ctx.path)
            virt = kvc.search_kvg_guests(domname)[0]
            options = {}
            options["name"] = virt.get_domain_name()
            if is_param(self.input, "pool"):
                options["pool"] = self.input.pool
            if is_param(self.input, "export_title"):
                #options["title"] = self.input.export_title
                options["title"] = "b64:" + base64_encode(self.input.export_title)
            options["quiet"] = None

            # disk check
            src_disk = "%s/%s/images/%s.img" % (VIRT_DOMAINS_DIR,options["name"],options["name"])
            s_size = os.path.getsize(src_disk) / (1024 * 1024) # a unit 'MB'
            if chk_create_disk(VIRT_DOMAINS_DIR, s_size) is False:
                return web.badrequest('Do not have enough free disk size.')

        finally:
            kvc.close()


        _cmd = dict2command(
            "%s/%s" % (karesansui.config['application.bin.dir'], VIRT_COMMAND_EXPORT_GUEST), options)

        # Job Register
        cmdname = ["Export Guest", "export guest"]
        _jobgroup = JobGroup(cmdname[0], karesansui.sheconf['env.uniqkey'])
        _jobgroup.jobs.append(Job('%s command' % cmdname[1], 0, _cmd))

        _machine2jobgroup = m2j_new(machine=model,
                                    jobgroup_id=-1,
                                    uniq_key=karesansui.sheconf['env.uniqkey'],
                                    created_user=self.me,
                                    modified_user=self.me,
                                    )

        # INSERT
        save_job_collaboration(self.orm,
                               self.pysilhouette.orm,
                               _machine2jobgroup,
                               _jobgroup,
                               )

        self.logger.debug("(%s) Job group id==%s" % (cmdname[0],_jobgroup.id))
        url = '%s/job/%s.part' % (web.ctx.home, _jobgroup.id)
        self.logger.debug('Returning Location: %s' % url)
        
        return web.accepted()

urls = (
    '/host/(\d+)/guest/export/?$', GuestExport,
    )
