<?php
/**
 * ページ区切り用のパラメータからデータを整列し、取得数を制限するためのSQLステートメントの一部を生成します。
 * この関数はORDER節と、LIMIT節を生成します。
 * @param string $order_expr ORDER BY句に続く整列条件を文字列で指示します。
 * @param bool $is_asc ソート順序が昇順であるかどうかを示す値
 * @param int $limit 取得する部屋数の上限を示す整数値。0を指定すると全ての部屋を取得します。
 * @param int $page 部屋を取得するときのオフセットを示す整数値。オフセットは$limitと$pageの積で決定されます。
 * @return array 整列用SQL、取得数の制限用SQLのセット
 */
function generatePaginationClauses($order_expr, $is_asc=true, $limit=0, $page=0) {
  $order_clause = 'ORDER BY '.$order_expr.($is_asc ? ' ASC' : ' DESC');
  $limit_clause = '';
  if (!empty($limit)) {
    if (empty($page)) {
      $limit_clause = "LIMIT {$limit}";
    }
    else {
      $offset = $page * $limit;
      $limit_clause = "LIMIT {$offset},{$limit}";
    }
  }
  return array($order_clause, $limit_clause);
}

/**
 * テーブル名と検索条件を指定して、テーブルをidによって検索するためのSQLステートメントの一部を生成します。
 * @param string $table 検索するテーブルの名前
 * @param mixed $id 一つ、または配列に格納された複数のid。true、または文字列'*'を指定すると指定した部屋の全てのユーザーを取得しようとします。
 * @param string $prefix バインディングのキーとして使用するプレフィックス文字列。nullを指定すると$tableの値が使用されます。
 * @return array id選択用SQLとバインディングする値のセット
 */
function generateIdSelector($table, $id, $prefix=null) {
  $values = array();
  if ($id == '*' || $id === true) {
    $selector = 1;
  }
  else if ($id !== false) {
    $pf = empty($prefix) ? $table : $prefix;
    foreach((is_array($id) ? $id : array($id)) as $index => $val) {
      $values[":{$pf}_{$index}"] = $val;
    }
    $selector = $table.'.id IN ('.implode(',', array_keys($values)).')';
  }
  else {
    $selector = 0;
  }
  return array($selector, $values);
}
/**
 * 入力文字列が妥当かどうか評価します。
 * @param string $text 入力文字列
 * @param bool $nullable NULLを許容するかどうかを示す値
 * @param int $min_length 文字列の長さの最小値
 * @param int $max_length 文字列の長さの最大値
 * @return bool 条件を満たした場合true, それ以外の場合false
 */
function validateText($text, $nullable=false, $min_length=-1, $max_length=-1) {
  if (!$nullable && $text === null) {
    return false;
  }
  $len = mb_strlen($text);
  if (0 <= $min_length && $len < $min_length) {
    return false;
  }
  if (0 <= $max_length && $max_length < $len) {
    return false;
  }
  return true;
}
