/**
 * ============================================================================================
 * Easing Equations v2.0
 * September 1, 2003
 * (c) 2003 Robert Penner, all rights reserved.
 * This work is subject to the terms in http://www.robertpenner.com/easing_terms_of_use.html.
 * ============================================================================================
 */

/**
 * イージング関数ライブラリ
 * enchant.jsプロジェクトで
 * 「ActionScript で広く使われている Robert Penner による Easing Equations を JavaScript に移植した」
 * ライブラリを、さらにType script用に移植したもの。
 *
 * @see http://www.robertpenner.com/easing/
 * @see http://www.robertpenner.com/easing/penner_chapter7_tweening.pdf
 *
 * Easing function library, from "Easing Equations" by Robert Penner.
 */
class Easing {
	static LINEAR(t:number, b:number, c:number, d:number):number {
		return c * t / d + b;
	}

	static SWING(t:number, b:number, c:number, d:number):number {
		return c * (0.5 - Math.cos(((t / d) * Math.PI)) / 2) + b;
	}

	static QUAD_EASEIN(t:number, b:number, c:number, d:number):number {
		return c * (t /= d) * t + b;
	}

	static QUAD_EASEOUT(t:number, b:number, c:number, d:number):number {
		return -c * (t /= d) * (t - 2) + b;
	}

	static QUAD_EASEINOUT(t:number, b:number, c:number, d:number):number {
		if ((t /= d / 2) < 1) {
			return c / 2 * t * t + b;
		}
		return -c / 2 * ((--t) * (t - 2) - 1) + b;
	}

	static CUBIC_EASEIN(t:number, b:number, c:number, d:number):number {
		return c * (t /= d) * t * t + b;
	}

	static CUBIC_EASEOUT(t:number, b:number, c:number, d:number):number {
		return c * ((t = t / d - 1) * t * t + 1) + b;
	}

	static CUBIC_EASEINOUT(t:number, b:number, c:number, d:number):number {
		if ((t /= d / 2) < 1) {
			return c / 2 * t * t * t + b;
		}
		return c / 2 * ((t -= 2) * t * t + 2) + b;
	}

	static QUART_EASEIN(t:number, b:number, c:number, d:number):number {
		return c * (t /= d) * t * t * t + b;
	}

	static QUART_EASEOUT(t:number, b:number, c:number, d:number):number {
		return -c * ((t = t / d - 1) * t * t * t - 1) + b;
	}

	static QUART_EASEINOUT(t:number, b:number, c:number, d:number):number {
		if ((t /= d / 2) < 1) {
			return c / 2 * t * t * t * t + b;
		}
		return -c / 2 * ((t -= 2) * t * t * t - 2) + b;
	}

	static QUINT_EASEIN(t:number, b:number, c:number, d:number):number {
		return c * (t /= d) * t * t * t * t + b;
	}

	static QUINT_EASEOUT(t:number, b:number, c:number, d:number):number {
		return c * ((t = t / d - 1) * t * t * t * t + 1) + b;
	}

	static QUINT_EASEINOUT(t:number, b:number, c:number, d:number):number {
		if ((t /= d / 2) < 1) {
			return c / 2 * t * t * t * t * t + b;
		}
		return c / 2 * ((t -= 2) * t * t * t * t + 2) + b;
	}

	static SIN_EASEIN(t:number, b:number, c:number, d:number):number {
		return -c * Math.cos(t / d * (Math.PI / 2)) + c + b;
	}

	static SIN_EASEOUT(t:number, b:number, c:number, d:number):number {
		return c * Math.sin(t / d * (Math.PI / 2)) + b;
	}

	static SIN_EASEINOUT(t:number, b:number, c:number, d:number):number {
		return -c / 2 * (Math.cos(Math.PI * t / d) - 1) + b;
	}

	static CIRC_EASEIN(t:number, b:number, c:number, d:number):number {
		return -c * (Math.sqrt(1 - (t /= d) * t) - 1) + b;
	}

	static CIRC_EASEOUT(t:number, b:number, c:number, d:number):number {
		return c * Math.sqrt(1 - (t = t / d - 1) * t) + b;
	}

	static CIRC_EASEINOUT(t:number, b:number, c:number, d:number):number {
		if ((t /= d / 2) < 1) {
			return -c / 2 * (Math.sqrt(1 - t * t) - 1) + b;
		}
		return c / 2 * (Math.sqrt(1 - (t -= 2) * t) + 1) + b;
	}

	static ELASTIC_EASEIN(t:number, b:number, c:number, d:number, a?:number, p?:number):number {
		if (t === 0) {
			return b;
		}
		if ((t /= d) === 1) {
			return b + c;
		}

		if (!p) {
			p = d * 0.3;
		}

		var s;
		if (!a || a < Math.abs(c)) {
			a = c;
			s = p / 4;
		} else {
			s = p / (2 * Math.PI) * Math.asin(c / a);
		}
		return -(a * Math.pow(2, 10 * (t -= 1)) * Math.sin((t * d - s) * (2 * Math.PI) / p)) + b;
	}

	static ELASTIC_EASEOUT(t:number, b:number, c:number, d:number, a?:number, p?:number):number {
		if (t === 0) {
			return b;
		}
		if ((t /= d) === 1) {
			return b + c;
		}
		if (!p) {
			p = d * 0.3;
		}
		var s;
		if (!a || a < Math.abs(c)) {
			a = c;
			s = p / 4;
		} else {
			s = p / (2 * Math.PI) * Math.asin(c / a);
		}
		return (a * Math.pow(2, -10 * t) * Math.sin((t * d - s) * (2 * Math.PI) / p) + c + b);
	}

	static ELASTIC_EASEINOUT(t:number, b:number, c:number, d:number, a?:number, p?:number):number {
		if (t === 0) {
			return b;
		}
		if ((t /= d / 2) === 2) {
			return b + c;
		}
		if (!p) {
			p = d * (0.3 * 1.5);
		}
		var s;
		if (!a || a < Math.abs(c)) {
			a = c;
			s = p / 4;
		} else {
			s = p / (2 * Math.PI) * Math.asin(c / a);
		}
		if (t < 1) {
			return -0.5 * (a * Math.pow(2, 10 * (t -= 1)) * Math.sin((t * d - s) * (2 * Math.PI) / p)) + b;
		}
		return a * Math.pow(2, -10 * (t -= 1)) * Math.sin((t * d - s) * (2 * Math.PI) / p) * 0.5 + c + b;
	}

	static BOUNCE_EASEOUT(t:number, b:number, c:number, d:number):number {
		if ((t /= d) < (1 / 2.75)) {
			return c * (7.5625 * t * t) + b;
		} else if (t < (2 / 2.75)) {
			return c * (7.5625 * (t -= (1.5 / 2.75)) * t + 0.75) + b;
		} else if (t < (2.5 / 2.75)) {
			return c * (7.5625 * (t -= (2.25 / 2.75)) * t + 0.9375) + b;
		} else {
			return c * (7.5625 * (t -= (2.625 / 2.75)) * t + 0.984375) + b;
		}
	}

	static BOUNCE_EASEIN(t:number, b:number, c:number, d:number):number {
		return c - Easing.BOUNCE_EASEOUT(d - t, 0, c, d) + b;
	}

	static BOUNCE_EASEINOUT(t:number, b:number, c:number, d:number):number {
		if (t < d / 2) {
			return Easing.BOUNCE_EASEIN(t * 2, 0, c, d) * 0.5 + b;
		} else {
			return Easing.BOUNCE_EASEOUT(t * 2 - d, 0, c, d) * 0.5 + c * 0.5 + b;
		}
	}

	static BACK_EASEIN(t:number, b:number, c:number, d:number, s?:number):number {
		if (s === undefined) {
			s = 1.70158;
		}
		return c * (t /= d) * t * ((s + 1) * t - s) + b;
	}

	static BACK_EASEOUT(t:number, b:number, c:number, d:number, s?:number):number {
		if (s === undefined) {
			s = 1.70158;
		}
		return c * ((t = t / d - 1) * t * ((s + 1) * t + s) + 1) + b;
	}

	static BACK_EASEINOUT(t:number, b:number, c:number, d:number, s?:number):number {
		if (s === undefined) {
			s = 1.70158;
		}
		if ((t /= d / 2) < 1) {
			return c / 2 * (t * t * (((s *= (1.525)) + 1) * t - s)) + b;
		}
		return c / 2 * ((t -= 2) * t * (((s *= (1.525)) + 1) * t + s) + 2) + b;
	}

	static EXPO_EASEIN(t:number, b:number, c:number, d:number):number {
		return (t === 0) ? b : c * Math.pow(2, 10 * (t / d - 1)) + b;
	}

	static EXPO_EASEOUT(t:number, b:number, c:number, d:number):number {
		return (t === d) ? b + c : c * (-Math.pow(2, -10 * t / d) + 1) + b;
	}

	static EXPO_EASEINOUT(t:number, b:number, c:number, d:number):number {
		if (t === 0) {
			return b;
		}
		if (t === d) {
			return b + c;
		}
		if ((t /= d / 2) < 1) {
			return c / 2 * Math.pow(2, 10 * (t - 1)) + b;
		}
		return c / 2 * (-Math.pow(2, -10 * --t) + 2) + b;
	}

	static RANDOM() {
		var ary = [];
		for (var i in Easing) {
			if (i != "RANDOM")
				ary.push(i);
		}
		return Easing[ary[Math.floor(Math.random() * ary.length)]];
	}
}