///<reference path="all.ts"/>
module jg {
	/**
	 * 読み込み中状態を管理するシーン
	 */
	export class LoadingScene extends Scene {
		/** 対象のリソース */
		resource: Resource;
		/** 読み込み完了を表すパーセンテージの外枠線 */
		shape: Shape;
		/** 読み込み完了を表すパーセンテージの内側の塗りつぶし */
		shapeP: Shape;
		/** リソース全体の読み込み予定数 */
		requestCount: number;
		/** ローディングシーンの表示が必要な状態が終わったときに発火されるイベント。このイベントが発生した際、既にLoadingSceneクラスは破棄されている。 */
		finished: Trigger;

		/**
		 * コンストラクタ
		 * @param game 対象のゲーム
		 * @param resource 対象のリソース
		 */
		constructor(game:Game, resource:Resource) {
			super(game);
			this.resource = resource;
			this.resource.loaded.handle(this, this.complete);
			this.resource.added.handle(this, this.added);
			this.requestCount = this.resource.requests.length;
			this.finished = new Trigger();

			this.init();
		}

		/**
		 * 初期化
		 */
		init() {
			this.shape = new Shape(this.game.width, 32);
			this.shape.moveTo(0, this.game.height / 2 - 16);

			this.shapeP = new Shape(1, 32, ShapeStyle.Fill);
			this.shapeP.moveTo(0, this.game.height / 2 - 16);

			this.append(this.shape);
			this.append(this.shapeP);
		}

		/**
		 * 読み込み完了を表すパーセンテージ表記を進める
		 */
		animate(per:number) {
			this.shapeP.width = this.game.width * per;
			this.shapeP.updated();
		}

		/**
		 * リソースクラスが読み込みを完了した際のイベントハンドラ
		 */
		complete(cnt:number) {
			var per = (this.requestCount-cnt) / this.requestCount;
			this.animate(per);

			if (per == 1) {
				this.resource.loaded.remove(this, this.complete);
				this.resource.added.remove(this, this.added);
				this.end();
				this.finished.fire();
			}
		}

		/**
		 * リソースクラスにリクエストが追加された場合のイベントハンドラ
		 */
		added(e:any) {
			this.requestCount++;
		}
	}
}