/*!
  \file
  \brief ǐ

  \author Satofumi KAMIMURA

  $Id$

  _Ci~NZ[^̐܂ TMY  dx_controller.c QlɂB
*/

#include "SteerCtrl.h"
#include "GridTypes.h"
#include "SerialCtrl.h"
#include "HoistTask.h"
#include <string.h>
#include <errno.h>
#include <string>
#include <fstream>
#include <map>


enum {
  RD = 1,
  RDWR = 2,
  NOPAR = -1,
  WRITE = 3,
  MAX_ANGLE = 300,
  GOAL_POS = 20,
};

typedef struct {
  char command[27];
  int add;
  int access;
  int length;
  int max;
}instruction_t;
instruction_t ins[]={
  {"Model_number", 0x00, RD, 2, 254},
  {"Version_of_Firmware", 0x02, RD, 1, 127},
  {"ID", 0x03, RDWR, 1, 254},
  {"Baud_rate", 0x04, RDWR, 1, 254},
  {"Return_delay_time", 0x05, RDWR, 1, 254},
  {"Cw_angle_limit", 0x06, RDWR, 2, 1023},
  {"Ccw_angle_limit", 0x08, RDWR, 2, 1023},
  {"Highest_limit_temperature", 0x0b, RDWR, 1, 150},
  {"Lowest_limit_voltage", 0x0c, RDWR, 1, 250},
  {"Highest_limit_voltage", 0x0d, RDWR, 1, 250},
  {"Max_torque", 0x0e, RDWR, 2, 1023},
  {"Status_return_level", 0x10, RDWR, 1, 2},
  {"Alarm_LED", 0x11, RDWR, 1, 127},
  {"Alarm_shutdown", 0x12, RDWR, 1, 127},
  {"Torque_enable", 0x18, RDWR, 1, 1},
  {"LED", 0x19, RDWR, 1, 1},
  {"Cw_compliance_margin", 0x1a, RDWR, 1, 254},
  {"Ccw_compliance_margin", 0x1b, RDWR, 1, 254},
  {"Cw_compliance_slope", 0x1c, RDWR, 1, 254},
  {"Ccw_compliance_slope", 0x1d, RDWR, 1, 254},
  {"Goal_position", 0x1e, RDWR, 2, 1023},
  {"Moving_speed", 0x20, RDWR, 2, 1023},
  {"Torque_limit", 0x22, RDWR, 2, 1023},
  {"Present_position", 0x24, RD, 2, 1023},
  {"Present_speed", 0x26, RD, 2, 1023},
  {"Present_load", 0x28, RD, 2, 2047},
  {"Present_voltage", 0x2a, RD, 1, 250},
  {"Present_temperature", 0x2b, RD, 1, 150},
  {"Registered_instruction", 0x2c, RDWR, 1, 1},
  {"Moving", 0x2e, RD, 1, 1},
  {"Lock", 0x2f, RDWR, 1, 1},
  {"Punch", 0x30, RDWR, 2, 1023},
};

struct SteerCtrl::pImpl {
  enum {
    DefaultBaudrate = 57600,
  };

  std::string error_message;
  typedef std::map<int,int> AngleTable;
  AngleTable angle_table;
  SerialCtrl con;
  int offset_degree;
  int pre_degree;

  pImpl(const char* table_file)
    : error_message("no error."), offset_degree(0), pre_degree(0) {
    if (! loadAngleData(table_file)) {
      throw;
    }
  }

  bool loadAngleData(const char* table_file) {

    std::ifstream fin(table_file);
    if (! fin.is_open()) {
      error_message = strerror(errno);
      return false;
    }

    // e[uf[^̓ǂݏo
    angle_table.clear();
    while (! fin.eof()) {
      enum { InvalidDegree = -360 };
      int motor_degree = -InvalidDegree, steer_degree = InvalidDegree;
      fin >> motor_degree >> steer_degree;

      if ((motor_degree != InvalidDegree) && (steer_degree != InvalidDegree)) {
	angle_table[steer_degree] = motor_degree;
      }
    }

    // ŵȂpxf[^`⊮
    if (angle_table.empty()) {
      error_message = "no angle data.";
      return false;
    }

    AngleTable::iterator it = angle_table.begin();
    int motor_1st = it->second;
    int steer_1st = it->first;
    for (++it; it != angle_table.end(); ++it) {
      int motor_2nd = it->second;
      int steer_2nd = it->first;

      int steer_range = steer_2nd - steer_1st;
      for (int i = steer_1st; i < steer_2nd; ++i) {
	angle_table[i] = motor_1st
	  + (motor_2nd - motor_1st) * (i - steer_1st) / steer_range;
      }

      motor_1st = motor_2nd;
      steer_1st = steer_2nd;
    }
    return true;
  }

  unsigned char checksum(int length, unsigned char *str){
  int i;

  str[length]=0;
  for(i=2;i<length;i++){
    str[length]+=str[i];
  }
  str[length]=~str[length];
  return(*str);
}

  bool DX_SendWritePacket(int id, int mode, int val){
    unsigned char str[9];

    //ރf[^Ó`FbN
    if((val > ins[mode].max) && (mode != NOPAR)){
      return false;
    }

    str[0] = str[1]=0xff;
    str[2] = id;
    str[3] = ins[mode].length+3;
    str[4] = WRITE;
    str[5] = ins[mode].add;

    //commandɂp[^1byte2byte
    if(ins[mode].length == 1)
      str[6] = val;
    else{
      str[6] = val;
      str[7] = val >> 8;
    }
    //`FbNTvZđM
    checksum(ins[mode].length+6, str);
    con.send((char*)(str), ins[mode].length +7);

    return true;
  }

  void Dx_Setrpm (int id, int rpm){
    enum {
      max_spd = 70,
      Mov_spd = 21,
    };
    int val;
    val= rpm * 0x3ff / max_spd;
    DX_SendWritePacket(id, Mov_spd, val);
  }

  void Dx_Turn (int id, int degree){
    int val;
    val= degree * 0x3ff / MAX_ANGLE;
    DX_SendWritePacket(id, GOAL_POS, val);
  }

  void Dx_Steer(int degree){
    if(degree<=150 && degree>=-150){
      Dx_Turn(254, 150+degree);	// u[hLXg
    }
  }
};


SteerCtrl::SteerCtrl(const char* table_file) : pimpl(new pImpl(table_file)) {
}


SteerCtrl::~SteerCtrl(void) {
}


const char* SteerCtrl::what(void) {
  return pimpl->error_message.c_str();
}


bool SteerCtrl::connect(const char* device) {

  if (pimpl->con.connect(device, pImpl::DefaultBaudrate) < 0) {
    pimpl->error_message = pimpl->con.what();
    return false;
  }

  // mFĐڑƂ݂Ȃ
  // !!!
  // !!! ̂

  pimpl->Dx_Setrpm(254, 15);
  return true;
}


void SteerCtrl::setSteer(Angle angle) {
  if (! pimpl->con.isConnected()) {
    return;
  }

  enum {
    SteerLimit = 20,		// [degree]
    OnceLimit = 5,		// [degree]
  };
  int degree = static_cast<int>(angle.to_deg());
  if (degree < -SteerLimit) {
    degree = -SteerLimit;
  } else if (degree > +SteerLimit) {
    degree = +SteerLimit;
  }
  fprintf(stderr, "base: %d, offset added: %d\n", degree, degree + pimpl->offset_degree);
  degree += pimpl->offset_degree;

  // ~ƂɑǂȂ悤AP񂠂̑Ǘʂ𐧌
  // ]̐R}hsꂽł̒~R}hɑǂȂ
  if (abs(pimpl->pre_degree - degree) > OnceLimit) {
    // !!! ςŌvZłC邯ǁAƂ肠ۗ
    if (pimpl->pre_degree < degree) {
      degree = pimpl->pre_degree + OnceLimit;
    } else {
      degree = pimpl->pre_degree - OnceLimit;
    }
  }
  pimpl->pre_degree = degree;

  pImpl::AngleTable::iterator it = pimpl->angle_table.find(degree);
  if (it == pimpl->angle_table.end()) {
    fprintf(stderr, "invalid degree: %d\n", degree);
    return;
  }

  // Ǌpx悤Ƀ[^Ɏw߂^
  pimpl->Dx_Steer(it->second);
}


void SteerCtrl::setSteerOffset(Angle angle) {
  pimpl->offset_degree = static_cast<int>(angle.to_deg());
}


int SteerCtrl::getSteerOffset(void) {
  return pimpl->offset_degree;
}


int SteerCtrl::getSteer(void) {
  int velocity = 0;
  int steer_degree = 0;
  hoist_getVelocity(velocity, steer_degree);

  return steer_degree;
}
