/*!
 * @file mspells1.c
 * @brief 󥹥ˡμ / Monster spells (attack player)
 * @date 2014/01/17
 * @author
 * Copyright (c) 1997 Ben Harrison, James E. Wilson, Robert A. Koeneke\n
 * This software may be copied and distributed for educational, research,\n
 * and not for profit purposes provided that this copyright and statement\n
 * are included in all such copies.  Other copyrights may also apply.\n
 * 2014 Deskull rearranged comment for Doxygen.\n
 * @details
 * And now for Intelligent monster attacks (including spells).\n
 *\n
 * Original idea and code by "DRS" (David Reeves Sward).\n
 * Major modifications by "BEN" (Ben Harrison).\n
 *\n
 * Give monsters more intelligent attack/spell selection based on\n
 * observations of previous attacks on the player, and/or by allowing\n
 * the monster to "cheat" and know the player status.\n
 *\n
 * Maintain an idea of the player status, and use that information\n
 * to occasionally eliminate "ineffective" spell attacks.  We could\n
 * also eliminate ineffective normal attacks, but there is no reason\n
 * for the monster to do this, since he gains no benefit.\n
 * Note that MINDLESS monsters are not allowed to use this code.\n
 * And non-INTELLIGENT monsters only use it partially effectively.\n
 *\n
 * Actually learn what the player resists, and use that information\n
 * to remove attacks or spells before using them.  This will require\n
 * much less space, if I am not mistaken.  Thus, each monster gets a\n
 * set of 32 bit flags, "smart", build from the various "SM_*" flags.\n
 *\n
 * This has the added advantage that attacks and spells are related.\n
 * The "smart_learn" option means that the monster "learns" the flags\n
 * that should be set, and "smart_cheat" means that he "knows" them.\n
 * So "smart_cheat" means that the "smart" field is always up to date,\n
 * while "smart_learn" means that the "smart" field is slowly learned.\n
 * Both of them have the same effect on the "choose spell" routine.\n
 */

#include "angband.h"


/*!
 * @brief 󥹥ץ쥤䡼μĤ뤫ɤȽ /
 * Internal probability routine
 * @param r_ptr 󥹥²ι¤λȥݥ
 * @param prob ܳΨ(%)
 * @return ŬʤTRUE֤
 */
static bool int_outof(monster_race *r_ptr, int prob)
{
	/* Non-Smart monsters are half as "smart" */
	if (!(r_ptr->flags2 & RF2_SMART)) prob = prob / 2;

	/* Roll the dice */
	return (randint0(100) < prob);
}


/*!
 * @brief 󥹥ˡŪŬʤˡ /
 * Remove the "bad" spells from a spell list
 * @param m_idx 󥹥ι¤λȥݥ
 * @param f4p 󥹥ˡΥե饰ꥹ1
 * @param f5p 󥹥ˡΥե饰ꥹ2
 * @param f6p 󥹥ˡΥե饰ꥹ3
 * @return ʤ
 */
static void remove_bad_spells(int m_idx, u32b *f4p, u32b *f5p, u32b *f6p)
{
	monster_type *m_ptr = &m_list[m_idx];
	monster_race *r_ptr = &r_info[m_ptr->r_idx];

	u32b f4 = (*f4p);
	u32b f5 = (*f5p);
	u32b f6 = (*f6p);

	u32b smart = 0L;


	/* Too stupid to know anything */
	if (r_ptr->flags2 & RF2_STUPID) return;


	/* Must be cheating or learning */
	if (!smart_cheat && !smart_learn) return;


	/* Update acquired knowledge */
	if (smart_learn)
	{
		/* Hack -- Occasionally forget player status */
		/* Only save SM_FRIENDLY, SM_PET or SM_CLONED */
		if (m_ptr->smart && (randint0(100) < 1)) m_ptr->smart &= (SM_FRIENDLY | SM_PET | SM_CLONED);

		/* Use the memorized flags */
		smart = m_ptr->smart;
	}


	/* Cheat if requested */
	if (smart_cheat)
	{
		/* Know basic info */
		if (p_ptr->resist_acid) smart |= (SM_RES_ACID);
		if (IS_OPPOSE_ACID()) smart |= (SM_OPP_ACID);
		if (p_ptr->immune_acid) smart |= (SM_IMM_ACID);
		if (p_ptr->resist_elec) smart |= (SM_RES_ELEC);
		if (IS_OPPOSE_ELEC()) smart |= (SM_OPP_ELEC);
		if (p_ptr->immune_elec) smart |= (SM_IMM_ELEC);
		if (p_ptr->resist_fire) smart |= (SM_RES_FIRE);
		if (IS_OPPOSE_FIRE()) smart |= (SM_OPP_FIRE);
		if (p_ptr->immune_fire) smart |= (SM_IMM_FIRE);
		if (p_ptr->resist_cold) smart |= (SM_RES_COLD);
		if (IS_OPPOSE_COLD()) smart |= (SM_OPP_COLD);
		if (p_ptr->immune_cold) smart |= (SM_IMM_COLD);

		/* Know poison info */
		if (p_ptr->resist_pois) smart |= (SM_RES_POIS);
		if (IS_OPPOSE_POIS()) smart |= (SM_OPP_POIS);

		/* Know special resistances */
		if (p_ptr->resist_neth) smart |= (SM_RES_NETH);
		if (p_ptr->resist_lite) smart |= (SM_RES_LITE);
		if (p_ptr->resist_dark) smart |= (SM_RES_DARK);
		if (p_ptr->resist_fear) smart |= (SM_RES_FEAR);
		if (p_ptr->resist_conf) smart |= (SM_RES_CONF);
		if (p_ptr->resist_chaos) smart |= (SM_RES_CHAOS);
		if (p_ptr->resist_disen) smart |= (SM_RES_DISEN);
		if (p_ptr->resist_blind) smart |= (SM_RES_BLIND);
		if (p_ptr->resist_nexus) smart |= (SM_RES_NEXUS);
		if (p_ptr->resist_sound) smart |= (SM_RES_SOUND);
		if (p_ptr->resist_shard) smart |= (SM_RES_SHARD);
		if (p_ptr->reflect) smart |= (SM_IMM_REFLECT);

		/* Know bizarre "resistances" */
		if (p_ptr->free_act) smart |= (SM_IMM_FREE);
		if (!p_ptr->msp) smart |= (SM_IMM_MANA);
	}


	/* Nothing known */
	if (!smart) return;


	if (smart & SM_IMM_ACID)
	{
		f4 &= ~(RF4_BR_ACID);
		f5 &= ~(RF5_BA_ACID);
		f5 &= ~(RF5_BO_ACID);
	}
	else if ((smart & (SM_OPP_ACID)) && (smart & (SM_RES_ACID)))
	{
		if (int_outof(r_ptr, 80)) f4 &= ~(RF4_BR_ACID);
		if (int_outof(r_ptr, 80)) f5 &= ~(RF5_BA_ACID);
		if (int_outof(r_ptr, 80)) f5 &= ~(RF5_BO_ACID);
	}
	else if ((smart & (SM_OPP_ACID)) || (smart & (SM_RES_ACID)))
	{
		if (int_outof(r_ptr, 30)) f4 &= ~(RF4_BR_ACID);
		if (int_outof(r_ptr, 30)) f5 &= ~(RF5_BA_ACID);
		if (int_outof(r_ptr, 30)) f5 &= ~(RF5_BO_ACID);
	}


	if (smart & (SM_IMM_ELEC))
	{
		f4 &= ~(RF4_BR_ELEC);
		f5 &= ~(RF5_BA_ELEC);
		f5 &= ~(RF5_BO_ELEC);
	}
	else if ((smart & (SM_OPP_ELEC)) && (smart & (SM_RES_ELEC)))
	{
		if (int_outof(r_ptr, 80)) f4 &= ~(RF4_BR_ELEC);
		if (int_outof(r_ptr, 80)) f5 &= ~(RF5_BA_ELEC);
		if (int_outof(r_ptr, 80)) f5 &= ~(RF5_BO_ELEC);
	}
	else if ((smart & (SM_OPP_ELEC)) || (smart & (SM_RES_ELEC)))
	{
		if (int_outof(r_ptr, 30)) f4 &= ~(RF4_BR_ELEC);
		if (int_outof(r_ptr, 30)) f5 &= ~(RF5_BA_ELEC);
		if (int_outof(r_ptr, 30)) f5 &= ~(RF5_BO_ELEC);
	}


	if (smart & (SM_IMM_FIRE))
	{
		f4 &= ~(RF4_BR_FIRE);
		f5 &= ~(RF5_BA_FIRE);
		f5 &= ~(RF5_BO_FIRE);
	}
	else if ((smart & (SM_OPP_FIRE)) && (smart & (SM_RES_FIRE)))
	{
		if (int_outof(r_ptr, 80)) f4 &= ~(RF4_BR_FIRE);
		if (int_outof(r_ptr, 80)) f5 &= ~(RF5_BA_FIRE);
		if (int_outof(r_ptr, 80)) f5 &= ~(RF5_BO_FIRE);
	}
	else if ((smart & (SM_OPP_FIRE)) || (smart & (SM_RES_FIRE)))
	{
		if (int_outof(r_ptr, 30)) f4 &= ~(RF4_BR_FIRE);
		if (int_outof(r_ptr, 30)) f5 &= ~(RF5_BA_FIRE);
		if (int_outof(r_ptr, 30)) f5 &= ~(RF5_BO_FIRE);
	}


	if (smart & (SM_IMM_COLD))
	{
		f4 &= ~(RF4_BR_COLD);
		f5 &= ~(RF5_BA_COLD);
		f5 &= ~(RF5_BO_COLD);
		f5 &= ~(RF5_BO_ICEE);
	}
	else if ((smart & (SM_OPP_COLD)) && (smart & (SM_RES_COLD)))
	{
		if (int_outof(r_ptr, 80)) f4 &= ~(RF4_BR_COLD);
		if (int_outof(r_ptr, 80)) f5 &= ~(RF5_BA_COLD);
		if (int_outof(r_ptr, 80)) f5 &= ~(RF5_BO_COLD);
		if (int_outof(r_ptr, 80)) f5 &= ~(RF5_BO_ICEE);
	}
	else if ((smart & (SM_OPP_COLD)) || (smart & (SM_RES_COLD)))
	{
		if (int_outof(r_ptr, 30)) f4 &= ~(RF4_BR_COLD);
		if (int_outof(r_ptr, 30)) f5 &= ~(RF5_BA_COLD);
		if (int_outof(r_ptr, 30)) f5 &= ~(RF5_BO_COLD);
		if (int_outof(r_ptr, 20)) f5 &= ~(RF5_BO_ICEE);
	}


	if ((smart & (SM_OPP_POIS)) && (smart & (SM_RES_POIS)))
	{
		if (int_outof(r_ptr, 80)) f4 &= ~(RF4_BR_POIS);
		if (int_outof(r_ptr, 80)) f5 &= ~(RF5_BA_POIS);
		if (int_outof(r_ptr, 60)) f4 &= ~(RF4_BA_NUKE);
		if (int_outof(r_ptr, 60)) f4 &= ~(RF4_BR_NUKE);
	}
	else if ((smart & (SM_OPP_POIS)) || (smart & (SM_RES_POIS)))
	{
		if (int_outof(r_ptr, 30)) f4 &= ~(RF4_BR_POIS);
		if (int_outof(r_ptr, 30)) f5 &= ~(RF5_BA_POIS);
	}


	if (smart & (SM_RES_NETH))
	{
		if (prace_is_(RACE_SPECTRE))
		{
			f4 &= ~(RF4_BR_NETH);
			f5 &= ~(RF5_BA_NETH);
			f5 &= ~(RF5_BO_NETH);
		}
		else
		{
			if (int_outof(r_ptr, 20)) f4 &= ~(RF4_BR_NETH);
			if (int_outof(r_ptr, 50)) f5 &= ~(RF5_BA_NETH);
			if (int_outof(r_ptr, 50)) f5 &= ~(RF5_BO_NETH);
		}
	}

	if (smart & (SM_RES_LITE))
	{
		if (int_outof(r_ptr, 50)) f4 &= ~(RF4_BR_LITE);
		if (int_outof(r_ptr, 50)) f5 &= ~(RF5_BA_LITE);
	}

	if (smart & (SM_RES_DARK))
	{
		if (prace_is_(RACE_VAMPIRE))
		{
			f4 &= ~(RF4_BR_DARK);
			f5 &= ~(RF5_BA_DARK);
		}
		else
		{
			if (int_outof(r_ptr, 50)) f4 &= ~(RF4_BR_DARK);
			if (int_outof(r_ptr, 50)) f5 &= ~(RF5_BA_DARK);
		}
	}

	if (smart & (SM_RES_FEAR))
	{
		f5 &= ~(RF5_SCARE);
	}

	if (smart & (SM_RES_CONF))
	{
		f5 &= ~(RF5_CONF);
		if (int_outof(r_ptr, 50)) f4 &= ~(RF4_BR_CONF);
	}

	if (smart & (SM_RES_CHAOS))
	{
		if (int_outof(r_ptr, 20)) f4 &= ~(RF4_BR_CHAO);
		if (int_outof(r_ptr, 50)) f4 &= ~(RF4_BA_CHAO);
	}

	if (smart & (SM_RES_DISEN))
	{
		if (int_outof(r_ptr, 40)) f4 &= ~(RF4_BR_DISE);
	}

	if (smart & (SM_RES_BLIND))
	{
		f5 &= ~(RF5_BLIND);
	}

	if (smart & (SM_RES_NEXUS))
	{
		if (int_outof(r_ptr, 50)) f4 &= ~(RF4_BR_NEXU);
		f6 &= ~(RF6_TELE_LEVEL);
	}

	if (smart & (SM_RES_SOUND))
	{
		if (int_outof(r_ptr, 50)) f4 &= ~(RF4_BR_SOUN);
	}

	if (smart & (SM_RES_SHARD))
	{
		if (int_outof(r_ptr, 40)) f4 &= ~(RF4_BR_SHAR);
	}

	if (smart & (SM_IMM_REFLECT))
	{
		if (int_outof(r_ptr, 150)) f5 &= ~(RF5_BO_COLD);
		if (int_outof(r_ptr, 150)) f5 &= ~(RF5_BO_FIRE);
		if (int_outof(r_ptr, 150)) f5 &= ~(RF5_BO_ACID);
		if (int_outof(r_ptr, 150)) f5 &= ~(RF5_BO_ELEC);
		if (int_outof(r_ptr, 150)) f5 &= ~(RF5_BO_NETH);
		if (int_outof(r_ptr, 150)) f5 &= ~(RF5_BO_WATE);
		if (int_outof(r_ptr, 150)) f5 &= ~(RF5_BO_MANA);
		if (int_outof(r_ptr, 150)) f5 &= ~(RF5_BO_PLAS);
		if (int_outof(r_ptr, 150)) f5 &= ~(RF5_BO_ICEE);
		if (int_outof(r_ptr, 150)) f5 &= ~(RF5_MISSILE);
		if (int_outof(r_ptr, 150)) f4 &= ~(RF4_SHOOT);
	}

	if (smart & (SM_IMM_FREE))
	{
		f5 &= ~(RF5_HOLD);
		f5 &= ~(RF5_SLOW);
	}

	if (smart & (SM_IMM_MANA))
	{
		f5 &= ~(RF5_DRAIN_MANA);
	}

	/* XXX XXX XXX No spells left? */
	/* if (!f4 && !f5 && !f6) ... */

	(*f4p) = f4;
	(*f5p) = f5;
	(*f6p) = f6;
}


/*!
 * @brief 󥹥ˤȤäƽԤɤ֤ /
 * Determine if there is a space near the player in which a summoned creature can appear
 * @param y1 ȽԤޥYɸ
 * @param x1 ȽԤޥXɸ
 * @return ԤʤTRUE֤
 */
bool summon_possible(int y1, int x1)
{
	int y, x;

	/* Start at the player's location, and check 2 grids in each dir */
	for (y = y1 - 2; y <= y1 + 2; y++)
	{
		for (x = x1 - 2; x <= x1 + 2; x++)
		{
			/* Ignore illegal locations */
			if (!in_bounds(y, x)) continue;

			/* Only check a circular area */
			if (distance(y1, x1, y, x)>2) continue;

			/* ...nor on the Pattern */
			if (pattern_tile(y, x)) continue;

			/* Require empty floor grid in line of projection */
			if (cave_empty_bold(y, x) && projectable(y1, x1, y, x) && projectable(y, x, y1, x1)) return (TRUE);
		}
	}

	return FALSE;
}


/*!
 * @brief 󥹥ˤȤäƻԤ٤֤ɤ֤ /
 * Determine if there is a space near the player in which a summoned creature can appear
 * @param m_ptr ȽԤ󥹥ι¤λȥݥ
 * @return 褬ͭʾ֤ʤTRUE֤
 */
bool raise_possible(monster_type *m_ptr)
{
	int xx, yy;
	int y = m_ptr->fy;
	int x = m_ptr->fx;
	s16b this_o_idx, next_o_idx = 0;
	cave_type *c_ptr;

	for (xx = x - 5; xx <= x + 5; xx++)
	{
		for (yy = y - 5; yy <= y + 5; yy++)
		{
			if (distance(y, x, yy, xx) > 5) continue;
			if (!los(y, x, yy, xx)) continue;
			if (!projectable(y, x, yy, xx)) continue;

			c_ptr = &cave[yy][xx];
			/* Scan the pile of objects */
			for (this_o_idx = c_ptr->o_idx; this_o_idx; this_o_idx = next_o_idx)
			{
				/* Acquire object */
				object_type *o_ptr = &o_list[this_o_idx];

				/* Acquire next object */
				next_o_idx = o_ptr->next_o_idx;

				/* Known to be worthless? */
				if (o_ptr->tval == TV_CORPSE)
				{
					if (!monster_has_hostile_align(m_ptr, 0, 0, &r_info[o_ptr->pval])) return TRUE;
				}
			}
		}
	}
	return FALSE;
}



/*!
 * @brief 󥹥ˤȤäƥܥȷˡͭʾ֤֤ /
 * Determine if a bolt spell will hit the player.
 * @param y1 ܥˡȯYɸ
 * @param x1 ܥˡȯXɸ
 * @param y2 ܥˡɸYɸ
 * @param x2 ܥˡɸXɸ
 * @param is_friend 󥹥ץ쥤䡼˳դʤ(ڥåȤͧŪ)ʤTRUEĤ
 * @return ܥȷˡͭʤTRUE֤
 * @details
 * Originally, it was possible for a friendly to shoot another friendly.\n
 * Change it so a "clean shot" means no equally friendly monster is\n
 * between the attacker and target.\n
 *\n
 * This is exactly like "projectable", but it will\n
 * return FALSE if a monster is in the way.\n
 * no equally friendly monster is\n
 * between the attacker and target.\n
 */
bool clean_shot(int y1, int x1, int y2, int x2, bool is_friend)
{
	/* Must be the same as projectable() */

	int i, y, x;

	int grid_n = 0;
	u16b grid_g[512];

	/* Check the projection path */
	grid_n = project_path(grid_g, MAX_RANGE, y1, x1, y2, x2, 0);

	/* No grid is ever projectable from itself */
	if (!grid_n) return (FALSE);

	/* Final grid */
	y = GRID_Y(grid_g[grid_n-1]);
	x = GRID_X(grid_g[grid_n-1]);

	/* May not end in an unrequested grid */
	if ((y != y2) || (x != x2)) return (FALSE);

	for (i = 0; i < grid_n; i++)
	{
		y = GRID_Y(grid_g[i]);
		x = GRID_X(grid_g[i]);

		if ((cave[y][x].m_idx > 0) && !((y == y2) && (x == x2)))
		{
			monster_type *m_ptr = &m_list[cave[y][x].m_idx];
			if (is_friend == is_pet(m_ptr))
			{
				return (FALSE);
			}
		}
		/* Pets may not shoot through the character - TNB */
		if (player_bold(y, x))
		{
			if (is_friend) return (FALSE);
		}
	}

	return (TRUE);
}

/*!
 * @brief 󥹥Υܥȷˡ /
 * Cast a bolt at the player Stop if we hit a monster Affect monsters and the player
 * @param m_idx 󥹥ID
 * @param y ɸYɸ
 * @param x ɸXɸ
 * @param typ °ID
 * @param dam_hp 
 * @param monspell 󥹥ˡID
 * @param target_type 󥹥󥹥طĤʤMONSTER_TO_MONSTER󥹥ץ쥤䡼ʤMONSTER_TO_PLAYER
 * @return ʤ
 */
void bolt(int m_idx, int y, int x, int typ, int dam_hp, int monspell, int target_type)
  {
    int flg;
    bool learnable = spell_learnable(m_idx);

    switch (target_type)
    {
    case MONSTER_TO_MONSTER:
        flg = PROJECT_STOP | PROJECT_KILL;
        break;
    case MONSTER_TO_PLAYER:
        flg = PROJECT_STOP | PROJECT_KILL | PROJECT_PLAYER;
        break;
    }
	if (typ != GF_ARROW) flg  |= PROJECT_REFLECTABLE;

	/* Target the player with a bolt attack */
	(void)project(m_idx, 0, y, x, dam_hp, typ, flg, (learnable ? monspell : -1));
}

/*!
 * @brief 󥹥Υӡ෿ˡ /
 * @param m_idx 󥹥ID
 * @param y ɸYɸ
 * @param x ɸXɸ
 * @param typ °ID
 * @param dam_hp 
 * @param monspell 󥹥ˡID
 * @param target_type 󥹥󥹥طĤʤMONSTER_TO_MONSTER󥹥ץ쥤䡼ʤMONSTER_TO_PLAYER
 * @return ʤ
 */
void beam(int m_idx, int y, int x, int typ, int dam_hp, int monspell, int target_type)
{
    int flg;
    bool learnable = spell_learnable(m_idx);

    switch (target_type)
    {
    case MONSTER_TO_MONSTER:
        flg = PROJECT_BEAM | PROJECT_KILL | PROJECT_THRU;
        break;
    case MONSTER_TO_PLAYER:
        flg = PROJECT_BEAM | PROJECT_KILL | PROJECT_THRU | PROJECT_PLAYER;
        break;
    }

	/* Target the player with a bolt attack */
	(void)project(m_idx, 0, y, x, dam_hp, typ, flg, (learnable ? monspell : -1));
}


/*!
 * @brief 󥹥Υܡ뷿֥쥹ˡ /
 * Cast a breath (or ball) attack at the player Pass over any monsters that may be in the way Affect grids, objects, monsters, and the player
 * @param y ɸYɸ
 * @param x ɸXɸ
 * @param m_idx 󥹥ID
 * @param typ °ID
 * @param dam_hp 
 * @param rad Ⱦ
 * @param breath TRUEʤХ֥쥹FALSEʤХܡ
 * @param monspell 󥹥ˡID
 * @param target_type 󥹥󥹥طĤʤMONSTER_TO_MONSTER󥹥ץ쥤䡼ʤMONSTER_TO_PLAYER
 * @return ʤ
 */
void breath(int y, int x, int m_idx, int typ, int dam_hp, int rad, bool breath, int monspell, int target_type)
{
    monster_type *m_ptr = &m_list[m_idx];
    monster_race *r_ptr = &r_info[m_ptr->r_idx];
    bool learnable = spell_learnable(m_idx);
	int flg;

    switch (target_type)
    {
        case MONSTER_TO_MONSTER:
            flg = PROJECT_GRID | PROJECT_ITEM | PROJECT_KILL;
            break;
        case MONSTER_TO_PLAYER:
            flg = PROJECT_GRID | PROJECT_ITEM | PROJECT_KILL | PROJECT_PLAYER;
            break;
    }

	/* Determine the radius of the blast */
	if ((rad < 1) && breath) rad = (r_ptr->flags2 & (RF2_POWERFUL)) ? 3 : 2;

	/* Handle breath attacks */
	if (breath) rad = 0 - rad;

	switch (typ)
	{
	case GF_ROCKET:
		flg |= PROJECT_STOP;
		break;
	case GF_DRAIN_MANA:
	case GF_MIND_BLAST:
	case GF_BRAIN_SMASH:
	case GF_CAUSE_1:
	case GF_CAUSE_2:
	case GF_CAUSE_3:
	case GF_CAUSE_4:
	case GF_HAND_DOOM:
		flg |= (PROJECT_HIDE | PROJECT_AIMED);
		break;
	}

	/* Target the player with a ball attack */
	(void)project(m_idx, rad, y, x, dam_hp, typ, flg, (learnable ? monspell : -1));
}

/*!
 * @brief 󥹥Υܡ뷿֥쥹ˡ /
 * @param power ʳ
 * @param o_ptr 򤫤֥Ȥι¤λȥݥ
 * @return ͿID
 */
u32b get_curse(int power, object_type *o_ptr)
{
	u32b new_curse;

	while(1)
	{
		new_curse = (1 << (randint0(MAX_CURSE)+4));
		if (power == 2)
		{
			if (!(new_curse & TRC_HEAVY_MASK)) continue;
		}
		else if (power == 1)
		{
			if (new_curse & TRC_SPECIAL_MASK) continue;
		}
		else if (power == 0)
		{
			if (new_curse & TRC_HEAVY_MASK) continue;
		}
		if (new_curse == TRC_LOW_MELEE && !object_is_weapon(o_ptr)) continue;
		if (new_curse == TRC_LOW_AC && !object_is_armour(o_ptr)) continue;
		break;
	}
	return new_curse;
}

/*!
 * @brief ؤμղȽղý /
 * @param chance δܳΨ
 * @param heavy_chance Ť뤫ݤ
 * @return ʤ
 */
void curse_equipment(int chance, int heavy_chance)
{
	bool        changed = FALSE;
	int         curse_power = 0;
	u32b        new_curse;
	u32b oflgs[TR_FLAG_SIZE];
	object_type *o_ptr = &inventory[INVEN_RARM + randint0(12)];
	char o_name[MAX_NLEN];

	if (randint1(100) > chance) return;

	if (!o_ptr->k_idx) return;

	object_flags(o_ptr, oflgs);

	object_desc(o_name, o_ptr, (OD_OMIT_PREFIX | OD_NAME_ONLY));

	/* Extra, biased saving throw for blessed items */
	if (have_flag(oflgs, TR_BLESSED) && (randint1(888) > chance))
	{
#ifdef JP
msg_format("%sϼķ֤", o_name,
#else
		msg_format("Your %s resist%s cursing!", o_name,
#endif

			((o_ptr->number > 1) ? "" : "s"));
		/* Hmmm -- can we wear multiple items? If not, this is unnecessary */
		return;
	}

	if ((randint1(100) <= heavy_chance) &&
	    (object_is_artifact(o_ptr) || object_is_ego(o_ptr)))
	{
		if (!(o_ptr->curse_flags & TRC_HEAVY_CURSE))
			changed = TRUE;
		o_ptr->curse_flags |= TRC_HEAVY_CURSE;
		o_ptr->curse_flags |= TRC_CURSED;
		curse_power++;
	}
	else
	{
		if (!object_is_cursed(o_ptr))
			changed = TRUE;
		o_ptr->curse_flags |= TRC_CURSED;
	}
	if (heavy_chance >= 50) curse_power++;

	new_curse = get_curse(curse_power, o_ptr);
	if (!(o_ptr->curse_flags & new_curse))
	{
		changed = TRUE;
		o_ptr->curse_flags |= new_curse;
	}

	if (changed)
	{
#ifdef JP
msg_format("դ餬%sȤޤ...", o_name);
#else
		msg_format("There is a malignant black aura surrounding %s...", o_name);
#endif

		o_ptr->feeling = FEEL_NONE;
	}
	p_ptr->update |= (PU_BONUS);
}


/*!
 * @brief IDͤ󥹥ˡIDɤ֤ /
 * Return TRUE if a spell is good for hurting the player (directly).
 * @param spell ȽоݤID
 * @return IDʤTRUE֤
 */
static bool spell_attack(byte spell)
{
	/* All RF4 spells hurt (except for shriek and dispel) */
	if (spell < 128 && spell > 98) return (TRUE);

	/* Various "ball" spells */
	if (spell >= 128 && spell <= 128 + 8) return (TRUE);

	/* "Cause wounds" and "bolt" spells */
	if (spell >= 128 + 12 && spell < 128 + 27) return (TRUE);

	/* Hand of Doom */
	if (spell == 160 + 1) return (TRUE);

	/* Psycho-Spear */
	if (spell == 160 + 11) return (TRUE);

	/* Doesn't hurt */
	return (FALSE);
}


/*!
 * @brief IDͤŪŬ󥹥ˡIDɤ֤ /
 * Return TRUE if a spell is good for escaping.
 * @param spell ȽоݤID
 * @return ŬˡIDʤTRUE֤
 */
static bool spell_escape(byte spell)
{
	/* Blink or Teleport */
	if (spell == 160 + 4 || spell == 160 + 5) return (TRUE);

	/* Teleport the player away */
	if (spell == 160 + 9 || spell == 160 + 10) return (TRUE);

	/* Isn't good for escaping */
	return (FALSE);
}

/*!
 * @brief IDͤ˸ŪŬ󥹥ˡIDɤ֤ /
 * Return TRUE if a spell is good for annoying the player.
 * @param spell ȽоݤID
 * @return ŬˡIDʤTRUE֤
 */
static bool spell_annoy(byte spell)
{
	/* Shriek */
	if (spell == 96 + 0) return (TRUE);

	/* Brain smash, et al (added curses) */
	if (spell >= 128 + 9 && spell <= 128 + 14) return (TRUE);

	/* Scare, confuse, blind, slow, paralyze */
	if (spell >= 128 + 27 && spell <= 128 + 31) return (TRUE);

	/* Teleport to */
	if (spell == 160 + 8) return (TRUE);

	/* Teleport level */
	if (spell == 160 + 10) return (TRUE);

	/* Darkness, make traps, cause amnesia */
	if (spell >= 160 + 12 && spell <= 160 + 14) return (TRUE);

	/* Doesn't annoy */
	return (FALSE);
}

/*!
 * @brief IDͤΥ󥹥ˡIDɤ֤ /
 * Return TRUE if a spell is good for annoying the player.
 * @param spell ȽоݤID
 * @return ˡIDʤTRUE֤
 */
static bool spell_summon(byte spell)
{
	/* All summon spells */
	if (spell >= 160 + 16) return (TRUE);

	/* Doesn't summon */
	return (FALSE);
}


/*!
 * @brief IDͤɤ֤ /
 * Return TRUE if a spell is good for annoying the player.
 * @param spell ȽоݤID
 * @return νʤTRUE֤
 */
static bool spell_raise(byte spell)
{
	/* All raise-dead spells */
	if (spell == 160 + 15) return (TRUE);

	/* Doesn't summon */
	return (FALSE);
}

/*!
 * @brief IDͤŪʥ󥹥ˡIDɤ֤ /
 * Return TRUE if a spell is good in a tactical situation.
 * @param spell ȽоݤID
 * @return ŪˡIDʤTRUE֤
 */
static bool spell_tactic(byte spell)
{
	/* Blink */
	if (spell == 160 + 4) return (TRUE);

	/* Not good */
	return (FALSE);
}

/*!
 * @brief ID̵ͤŨ󥹥ˡIDɤ֤ /
 * Return TRUE if a spell makes invulnerable.
 * @param spell ȽоݤID
 * @return ˡIDʤTRUE֤
 */
static bool spell_invulner(byte spell)
{
	/* Invulnerability */
	if (spell == 160 + 3) return (TRUE);

	/* No invulnerability */
	return (FALSE);
}

/*!
 * @brief IDͤ®󥹥ˡIDɤ֤ /
 * Return TRUE if a spell hastes.
 * @param spell ȽоݤID
 * @return ˡIDʤTRUE֤
 */
static bool spell_haste(byte spell)
{
	/* Haste self */
	if (spell == 160 + 0) return (TRUE);

	/* Not a haste spell */
	return (FALSE);
}


/*!
 * @brief IDͤߤԤ󥹥ˡIDɤ֤ /
 * Return TRUE if a spell world.
 * @param spell ȽоݤID
 * @return ˡIDʤTRUE֤
 */
static bool spell_world(byte spell)
{
	if (spell == 160 + 6) return (TRUE);
	return (FALSE);
}


/*!
 * @brief ID̸̤ͤΥ󥹥ˡIDɤ֤ /
 * Return TRUE if a spell special.
 * @param spell ȽоݤID
 * @return ̸ˡIDʤTRUE֤
 */
static bool spell_special(byte spell)
{
	if (p_ptr->inside_battle) return FALSE;
	if (spell == 160 + 7) return (TRUE);
	return (FALSE);
}


/*!
 * @brief IDͤηΥ󥹥ˡIDɤ֤ /
 * Return TRUE if a spell psycho-spear.
 * @param spell ȽоݤID
 * @return ηIDʤTRUE֤
 */
static bool spell_psy_spe(byte spell)
{
	/* world */
	if (spell == 160 + 11) return (TRUE);

	/* Not a haste spell */
	return (FALSE);
}


/*!
 * @brief IDͤˡɤ֤ /
 * Return TRUE if a spell is good for healing.
 * @param spell ȽоݤID
 * @return ˡIDʤTRUE֤
 */
static bool spell_heal(byte spell)
{
	/* Heal */
	if (spell == 160 + 2) return (TRUE);

	/* No healing */
	return (FALSE);
}


/*!
 * @brief IDͤϾõɤ֤ /
 * Return TRUE if a spell is good for dispel.
 * @param spell ȽоݤID
 * @return ϾõIDʤTRUE֤
 */
static bool spell_dispel(byte spell)
{
	/* Dispel */
	if (spell == 96 + 2) return (TRUE);

	/* No dispel */
	return (FALSE);
}


/*!
 * @brief 󥹥ץ쥤䡼ϾõͿ٤Ƚꤹ롼
 * Check should monster cast dispel spell.
 * @param m_idx 󥹥ι¤ID
 * @return Ͼõ򤫤٤ʤTRUE֤
 */
bool dispel_check(int m_idx)
{
	monster_type *m_ptr = &m_list[m_idx];
	monster_race *r_ptr = &r_info[m_ptr->r_idx];

	/* Invulnabilty (including the song) */
	if (IS_INVULN()) return (TRUE);

	/* Wraith form */
	if (p_ptr->wraith_form) return (TRUE);

	/* Shield */
	if (p_ptr->shield) return (TRUE);

	/* Magic defence */
	if (p_ptr->magicdef) return (TRUE);

	/* Multi Shadow */
	if (p_ptr->multishadow) return (TRUE);

	/* Robe of dust */
	if (p_ptr->dustrobe) return (TRUE);

	/* Berserk Strength */
	if (p_ptr->shero && (p_ptr->pclass != CLASS_BERSERKER)) return (TRUE);

	/* Demon Lord */
	if (p_ptr->mimic_form == MIMIC_DEMON_LORD) return (TRUE);

	/* Elemental resistances */
	if (r_ptr->flags4 & RF4_BR_ACID)
	{
		if (!p_ptr->immune_acid && (p_ptr->oppose_acid || music_singing(MUSIC_RESIST))) return (TRUE);
		if (p_ptr->special_defense & DEFENSE_ACID) return (TRUE);
	}

	if (r_ptr->flags4 & RF4_BR_FIRE)
	{
		if (!((p_ptr->prace == RACE_DEMON) && p_ptr->lev > 44))
		{
			if (!p_ptr->immune_fire && (p_ptr->oppose_fire || music_singing(MUSIC_RESIST))) return (TRUE);
			if (p_ptr->special_defense & DEFENSE_FIRE) return (TRUE);
		}
	}

	if (r_ptr->flags4 & RF4_BR_ELEC)
	{
		if (!p_ptr->immune_elec && (p_ptr->oppose_elec || music_singing(MUSIC_RESIST))) return (TRUE);
		if (p_ptr->special_defense & DEFENSE_ELEC) return (TRUE);
	}

	if (r_ptr->flags4 & RF4_BR_COLD)
	{
		if (!p_ptr->immune_cold && (p_ptr->oppose_cold || music_singing(MUSIC_RESIST))) return (TRUE);
		if (p_ptr->special_defense & DEFENSE_COLD) return (TRUE);
	}

	if (r_ptr->flags4 & (RF4_BR_POIS | RF4_BR_NUKE))
	{
		if (!((p_ptr->pclass == CLASS_NINJA) && p_ptr->lev > 44))
		{
			if (p_ptr->oppose_pois || music_singing(MUSIC_RESIST)) return (TRUE);
			if (p_ptr->special_defense & DEFENSE_POIS) return (TRUE);
		}
	}

	/* Ultimate resistance */
	if (p_ptr->ult_res) return (TRUE);

	/* Potion of Neo Tsuyosi special */
	if (p_ptr->tsuyoshi) return (TRUE);

	/* Elemental Brands */
	if ((p_ptr->special_attack & ATTACK_ACID) && !(r_ptr->flagsr & RFR_EFF_IM_ACID_MASK)) return (TRUE);
	if ((p_ptr->special_attack & ATTACK_FIRE) && !(r_ptr->flagsr & RFR_EFF_IM_FIRE_MASK)) return (TRUE);
	if ((p_ptr->special_attack & ATTACK_ELEC) && !(r_ptr->flagsr & RFR_EFF_IM_ELEC_MASK)) return (TRUE);
	if ((p_ptr->special_attack & ATTACK_COLD) && !(r_ptr->flagsr & RFR_EFF_IM_COLD_MASK)) return (TRUE);
	if ((p_ptr->special_attack & ATTACK_POIS) && !(r_ptr->flagsr & RFR_EFF_IM_POIS_MASK)) return (TRUE);

	/* Speed */
	if (p_ptr->pspeed < 145)
	{
		if (IS_FAST()) return (TRUE);
	}

	/* Light speed */
	if (p_ptr->lightspeed && (m_ptr->mspeed < 136)) return (TRUE);

	if (p_ptr->riding && (m_list[p_ptr->riding].mspeed < 135))
	{
		if (MON_FAST(&m_list[p_ptr->riding])) return (TRUE);
	}

	/* No need to cast dispel spell */
	return (FALSE);
}


/*!
 * @brief 󥹥ˡ롼
 * Have a monster choose a spell from a list of "useful" spells.
 * @param m_idx 󥹥ι¤ID
 * @param spells ˡIDޤȤ᤿
 * @param num spellsĹ
 * @return 򤷤󥹥ˡID
 * @details
 * Note that this list does NOT include spells that will just hit\n
 * other monsters, and the list is restricted when the monster is\n
 * "desperate".  Should that be the job of this function instead?\n
 *\n
 * Stupid monsters will just pick a spell randomly.  Smart monsters\n
 * will choose more "intelligently".\n
 *\n
 * Use the helper functions above to put spells into categories.\n
 *\n
 * This function may well be an efficiency bottleneck.\n
 */
static int choose_attack_spell(int m_idx, byte spells[], byte num)
{
	monster_type *m_ptr = &m_list[m_idx];
	monster_race *r_ptr = &r_info[m_ptr->r_idx];

	byte escape[96], escape_num = 0;
	byte attack[96], attack_num = 0;
	byte summon[96], summon_num = 0;
	byte tactic[96], tactic_num = 0;
	byte annoy[96], annoy_num = 0;
	byte invul[96], invul_num = 0;
	byte haste[96], haste_num = 0;
	byte world[96], world_num = 0;
	byte special[96], special_num = 0;
	byte psy_spe[96], psy_spe_num = 0;
	byte raise[96], raise_num = 0;
	byte heal[96], heal_num = 0;
	byte dispel[96], dispel_num = 0;

	int i;

	/* Stupid monsters choose randomly */
	if (r_ptr->flags2 & (RF2_STUPID))
	{
		/* Pick at random */
		return (spells[randint0(num)]);
	}

	/* Categorize spells */
	for (i = 0; i < num; i++)
	{
		/* Escape spell? */
		if (spell_escape(spells[i])) escape[escape_num++] = spells[i];

		/* Attack spell? */
		if (spell_attack(spells[i])) attack[attack_num++] = spells[i];

		/* Summon spell? */
		if (spell_summon(spells[i])) summon[summon_num++] = spells[i];

		/* Tactical spell? */
		if (spell_tactic(spells[i])) tactic[tactic_num++] = spells[i];

		/* Annoyance spell? */
		if (spell_annoy(spells[i])) annoy[annoy_num++] = spells[i];

		/* Invulnerability spell? */
		if (spell_invulner(spells[i])) invul[invul_num++] = spells[i];

		/* Haste spell? */
		if (spell_haste(spells[i])) haste[haste_num++] = spells[i];

		/* World spell? */
		if (spell_world(spells[i])) world[world_num++] = spells[i];

		/* Special spell? */
		if (spell_special(spells[i])) special[special_num++] = spells[i];

		/* Psycho-spear spell? */
		if (spell_psy_spe(spells[i])) psy_spe[psy_spe_num++] = spells[i];

		/* Raise-dead spell? */
		if (spell_raise(spells[i])) raise[raise_num++] = spells[i];

		/* Heal spell? */
		if (spell_heal(spells[i])) heal[heal_num++] = spells[i];

		/* Dispel spell? */
		if (spell_dispel(spells[i])) dispel[dispel_num++] = spells[i];
	}

	/*** Try to pick an appropriate spell type ***/

	/* world */
	if (world_num && (randint0(100) < 15) && !world_monster)
	{
		/* Choose haste spell */
		return (world[randint0(world_num)]);
	}

	/* special */
	if (special_num)
	{
		bool success = FALSE;
		switch(m_ptr->r_idx)
		{
			case MON_BANOR:
			case MON_LUPART:
				if ((m_ptr->hp < m_ptr->maxhp / 2) && r_info[MON_BANOR].max_num && r_info[MON_LUPART].max_num) success = TRUE;
				break;
			default: break;
		}
		if (success) return (special[randint0(special_num)]);
	}

	/* Still hurt badly, couldn't flee, attempt to heal */
	if (m_ptr->hp < m_ptr->maxhp / 3 && one_in_(2))
	{
		/* Choose heal spell if possible */
		if (heal_num) return (heal[randint0(heal_num)]);
	}

	/* Hurt badly or afraid, attempt to flee */
	if (((m_ptr->hp < m_ptr->maxhp / 3) || MON_MONFEAR(m_ptr)) && one_in_(2))
	{
		/* Choose escape spell if possible */
		if (escape_num) return (escape[randint0(escape_num)]);
	}

	/* special */
	if (special_num)
	{
		bool success = FALSE;
		switch (m_ptr->r_idx)
		{
			case MON_OHMU:
			case MON_BANOR:
			case MON_LUPART:
				break;
			case MON_BANORLUPART:
				if (randint0(100) < 70) success = TRUE;
				break;
			case MON_ROLENTO:
				if (randint0(100) < 40) success = TRUE;
				break;
			default:
				if (randint0(100) < 50) success = TRUE;
				break;
		}
		if (success) return (special[randint0(special_num)]);
	}

	/* Player is close and we have attack spells, blink away */
	if ((distance(py, px, m_ptr->fy, m_ptr->fx) < 4) && (attack_num || (r_ptr->flags6 & RF6_TRAPS)) && (randint0(100) < 75) && !world_monster)
	{
		/* Choose tactical spell */
		if (tactic_num) return (tactic[randint0(tactic_num)]);
	}

	/* Summon if possible (sometimes) */
	if (summon_num && (randint0(100) < 40))
	{
		/* Choose summon spell */
		return (summon[randint0(summon_num)]);
	}

	/* dispel */
	if (dispel_num && one_in_(2))
	{
		/* Choose dispel spell if possible */
		if (dispel_check(m_idx))
		{
			return (dispel[randint0(dispel_num)]);
		}
	}

	/* Raise-dead if possible (sometimes) */
	if (raise_num && (randint0(100) < 40))
	{
		/* Choose raise-dead spell */
		return (raise[randint0(raise_num)]);
	}

	/* Attack spell (most of the time) */
	if (IS_INVULN())
	{
		if (psy_spe_num && (randint0(100) < 50))
		{
			/* Choose attack spell */
			return (psy_spe[randint0(psy_spe_num)]);
		}
		else if (attack_num && (randint0(100) < 40))
		{
			/* Choose attack spell */
			return (attack[randint0(attack_num)]);
		}
	}
	else if (attack_num && (randint0(100) < 85))
	{
		/* Choose attack spell */
		return (attack[randint0(attack_num)]);
	}

	/* Try another tactical spell (sometimes) */
	if (tactic_num && (randint0(100) < 50) && !world_monster)
	{
		/* Choose tactic spell */
		return (tactic[randint0(tactic_num)]);
	}

	/* Cast globe of invulnerability if not already in effect */
	if (invul_num && !m_ptr->mtimed[MTIMED_INVULNER] && (randint0(100) < 50))
	{
		/* Choose Globe of Invulnerability */
		return (invul[randint0(invul_num)]);
	}

	/* We're hurt (not badly), try to heal */
	if ((m_ptr->hp < m_ptr->maxhp * 3 / 4) && (randint0(100) < 25))
	{
		/* Choose heal spell if possible */
		if (heal_num) return (heal[randint0(heal_num)]);
	}

	/* Haste self if we aren't already somewhat hasted (rarely) */
	if (haste_num && (randint0(100) < 20) && !MON_FAST(m_ptr))
	{
		/* Choose haste spell */
		return (haste[randint0(haste_num)]);
	}

	/* Annoy player (most of the time) */
	if (annoy_num && (randint0(100) < 80))
	{
		/* Choose annoyance spell */
		return (annoy[randint0(annoy_num)]);
	}

	/* Choose no spell */
	return (0);
}


/*!
 * @brief IDͤŪü쵻ǽɤ֤ /
 * Return TRUE if a spell is inate spell.
 * @param spell ȽоݤID
 * @return Ūü쵻ǽʤTRUE֤
 */
bool spell_is_inate(u16b spell)
{
	if (spell < 32 * 4) /* Set RF4 */
	{
		if ((1L << (spell - 32 * 3)) & RF4_NOMAGIC_MASK) return TRUE;
	}
	else if (spell < 32 * 5) /* Set RF5 */
	{
		if ((1L << (spell - 32 * 4)) & RF5_NOMAGIC_MASK) return TRUE;
	}
	else if (spell < 32 * 6) /* Set RF6 */
	{
		if ((1L << (spell - 32 * 5)) & RF6_NOMAGIC_MASK) return TRUE;
	}

	/* This spell is not "inate" */
	return FALSE;
}


/*!
 * @brief 󥹥ץ쥤䡼˥᡼Ϳ뤿κŬʺɸ򻻽Ф /
 * @param m_ptr ǽѤ󥹥¤Τλȥݥ
 * @param yp ŬɸYɸ֤ȥݥ
 * @param xp ŬɸXɸ֤ȥݥ
 * @param f_flag Τ򤱤Ϸνե饰
 * @param path_check Ƚꤹ뤿δؿݥ
 * @return ͭʺɸäTRUE֤
 */
static bool adjacent_grid_check(monster_type *m_ptr, int *yp, int *xp,
	int f_flag, bool (*path_check)(int, int, int, int))
{
	int i;
	int tonari;
	static int tonari_y[4][8] = {{-1, -1, -1,  0,  0,  1,  1,  1},
			                     {-1, -1, -1,  0,  0,  1,  1,  1},
			                     { 1,  1,  1,  0,  0, -1, -1, -1},
			                     { 1,  1,  1,  0,  0, -1, -1, -1}};
	static int tonari_x[4][8] = {{-1,  0,  1, -1,  1, -1,  0,  1},
			                     { 1,  0, -1,  1, -1,  1,  0, -1},
			                     {-1,  0,  1, -1,  1, -1,  0,  1},
			                     { 1,  0, -1,  1, -1,  1,  0, -1}};

	if (m_ptr->fy < py && m_ptr->fx < px) tonari = 0;
	else if (m_ptr->fy < py) tonari = 1;
	else if (m_ptr->fx < px) tonari = 2;
	else tonari = 3;

	for (i = 0; i < 8; i++)
	{
		int next_x = *xp + tonari_x[tonari][i];
		int next_y = *yp + tonari_y[tonari][i];
		cave_type *c_ptr;

		/* Access the next grid */
		c_ptr = &cave[next_y][next_x];

		/* Skip this feature */
		if (!cave_have_flag_grid(c_ptr, f_flag)) continue;

		if (path_check(m_ptr->fy, m_ptr->fx, next_y, next_x))
		{
			*yp = next_y;
			*xp = next_x;
			return TRUE;
		}
	}

	return FALSE;
}

#define DO_SPELL_NONE    0
#define DO_SPELL_BR_LITE 1
#define DO_SPELL_BR_DISI 2
#define DO_SPELL_BA_LITE 3

/*!
 * @brief 󥹥ü쵻ǽᥤ롼 /
 * Creatures can cast spells, shoot missiles, and breathe.
 * @param m_idx 󥹥¤ID
 * @return ºݤü쵻ǽѤTRUE֤
 * @details
 * Returns "TRUE" if a spell (or whatever) was (successfully) cast.\n
 *\n
 * XXX XXX XXX This function could use some work, but remember to\n
 * keep it as optimized as possible, while retaining generic code.\n
 *\n
 * Verify the various "blind-ness" checks in the code.\n
 *\n
 * XXX XXX XXX Note that several effects should really not be "seen"\n
 * if the player is blind.  See also "effects.c" for other "mistakes".\n
 *\n
 * Perhaps monsters should breathe at locations *near* the player,\n
 * since this would allow them to inflict "partial" damage.\n
 *\n
 * Perhaps smart monsters should decline to use "bolt" spells if\n
 * there is a monster in the way, unless they wish to kill it.\n
 *\n
 * Note that, to allow the use of the "track_target" option at some\n
 * later time, certain non-optimal things are done in the code below,\n
 * including explicit checks against the "direct" variable, which is\n
 * currently always true by the time it is checked, but which should\n
 * really be set according to an explicit "projectable()" test, and\n
 * the use of generic "x,y" locations instead of the player location,\n
 * with those values being initialized with the player location.\n
 *\n
 * It will not be possible to "correctly" handle the case in which a\n
 * monster attempts to attack a location which is thought to contain\n
 * the player, but which in fact is nowhere near the player, since this\n
 * might induce all sorts of messages about the attack itself, and about\n
 * the effects of the attack, which the player might or might not be in\n
 * a position to observe.  Thus, for simplicity, it is probably best to\n
 * only allow "faulty" attacks by a monster if one of the important grids\n
 * (probably the initial or final grid) is in fact in view of the player.\n
 * It may be necessary to actually prevent spell attacks except when the\n
 * monster actually has line of sight to the player.  Note that a monster\n
 * could be left in a bizarre situation after the player ducked behind a\n
 * pillar and then teleported away, for example.\n
 *\n
 * @note
 * that certain spell attacks do not use the "project()" function\n
 * but "simulate" it via the "direct" variable, which is always at least\n
 * as restrictive as the "project()" function.  This is necessary to\n
 * prevent "blindness" attacks and such from bending around walls, etc,\n
 * and to allow the use of the "track_target" option in the future.\n
 *\n
 * Note that this function attempts to optimize the use of spells for the\n
 * cases in which the monster has no spells, or has spells but cannot use\n
 * them, or has spells but they will have no "useful" effect.  Note that\n
 * this function has been an efficiency bottleneck in the past.\n
 *\n
 * Note the special "MFLAG_NICE" flag, which prevents a monster from using\n
 * any spell attacks until the player has had a single chance to move.\n
 */
bool make_attack_spell(int m_idx)
{
	int             k, thrown_spell = 0, rlev, failrate;
	byte            spell[96], num = 0;
	u32b            f4, f5, f6;
	monster_type    *m_ptr = &m_list[m_idx];
	monster_race    *r_ptr = &r_info[m_ptr->r_idx];
	char            m_name[80];
#ifndef JP
	char            m_poss[80];
#endif
	bool            no_inate = FALSE;
	bool            do_spell = DO_SPELL_NONE;
	int             dam = 0;
	u32b mode = 0L;
	int s_num_6 = (easy_band ? 2 : 6);
	int s_num_4 = (easy_band ? 1 : 4);
	int rad = 0; //For elemental spells

	/* Target location */
	int x = px;
	int y = py;

	/* Target location for lite breath */
	int x_br_lite = 0;
	int y_br_lite = 0;

	/* Summon count */
	int count = 0;

	/* Extract the blind-ness */
	bool blind = (p_ptr->blind ? TRUE : FALSE);

	/* Extract the "see-able-ness" */
	bool seen = (!blind && m_ptr->ml);
	bool maneable = player_has_los_bold(m_ptr->fy, m_ptr->fx);

	/* Check "projectable" */
	bool direct;

	bool in_no_magic_dungeon = (d_info[dungeon_type].flags1 & DF1_NO_MAGIC) && dun_level
		&& (!p_ptr->inside_quest || is_fixed_quest_idx(p_ptr->inside_quest));

	bool can_use_lite_area = FALSE;

	bool can_remember;

	/* Cannot cast spells when confused */
	if (MON_CONFUSED(m_ptr))
	{
		reset_target(m_ptr);
		return (FALSE);
	}

	/* Cannot cast spells when nice */
	if (m_ptr->mflag & MFLAG_NICE) return (FALSE);
	if (!is_hostile(m_ptr)) return (FALSE);


	/* Sometimes forbid inate attacks (breaths) */
	if (randint0(100) >= (r_ptr->freq_spell * 2)) no_inate = TRUE;

	/* XXX XXX XXX Handle "track_target" option (?) */


	/* Extract the racial spell flags */
	f4 = r_ptr->flags4;
	f5 = r_ptr->flags5;
	f6 = r_ptr->flags6;

	/*** require projectable player ***/

	/* Check range */
	if ((m_ptr->cdis > MAX_RANGE) && !m_ptr->target_y) return (FALSE);

	/* Check path for lite breath */
	if (f4 & RF4_BR_LITE)
	{
		y_br_lite = y;
		x_br_lite = x;

		if (los(m_ptr->fy, m_ptr->fx, y_br_lite, x_br_lite))
		{
			feature_type *f_ptr = &f_info[cave[y_br_lite][x_br_lite].feat];

			if (!have_flag(f_ptr->flags, FF_LOS))
			{
				if (have_flag(f_ptr->flags, FF_PROJECT) && one_in_(2)) f4 &= ~(RF4_BR_LITE);
			}
		}

		/* Check path to next grid */
		else if (!adjacent_grid_check(m_ptr, &y_br_lite, &x_br_lite, FF_LOS, los)) f4 &= ~(RF4_BR_LITE);

		/* Don't breath lite to the wall if impossible */
		if (!(f4 & RF4_BR_LITE))
		{
			y_br_lite = 0;
			x_br_lite = 0;
		}
	}

	/* Check path */
	if (projectable(m_ptr->fy, m_ptr->fx, y, x))
	{
		feature_type *f_ptr = &f_info[cave[y][x].feat];

		if (!have_flag(f_ptr->flags, FF_PROJECT))
		{
			/* Breath disintegration to the wall if possible */
			if ((f4 & RF4_BR_DISI) && have_flag(f_ptr->flags, FF_HURT_DISI) && one_in_(2)) do_spell = DO_SPELL_BR_DISI;

			/* Breath lite to the transparent wall if possible */
			else if ((f4 & RF4_BR_LITE) && have_flag(f_ptr->flags, FF_LOS) && one_in_(2)) do_spell = DO_SPELL_BR_LITE;
		}
	}

	/* Check path to next grid */
	else
	{
		bool success = FALSE;

		if ((f4 & RF4_BR_DISI) && (m_ptr->cdis < MAX_RANGE/2) &&
		    in_disintegration_range(m_ptr->fy, m_ptr->fx, y, x) &&
		    (one_in_(10) || (projectable(y, x, m_ptr->fy, m_ptr->fx) && one_in_(2))))
		{
			do_spell = DO_SPELL_BR_DISI;
			success = TRUE;
		}
		else if ((f4 & RF4_BR_LITE) && (m_ptr->cdis < MAX_RANGE/2) &&
		    los(m_ptr->fy, m_ptr->fx, y, x) && one_in_(5))
		{
			do_spell = DO_SPELL_BR_LITE;
			success = TRUE;
		}
		else if ((f5 & RF5_BA_LITE) && (m_ptr->cdis <= MAX_RANGE))
		{
			int by = y, bx = x;
			get_project_point(m_ptr->fy, m_ptr->fx, &by, &bx, 0L);
			if ((distance(by, bx, y, x) <= 3) && los(by, bx, y, x) && one_in_(5))
			{
				do_spell = DO_SPELL_BA_LITE;
				success = TRUE;
			}
		}

		if (!success) success = adjacent_grid_check(m_ptr, &y, &x, FF_PROJECT, projectable);

		if (!success)
		{
			if (m_ptr->target_y && m_ptr->target_x)
			{
				y = m_ptr->target_y;
				x = m_ptr->target_x;
				f4 &= (RF4_INDIRECT_MASK);
				f5 &= (RF5_INDIRECT_MASK);
				f6 &= (RF6_INDIRECT_MASK);
				success = TRUE;
			}

			if (y_br_lite && x_br_lite && (m_ptr->cdis < MAX_RANGE/2) && one_in_(5))
			{
				if (!success)
				{
					y = y_br_lite;
					x = x_br_lite;
					do_spell = DO_SPELL_BR_LITE;
					success = TRUE;
				}
				else f4 |= (RF4_BR_LITE);
			}
		}

		/* No spells */
		if (!success) return FALSE;
	}

	reset_target(m_ptr);

	/* Extract the monster level */
	rlev = ((r_ptr->level >= 1) ? r_ptr->level : 1);

	/* Forbid inate attacks sometimes */
	if (no_inate)
	{
		f4 &= ~(RF4_NOMAGIC_MASK);
		f5 &= ~(RF5_NOMAGIC_MASK);
		f6 &= ~(RF6_NOMAGIC_MASK);
	}

	if (f6 & RF6_DARKNESS)
	{
		if ((p_ptr->pclass == CLASS_NINJA) &&
		    !(r_ptr->flags3 & (RF3_UNDEAD | RF3_HURT_LITE)) &&
		    !(r_ptr->flags7 & RF7_DARK_MASK))
			can_use_lite_area = TRUE;

		if (!(r_ptr->flags2 & RF2_STUPID))
		{
			if (d_info[dungeon_type].flags1 & DF1_DARKNESS) f6 &= ~(RF6_DARKNESS);
			else if ((p_ptr->pclass == CLASS_NINJA) && !can_use_lite_area) f6 &= ~(RF6_DARKNESS);
		}
	}

	if (in_no_magic_dungeon && !(r_ptr->flags2 & RF2_STUPID))
	{
		f4 &= (RF4_NOMAGIC_MASK);
		f5 &= (RF5_NOMAGIC_MASK);
		f6 &= (RF6_NOMAGIC_MASK);
	}

	if (r_ptr->flags2 & RF2_SMART)
	{
		/* Hack -- allow "desperate" spells */
		if ((m_ptr->hp < m_ptr->maxhp / 10) &&
			(randint0(100) < 50))
		{
			/* Require intelligent spells */
			f4 &= (RF4_INT_MASK);
			f5 &= (RF5_INT_MASK);
			f6 &= (RF6_INT_MASK);
		}

		/* Hack -- decline "teleport level" in some case */
		if ((f6 & RF6_TELE_LEVEL) && TELE_LEVEL_IS_INEFF(0))
		{
			f6 &= ~(RF6_TELE_LEVEL);
		}
	}

	/* No spells left */
	if (!f4 && !f5 && !f6) return (FALSE);

	/* Remove the "ineffective" spells */
	remove_bad_spells(m_idx, &f4, &f5, &f6);

	if (p_ptr->inside_arena || p_ptr->inside_battle)
	{
		f4 &= ~(RF4_SUMMON_MASK);
		f5 &= ~(RF5_SUMMON_MASK);
		f6 &= ~(RF6_SUMMON_MASK | RF6_TELE_LEVEL);

		if (m_ptr->r_idx == MON_ROLENTO) f6 &= ~(RF6_SPECIAL);
	}

	/* No spells left */
	if (!f4 && !f5 && !f6) return (FALSE);

	if (!(r_ptr->flags2 & RF2_STUPID))
	{
		if (!p_ptr->csp) f5 &= ~(RF5_DRAIN_MANA);

		/* Check for a clean bolt shot */
		if (((f4 & RF4_BOLT_MASK) ||
		     (f5 & RF5_BOLT_MASK) ||
		     (f6 & RF6_BOLT_MASK)) &&
		    !clean_shot(m_ptr->fy, m_ptr->fx, py, px, FALSE))
		{
			/* Remove spells that will only hurt friends */
			f4 &= ~(RF4_BOLT_MASK);
			f5 &= ~(RF5_BOLT_MASK);
			f6 &= ~(RF6_BOLT_MASK);
		}

		/* Check for a possible summon */
		if (((f4 & RF4_SUMMON_MASK) ||
		     (f5 & RF5_SUMMON_MASK) ||
		     (f6 & RF6_SUMMON_MASK)) &&
		    !(summon_possible(y, x)))
		{
			/* Remove summoning spells */
			f4 &= ~(RF4_SUMMON_MASK);
			f5 &= ~(RF5_SUMMON_MASK);
			f6 &= ~(RF6_SUMMON_MASK);
		}

		/* Check for a possible raise dead */
		if ((f6 & RF6_RAISE_DEAD) && !raise_possible(m_ptr))
		{
			/* Remove raise dead spell */
			f6 &= ~(RF6_RAISE_DEAD);
		}

		/* Special moves restriction */
		if (f6 & RF6_SPECIAL)
		{
			if ((m_ptr->r_idx == MON_ROLENTO) && !summon_possible(y, x))
			{
				f6 &= ~(RF6_SPECIAL);
			}
		}

		/* No spells left */
		if (!f4 && !f5 && !f6) return (FALSE);
	}

	/* Extract the "inate" spells */
	for (k = 0; k < 32; k++)
	{
		if (f4 & (1L << k)) spell[num++] = k + 32 * 3;
	}

	/* Extract the "normal" spells */
	for (k = 0; k < 32; k++)
	{
		if (f5 & (1L << k)) spell[num++] = k + 32 * 4;
	}

	/* Extract the "bizarre" spells */
	for (k = 0; k < 32; k++)
	{
		if (f6 & (1L << k)) spell[num++] = k + 32 * 5;
	}

	/* No spells left */
	if (!num) return (FALSE);

	/* Stop if player is dead or gone */
	if (!p_ptr->playing || p_ptr->is_dead) return (FALSE);

	/* Stop if player is leaving */
	if (p_ptr->leaving) return (FALSE);

	/* Get the monster name (or "it") */
	monster_desc(m_name, m_ptr, 0x00);

#ifndef JP
	/* Get the monster possessive ("his"/"her"/"its") */
	monster_desc(m_poss, m_ptr, MD_PRON_VISIBLE | MD_POSSESSIVE);
#endif

	switch (do_spell)
	{
	case DO_SPELL_NONE:
		{
			int attempt = 10;
			while (attempt--)
			{
				thrown_spell = choose_attack_spell(m_idx, spell, num);
				if (thrown_spell) break;
			}
		}
		break;

	case DO_SPELL_BR_LITE:
		thrown_spell = 96+14; /* RF4_BR_LITE */
		break;

	case DO_SPELL_BR_DISI:
		thrown_spell = 96+31; /* RF4_BR_DISI */
		break;

	case DO_SPELL_BA_LITE:
		thrown_spell = 128+20; /* RF5_BA_LITE */
		break;

	default:
		return FALSE; /* Paranoia */
	}

	/* Abort if no spell was chosen */
	if (!thrown_spell) return (FALSE);

	/* Calculate spell failure rate */
	failrate = 25 - (rlev + 3) / 4;

	/* Hack -- Stupid monsters will never fail (for jellies and such) */
	if (r_ptr->flags2 & RF2_STUPID) failrate = 0;

	/* Check for spell failure (inate attacks never fail) */
	if (!spell_is_inate(thrown_spell)
	    && (in_no_magic_dungeon || (MON_STUNNED(m_ptr) && one_in_(2)) || (randint0(100) < failrate)))
	{
		disturb(1, 1);
		/* Message */
		msg_format(_("%^sϼʸ򾧤褦ȤԤ", "%^s tries to cast a spell, but fails."), m_name);

		return (TRUE);
	}

	/* Hex: Anti Magic Barrier */
	if (!spell_is_inate(thrown_spell) && magic_barrier(m_idx))
	{
		msg_format(_("ȿˡХꥢ%^sμʸ򤫤ä", "Anti magic barrier cancels the spell which %^s casts."), m_name);
		return (TRUE);
	}

	/* Projectable? */
	direct = player_bold(y, x);

	can_remember = is_original_ap_and_seen(m_ptr);

    if (!direct)
    {
        switch (thrown_spell)
        {
            case 96 + 2:    /* RF4_DISPEL */
            case 96 + 4:    /* RF4_SHOOT */
            case 128 + 9:   /* RF5_DRAIN_MANA */
            case 128 + 10:  /* RF5_MIND_BLAST */
            case 128 + 11:  /* RF5_BRAIN_SMASH */
            case 128 + 12:  /* RF5_CAUSE_1 */
            case 128 + 13:  /* RF5_CAUSE_2 */
            case 128 + 14:  /* RF5_CAUSE_3 */
            case 128 + 15:  /* RF5_CAUSE_4 */
            case 128 + 16:  /* RF5_BO_ACID */
            case 128 + 17:  /* RF5_BO_ELEC */
            case 128 + 18:  /* RF5_BO_FIRE */
            case 128 + 19:  /* RF5_BO_COLD */
            case 128 + 21:  /* RF5_BO_NETH */
            case 128 + 22:  /* RF5_BO_WATE */
            case 128 + 23:  /* RF5_BO_MANA */
            case 128 + 24:  /* RF5_BO_PLAS */
            case 128 + 25:  /* RF5_BO_ICEE */
            case 128 + 26:  /* RF5_MISSILE */
            case 128 + 27:  /* RF5_SCARE */
            case 128 + 28:  /* RF5_BLIND */
            case 128 + 29:  /* RF5_CONF */
            case 128 + 30:  /* RF5_SLOW */
            case 128 + 31:  /* RF5_HOLD */
            case 160 + 1:   /* RF6_HAND_DOOM */
            case 160 + 8:   /* RF6_TELE_TO */
            case 160 + 9:   /* RF6_TELE_AWAY */
            case 160 + 10:  /* RF6_TELE_LEVEL */
            case 160 + 11:  /* RF6_PSY_SPEAR */
            case 160 + 12:  /* RF6_DARKNESS */
            case 160 + 14:  /* RF6_FORGET */
                return (FALSE);
        }
    }

	/* Cast the spell. */
	switch (thrown_spell)
	{
        case 96 + 0:   MP_spell_RF4_SHRIEK(m_idx); break;  /* RF4_SHRIEK */
        case 96 + 1:   break;   /* RF4_XXX1 */
        case 96 + 2:   MP_spell_RF4_DISPEL(m_idx); break;  /* RF4_DISPEL */
        case 96 + 3:   dam = MP_spell_RF4_ROCKET(y, x, m_idx); break;   /* RF4_ROCKET */
        case 96 + 4:   dam = MP_spell_RF4_SHOOT(y, x, m_idx); break;    /* RF4_SHOOT */
        case 96 + 5:   break;   /* RF4_XXX2 */
        case 96 + 6:   break;   /* RF4_XXX3 */
        case 96 + 7:   break;   /* RF4_XXX4 */
        case 96 + 8:   dam = MP_spell_RF4_BREATH(GF_ACID, y, x, m_idx); break;    /* RF4_BR_ACID */
        case 96 + 9:   dam = MP_spell_RF4_BREATH(GF_ELEC, y, x, m_idx); break;    /* RF4_BR_ELEC */
        case 96 + 10:  dam = MP_spell_RF4_BREATH(GF_FIRE, y, x, m_idx); break;    /* RF4_BR_FIRE */
        case 96 + 11:  dam = MP_spell_RF4_BREATH(GF_COLD, y, x, m_idx); break;    /* RF4_BR_COLD */
        case 96 + 12:  dam = MP_spell_RF4_BREATH(GF_POIS, y, x, m_idx); break;    /* RF4_BR_POIS */
        case 96 + 13:  dam = MP_spell_RF4_BREATH(GF_NETHER, y, x, m_idx); break;    /* RF4_BR_NETH */
        case 96 + 14:  dam = MP_spell_RF4_BREATH(GF_LITE, y_br_lite, x_br_lite, m_idx); break;    /* RF4_BR_LITE */
        case 96 + 15:  dam = MP_spell_RF4_BREATH(GF_DARK, y, x, m_idx); break;    /* RF4_BR_DARK */
        case 96 + 16:  dam = MP_spell_RF4_BREATH(GF_CONFUSION, y, x, m_idx); break;    /* RF4_BR_CONF */
        case 96 + 17:  dam = MP_spell_RF4_BREATH(GF_SOUND, y, x, m_idx); break;    /* RF4_BR_SOUN */
        case 96 + 18:  dam = MP_spell_RF4_BREATH(GF_CHAOS, y, x, m_idx); break;    /* RF4_BR_CHAO */
        case 96 + 19:  dam = MP_spell_RF4_BREATH(GF_DISENCHANT, y, x, m_idx); break;    /* RF4_BR_DISE */
        case 96 + 20:  dam = MP_spell_RF4_BREATH(GF_NEXUS, y, x, m_idx); break;    /* RF4_BR_NEXU */
        case 96 + 21:  dam = MP_spell_RF4_BREATH(GF_TIME, y, x, m_idx); break;    /* RF4_BR_TIME */
        case 96 + 22:  dam = MP_spell_RF4_BREATH(GF_INERTIA, y, x, m_idx); break;    /* RF4_BR_INER */
        case 96 + 23:  dam = MP_spell_RF4_BREATH(GF_GRAVITY, y, x, m_idx); break;    /* RF4_BR_GRAV */
        case 96 + 24:  dam = MP_spell_RF4_BREATH(GF_SHARDS, y, x, m_idx); break;    /* RF4_BR_SHAR */
        case 96 + 25:  dam = MP_spell_RF4_BREATH(GF_PLASMA, y, x, m_idx); break;    /* RF4_BR_PLAS */
        case 96 + 26:  dam = MP_spell_RF4_BREATH(GF_FORCE, y, x, m_idx); break;    /* RF4_BR_WALL */
        case 96 + 27:  dam = MP_spell_RF4_BREATH(GF_MANA, y, x, m_idx); break;    /* RF4_BR_MANA */
        case 96 + 28:  dam = spell_RF4_BA_NUKE(y, x, m_idx); break;   /* RF4_BA_NUKE */
        case 96 + 29:  dam = MP_spell_RF4_BREATH(GF_NUKE, y, x, m_idx); break;    /* RF4_BR_NUKE */
        case 96 + 30:  dam = spell_RF4_BA_CHAO(y, x, m_idx); break;  /* RF4_BA_CHAO */
        case 96 + 31:  dam = MP_spell_RF4_BREATH(GF_DISINTEGRATE, y, x, m_idx); break;    /* RF4_BR_DISI */
        case 128 + 0:  dam = spell_RF5_BA_ACID(y, x, m_idx); break;    /* RF5_BA_ACID */
        case 128 + 1:  dam = spell_RF5_BA_ELEC(y, x, m_idx); break;    /* RF5_BA_ELEC */
        case 128 + 2:  dam = spell_RF5_BA_FIRE(y, x, m_idx); break;    /* RF5_BA_FIRE */
        case 128 + 3:  dam = spell_RF5_BA_COLD(y, x, m_idx); break;    /* RF5_BA_COLD */
        case 128 + 4:  dam = spell_RF5_BA_POIS(y, x, m_idx); break;    /* RF5_BA_POIS */
        case 128 + 5:  dam = spell_RF5_BA_NETH(y, x, m_idx); break;    /* RF5_BA_NETH */
        case 128 + 6:  dam = spell_RF5_BA_WATE(y, x, m_idx); break;    /* RF5_BA_WATE */
        case 128 + 7:  dam = spell_RF5_BA_MANA(y, x, m_idx); break;    /* RF5_BA_MANA */
        case 128 + 8:  dam = spell_RF5_BA_DARK(y, x, m_idx); break;    /* RF5_BA_DARK */
        case 128 + 9:  dam = spell_RF5_DRAIN_MANA(y, x, m_idx); break;    /* RF5_DRAIN_MANA */
        case 128 + 10: dam = spell_RF5_MIND_BLAST(y, x, m_idx); break;    /* RF5_MIND_BLAST */
        case 128 + 11: dam = spell_RF5_BRAIN_SMASH(y, x, m_idx); break;    /* RF5_MIND_BLAST */
        case 128 + 12: dam = spell_RF5_CAUSE_1(y, x, m_idx); break;    /* RF5_CAUSE_1 */
        case 128 + 13: dam = spell_RF5_CAUSE_2(y, x, m_idx); break;    /* RF5_CAUSE_2 */
        case 128 + 14: dam = spell_RF5_CAUSE_3(y, x, m_idx); break;    /* RF5_CAUSE_3 */
        case 128 + 15: dam = spell_RF5_CAUSE_4(y, x, m_idx); break;    /* RF5_CAUSE_4 */
        case 128 + 16: dam = spell_RF5_BO_ACID(y, x, m_idx); break;    /* RF5_BO_ACID */
        case 128 + 17: dam = spell_RF5_BO_ELEC(y, x, m_idx); break;    /* RF5_BO_ELEC */
        case 128 + 18: dam = spell_RF5_BO_FIRE(y, x, m_idx); break;    /* RF5_BO_FIRE */
        case 128 + 19: dam = spell_RF5_BO_COLD(y, x, m_idx); break;    /* RF5_BO_COLD */
        case 128 + 20: dam = spell_RF5_BA_LITE(y, x, m_idx); break;    /* RF5_BA_LITE */
        case 128 + 21: dam = spell_RF5_BO_NETH(y, x, m_idx); break;    /* RF5_BO_NETH */
        case 128 + 22: dam = spell_RF5_BO_WATE(y, x, m_idx); break;    /* RF5_BO_WATE */
        case 128 + 23: dam = spell_RF5_BO_MANA(y, x, m_idx); break;    /* RF5_BO_MANA */
        case 128 + 24: dam = spell_RF5_BO_PLAS(y, x, m_idx); break;    /* RF5_BO_PLAS */
        case 128 + 25: dam = spell_RF5_BO_ICEE(y, x, m_idx); break;    /* RF5_BO_ICEE */
        case 128 + 26: dam = spell_RF5_MISSILE(y, x, m_idx); break;    /* RF5_MISSILE */
        case 128 + 27: spell_RF5_SCARE(y, x, m_idx); break;    /* RF5_SCARE */
        case 128 + 28: spell_RF5_BLIND(y, x, m_idx); break;    /* RF5_BLIND */
        case 128 + 29: spell_RF5_CONF(y, x, m_idx); break;    /* RF5_CONF */
        case 128 + 30: spell_RF5_SLOW(y, x, m_idx); break;    /* RF5_SLOW */
        case 128 + 31: spell_RF5_HOLD(y, x, m_idx); break;    /* RF5_HOLD */
        case 160 + 0:  spell_RF6_HASTE(m_idx); break;    /* RF6_HASTE */
        case 160 + 1:  dam = spell_RF6_HAND_DOOM(y, x, m_idx); break;    /* RF6_HAND_DOOM */
        case 160 + 2:  spell_RF6_HEAL(m_idx); break;     /* RF6_HEAL */
        case 160 + 3:  spell_RF6_INVULNER(m_idx); break;     /* RF6_INVULNER */
        case 160 + 4:  spell_RF6_BLINK(m_idx); break;     /* RF6_BLINK */
        case 160 + 5:  spell_RF6_TPORT(m_idx); break;     /* RF6_TPORT */
        case 160 + 6:  dam = spell_RF6_WORLD(m_idx); break;     /* RF6_WORLD */
            
        /* RF6_SPECIAL */
        case 160 + 7:  
            dam = spell_RF6_SPECIAL(y, x, m_idx);
            if (dam < 0) return FALSE; 
            break;

        case 160 + 8:  spell_RF6_TELE_TO(m_idx); break;     /* RF6_TELE_TO */
        case 160 + 9:  spell_RF6_TELE_AWAY(m_idx); break;     /* RF6_TELE_AWAY */
        case 160 + 10: spell_RF6_TELE_LEVEL(m_idx); break;     /* RF6_TELE_LEVEL */
        case 160 + 11: spell_RF6_PSY_SPEAR(y, x, m_idx); break;    /* RF6_PSY_SPEAR */
        case 160 + 12: spell_RF6_DARKNESS(m_idx); break;     /* RF6_DARKNESS */
        case 160 + 13: spell_RF6_TRAPS(y, x, m_idx); break;     /* RF6_TRAPS */
        case 160 + 14: spell_RF6_FORGET(m_idx); break;     /* RF6_FORGET */
        case 160 + 15: spell_RF6_RAISE_DEAD(m_idx); break;    /* RF6_RAISE_DEAD */
        case 160 + 16: spell_RF6_S_KIN(y, x, m_idx); break;  /* RF6_S_KIN */
        case 160 + 17: spell_RF6_S_CYBER(y, x, m_idx); break;    /* RF6_S_CYBER */
        case 160 + 18: spell_RF6_S_MONSTER(y, x, m_idx); break;    /* RF6_S_MONSTER */
        case 160 + 19: spell_RF6_S_MONSTERS(y, x, m_idx); break;    /* RF6_S_MONSTER */
        case 160 + 20: spell_RF6_S_ANT(y, x, m_idx); break;    /* RF6_S_ANT */
        case 160 + 21: spell_RF6_S_SPIDER(y, x, m_idx); break;    /* RF6_S_SPIDER */
        case 160 + 22: spell_RF6_S_HOUND(y, x, m_idx); break;    /* RF6_S_HOUND */
        case 160 + 23: spell_RF6_S_HYDRA(y, x, m_idx); break;    /* RF6_S_HYDRA */
        case 160 + 24: spell_RF6_S_ANGEL(y, x, m_idx); break;   /* RF6_S_ANGEL */
        case 160 + 25: spell_RF6_S_DEMON(y, x, m_idx); break;   /* RF6_S_DEMON */
        case 160 + 26: spell_RF6_S_UNDEAD(y, x, m_idx); break;   /* RF6_S_UNDEAD */
        case 160 + 27: spell_RF6_S_DRAGON(y, x, m_idx); break;   /* RF6_S_DRAGON */
        case 160 + 28: spell_RF6_S_HI_UNDEAD(y, x, m_idx); break;  /* RF6_S_HI_UNDEAD */
        case 160 + 29: spell_RF6_S_HI_DRAGON(y, x, m_idx); break;  /* RF6_S_HI_DRAGON */
        case 160 + 30: spell_RF6_S_AMBERITES(y, x, m_idx); break;  /* RF6_S_AMBERITES */
        case 160 + 31: spell_RF6_S_UNIQUE(y, x, m_idx); break;     /* RF6_S_UNIQUE */
	}

	if ((p_ptr->action == ACTION_LEARN) && thrown_spell > 175)
	{
		learn_spell(thrown_spell - 96);
	}

	if (seen && maneable && !world_monster && (p_ptr->pclass == CLASS_IMITATOR))
	{
		if (thrown_spell != 167) /* Not RF6_SPECIAL */
		{
			if (p_ptr->mane_num == MAX_MANE)
			{
				int i;
				p_ptr->mane_num--;
				for (i = 0;i < p_ptr->mane_num;i++)
				{
					p_ptr->mane_spell[i] = p_ptr->mane_spell[i+1];
					p_ptr->mane_dam[i] = p_ptr->mane_dam[i+1];
				}
			}
			p_ptr->mane_spell[p_ptr->mane_num] = thrown_spell - 96;
			p_ptr->mane_dam[p_ptr->mane_num] = dam;
			p_ptr->mane_num++;
			new_mane = TRUE;

			p_ptr->redraw |= (PR_IMITATION);
		}
	}

	/* Remember what the monster did to us */
	if (can_remember)
	{
		/* Inate spell */
		if (thrown_spell < 32 * 4)
		{
			r_ptr->r_flags4 |= (1L << (thrown_spell - 32 * 3));
			if (r_ptr->r_cast_spell < MAX_UCHAR) r_ptr->r_cast_spell++;
		}

		/* Bolt or Ball */
		else if (thrown_spell < 32 * 5)
		{
			r_ptr->r_flags5 |= (1L << (thrown_spell - 32 * 4));
			if (r_ptr->r_cast_spell < MAX_UCHAR) r_ptr->r_cast_spell++;
		}

		/* Special spell */
		else if (thrown_spell < 32 * 6)
		{
			r_ptr->r_flags6 |= (1L << (thrown_spell - 32 * 5));
			if (r_ptr->r_cast_spell < MAX_UCHAR) r_ptr->r_cast_spell++;
		}
	}


	/* Always take note of monsters that kill you */
	if (p_ptr->is_dead && (r_ptr->r_deaths < MAX_SHORT) && !p_ptr->inside_arena)
	{
		r_ptr->r_deaths++; /* Ignore appearance difference */
	}

	/* A spell was cast */
	return (TRUE);
}
