/* GIMP - The GNU Image Manipulation Program
 * Copyright (C) 1995-1999 Spencer Kimball and Peter Mattis
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "config.h"

#include <gtk/gtk.h>

#include "libgimpwidgets/gimpwidgets.h"

#include "tools-types.h"

#include "core/gimptoolinfo.h"

#include "paint/gimppaintoptions.h"

#include "widgets/gimppropwidgets.h"
#include "widgets/gimpviewablebox.h"
#include "widgets/gimppopupbutton.h"
#include "widgets/gimpwidgets-utils.h"

#include "gimpairbrushtool.h"
#include "gimpclonetool.h"
#include "gimpconvolvetool.h"
#include "gimpdodgeburntool.h"
#include "gimperasertool.h"
#include "gimphealtool.h"
#include "gimpinktool.h"
#include "gimppaintoptions-gui.h"
#include "gimppenciltool.h"
#include "gimpperspectiveclonetool.h"
#include "gimpsmudgetool.h"
#include "gimptooloptions-gui.h"
#include "gimpbrushoptions-gui.h"
#include "gimpdynamicsoptions-gui.h"

#include "gimp-intl.h"

typedef void (*GimpContextNotifyCallback)   (GObject *config, GParamSpec *param_spec, GtkWidget *label);

static GtkWidget * fade_options_gui      (GimpPaintOptions *paint_options,
                                          GType             tool_type,
                                          gboolean          horizontal);
static GtkWidget * gradient_options_gui  (GimpPaintOptions *paint_options,
                                          GType             tool_type,
                                          GtkWidget        *incremental_toggle,
                                          gboolean          horizontal);
static GtkWidget * jitter_options_gui    (GimpPaintOptions *paint_options,
                                          GType             tool_type,
                                          gboolean          horizontal);
static GtkWidget * smoothing_options_gui (GimpPaintOptions *paint_options,
                                          GType             tool_type,
                                          gboolean          horizontal);
static void       fade_options_create_view (GtkWidget *button, 
                                             GtkWidget **result, GObject *config);
static void       jitter_options_create_view (GtkWidget *button, 
                                             GtkWidget **result, GObject *config);
static void       gradient_options_create_view (GtkWidget *button, 
                                             GtkWidget **result, GObject *config);
static void       smoothing_options_create_view (GtkWidget *button, 
                                             GtkWidget **result, GObject *config);

/*  public functions  */

GtkWidget *
gimp_paint_options_gui (GimpToolOptions *tool_options)
{
  return gimp_paint_options_gui_full (tool_options, FALSE);
}

GtkWidget *
gimp_paint_options_gui_horizontal (GimpToolOptions *tool_options)
{
  return gimp_paint_options_gui_full (tool_options, TRUE);
}

GtkWidget *
gimp_paint_options_gui_full (GimpToolOptions *tool_options, gboolean horizontal)
{
  GObject          *config  = G_OBJECT (tool_options);
  GimpPaintOptions *options = GIMP_PAINT_OPTIONS (tool_options);
  GtkWidget        *vbox    = gimp_tool_options_gui_full (tool_options, horizontal);
  GtkWidget        *frame;
  GtkWidget        *table;
  GtkWidget        *menu;
  GtkWidget        *label;
  GtkWidget        *button;
  GtkWidget        *incremental_toggle = NULL;
  GType             tool_type;
  GimpToolOptionsTableIncrement inc = gimp_tool_options_table_increment (horizontal);  

  tool_type = tool_options->tool_info->tool_type;

  /*  the main table  */
  table = gimp_tool_options_table (3, horizontal);
  gtk_table_set_col_spacings (GTK_TABLE (table), 2);
  gtk_table_set_row_spacings (GTK_TABLE (table), 2);
  gtk_box_pack_start (GTK_BOX (vbox), table, FALSE, FALSE, 0);
  gtk_widget_show (table);

  g_object_set_data (G_OBJECT (vbox), GIMP_PAINT_OPTIONS_TABLE_KEY, table);

  /*  the paint mode menu  */
  menu  = gimp_prop_paint_mode_menu_new (config, "paint-mode", TRUE, FALSE);
  label = gimp_table_attach_aligned (GTK_TABLE (table), 
                                     gimp_tool_options_table_increment_get_col (&inc),
                                     gimp_tool_options_table_increment_get_row (&inc),
                                     _("Mode:"), 0.0, 0.5,
                                     menu, 2, FALSE);
  gimp_tool_options_table_increment_next (&inc);

  if (tool_type == GIMP_TYPE_ERASER_TOOL     ||
      tool_type == GIMP_TYPE_CONVOLVE_TOOL   ||
      tool_type == GIMP_TYPE_DODGE_BURN_TOOL ||
      tool_type == GIMP_TYPE_SMUDGE_TOOL)
    {
      gtk_widget_set_sensitive (menu, FALSE);
      gtk_widget_set_sensitive (label, FALSE);
    }

  /*  the opacity scale  */
  gimp_tool_options_opacity_entry_new (config, "opacity",
                                       GTK_TABLE (table),
                                       gimp_tool_options_table_increment_get_col (&inc),
                                       gimp_tool_options_table_increment_get_row (&inc),
                                       _("Opacity:"),
                                       horizontal);
  gimp_tool_options_table_increment_next (&inc);

  /*  the brush  */
  if (g_type_is_a (tool_type, GIMP_TYPE_BRUSH_TOOL))
    {
      GtkWidget *button;

      if (horizontal)
        button = gimp_brush_button_with_popup (config);
      else
        button = gimp_prop_brush_box_new (NULL, GIMP_CONTEXT (tool_options), 2,
                                          "brush-view-type", "brush-view-size");
      gimp_table_attach_aligned (GTK_TABLE (table),
                                 gimp_tool_options_table_increment_get_col (&inc),
                                 gimp_tool_options_table_increment_get_row (&inc),
                                 _("Brush:"), 0.0, 0.5,
                                 button, 2, FALSE);
      gimp_tool_options_table_increment_next (&inc);

      if (!horizontal)
        {
          gimp_tool_options_scale_entry_new (config, "brush-scale",
                                             GTK_TABLE (table),
                                             gimp_tool_options_table_increment_get_col (&inc),
                                             gimp_tool_options_table_increment_get_row (&inc),
                                             _("Scale:"),
                                             0.01, 0.1, 2,
                                             FALSE, 0.0, 0.0, TRUE, horizontal);
          gimp_tool_options_table_increment_next (&inc);

          gimp_tool_options_scale_entry_new (config, "brush-aspect-ratio",
                                             GTK_TABLE (table),
                                             gimp_tool_options_table_increment_get_col (&inc),
                                             gimp_tool_options_table_increment_get_row (&inc),
                                             _("Aspect ratio:"),
                                             0.01, 0.1, 2,
                                             FALSE, 0.0, 0.0, FALSE, horizontal);
          gimp_tool_options_table_increment_next (&inc);

          gimp_tool_options_scale_entry_new (config, "brush-angle",
                                             GTK_TABLE (table),
                                             gimp_tool_options_table_increment_get_col (&inc),
                                             gimp_tool_options_table_increment_get_row (&inc),
                                             _("Angle:"),
                                             1.0, 5.0, 2,
                                             FALSE, 0.0, 0.0, FALSE, horizontal);
          gimp_tool_options_table_increment_next (&inc);
        }
    }


  if (g_type_is_a (tool_type, GIMP_TYPE_BRUSH_TOOL))
    {
      frame = fade_options_gui (options, tool_type, horizontal);
      gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
      gtk_widget_show (frame);

      frame = jitter_options_gui (options, tool_type, horizontal);
      gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
      gtk_widget_show (frame);
    }

    frame = smoothing_options_gui (options, tool_type, horizontal);
    gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
    gtk_widget_show (frame);
    

  /*  the "incremental" toggle  */
  if (tool_type == GIMP_TYPE_PENCIL_TOOL     ||
      tool_type == GIMP_TYPE_PAINTBRUSH_TOOL ||
      tool_type == GIMP_TYPE_ERASER_TOOL)
    {
      incremental_toggle =
        gimp_prop_enum_check_button_new (config,
                                         "application-mode",
                                         _("Incremental"),
                                         GIMP_PAINT_CONSTANT,
                                         GIMP_PAINT_INCREMENTAL);
      gtk_box_pack_start (GTK_BOX (vbox), incremental_toggle, FALSE, FALSE, 0);
      gtk_widget_show (incremental_toggle);
    }

  /* the "hard edge" toggle */
  if (tool_type == GIMP_TYPE_ERASER_TOOL            ||
      tool_type == GIMP_TYPE_CLONE_TOOL             ||
      tool_type == GIMP_TYPE_HEAL_TOOL              ||
      tool_type == GIMP_TYPE_PERSPECTIVE_CLONE_TOOL ||
      tool_type == GIMP_TYPE_CONVOLVE_TOOL          ||
      tool_type == GIMP_TYPE_DODGE_BURN_TOOL        ||
      tool_type == GIMP_TYPE_SMUDGE_TOOL)
    {
      button = gimp_prop_check_button_new (config, "hard", _("Hard edge"));
      gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 0);
      gtk_widget_show (button);
    }

  if (g_type_is_a (tool_type, GIMP_TYPE_PAINTBRUSH_TOOL))
    {
      frame = gradient_options_gui (options, tool_type, incremental_toggle, horizontal);
      gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
      gtk_widget_show (frame);
    }
    
  if (tool_type == GIMP_TYPE_SMUDGE_TOOL)
    {
      button = gimp_prop_check_button_new (config, "use-color-blending", _("Color Blending"));
      gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 0);
      gtk_widget_show (button);
    }

  if (horizontal)
    {
      button = gimp_dynamics_button_with_popup (config);
      gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 0);
      gtk_widget_show (button);
    }

  return vbox;
}


/*  private functions  */



static GtkWidget *
fade_options_gui (GimpPaintOptions            *paint_options,
                  GType                        tool_type, 
                  gboolean                     horizontal)
{
  return gimp_tool_options_toggle_gui_with_popup (G_OBJECT (paint_options), tool_type, 
                             "use-fade", _("Fade"), _("Fade out"),
                             horizontal, fade_options_create_view);
}

static void
fade_options_create_view (GtkWidget *button, GtkWidget **result, GObject *config)
{
  GtkWidget *table;
  GtkWidget *spinbutton;
  GtkWidget *menu;
  GtkWidget *combo;
  GtkWidget *checkbox;
  GList     *children;

  table = gtk_table_new (3, 3, FALSE);
  gtk_table_set_col_spacings (GTK_TABLE (table), 2);

  /*  the fade-out sizeentry  */
  spinbutton = gimp_prop_spin_button_new (config, "fade-length",
                                          1.0, 50.0, 0);
  gtk_entry_set_width_chars (GTK_ENTRY (spinbutton), 6);

  gimp_table_attach_aligned (GTK_TABLE (table), 0, 0,
                             _("Length:"), 0.0, 0.5,
                             spinbutton, 1, FALSE);

  /*  the fade-out unitmenu  */
  menu = gimp_prop_unit_combo_box_new (config, "fade-unit");
  gtk_table_attach (GTK_TABLE (table), menu, 2, 3, 0, 1,
                    GTK_SHRINK | GTK_FILL, GTK_SHRINK | GTK_FILL, 0, 0);
  gtk_widget_show (menu);

#if 0
  /* FIXME pixel digits */
  g_object_set_data (G_OBJECT (menu), "set_digits", spinbutton);
  gimp_unit_menu_set_pixel_digits (GIMP_UNIT_MENU (menu), 0);
#endif

    /*  the repeat type  */
  combo = gimp_prop_enum_combo_box_new (config, "fade-repeat", 0, 0);
  gimp_table_attach_aligned (GTK_TABLE (table), 0, 2,
                             _("Repeat:"), 0.0, 0.5,
                             combo, 2, FALSE);

  checkbox = gimp_prop_check_button_new (config,
                                         "fade-reverse",
                                          _("Reverse"));
  gtk_table_attach (GTK_TABLE (table), checkbox, 0, 2, 3, 4,
                    GTK_SHRINK | GTK_FILL, GTK_SHRINK | GTK_FILL, 0, 0);
  gtk_widget_show (checkbox);
  
  children = gtk_container_get_children (GTK_CONTAINER (table));  
  gimp_tool_options_setup_popup_layout (children, FALSE);

  *result = table;
}

static GtkWidget *
jitter_options_gui (GimpPaintOptions           *paint_options,
                    GType                       tool_type,
                    gboolean                    horizontal)
{
  return gimp_tool_options_toggle_gui_with_popup (G_OBJECT (paint_options), tool_type,
                             "use-jitter", _("Jitter"), _("Apply Jitter"),
                             horizontal, jitter_options_create_view);
}

static void
jitter_options_create_view (GtkWidget *button, GtkWidget **result, GObject *config)
{
  GtkWidget *table;
  GList     *children;

  table = gtk_table_new (1, 3, FALSE);
  gtk_table_set_col_spacings (GTK_TABLE (table), 2);

  gimp_prop_scale_entry_new (config, "jitter-amount",
                             GTK_TABLE (table), 0, 0,
                             _("Amount:"),
                             0.01, 0.1, 2,
                             TRUE, 0.0, 5.0);

  children = gtk_container_get_children (GTK_CONTAINER (table));
  gimp_tool_options_setup_popup_layout (children, FALSE);

  *result = table;
}

static GtkWidget *
gradient_options_gui (GimpPaintOptions         *paint_options,
                    GType                       tool_type,
                    GtkWidget                  *incremental_toggle,
                    gboolean                    horizontal)
{
  return gimp_tool_options_toggle_gui_with_popup (G_OBJECT (paint_options), tool_type,
                             "use-gradient", _("Gradient"), _("Use color from gradient"),
                             horizontal, gradient_options_create_view);
}

static void
gradient_options_create_view (GtkWidget *event_target, GtkWidget **result, GObject *config)
{
  GtkWidget *table;
  GtkWidget *button;

  table = gtk_table_new (3, 3, FALSE);
  gtk_table_set_col_spacings (GTK_TABLE (table), 2);
  gtk_table_set_row_spacings (GTK_TABLE (table), 2);

/*
  if (incremental_toggle)
    {
      gtk_widget_set_sensitive (incremental_toggle,
                                ! paint_options->gradient_options->use_gradient);
      g_object_set_data (G_OBJECT (button), "inverse_sensitive",
                         incremental_toggle);
    }
*/

  /*  the gradient view  */
  button = gimp_prop_gradient_box_new (NULL, GIMP_CONTEXT (config), 2,
                                       "gradient-view-type",
                                       "gradient-view-size",
                                       "gradient-reverse");
  gimp_table_attach_aligned (GTK_TABLE (table), 0, 0,
                             _("Gradient:"), 0.0, 0.5,
                             button, 2, TRUE);

  *result = table;
}

static GtkWidget *
smoothing_options_gui (GimpPaintOptions         *paint_options,
                    GType                       tool_type,
                    gboolean                    horizontal)
{
  return gimp_tool_options_toggle_gui_with_popup (G_OBJECT (paint_options), tool_type,
                             "use-smoothing", _("Smoothing"), _("Apply Smoothing"),
                             horizontal, smoothing_options_create_view);
}

static void
smoothing_options_create_view (GtkWidget *button, GtkWidget **result, GObject *config)
{
  GtkWidget *table;
  GtkObject *factor;
  GList     *children;

  table = gtk_table_new (2, 3, FALSE);
  gtk_table_set_col_spacings (GTK_TABLE (table), 2);

  gimp_prop_scale_entry_new (config, "smoothing-history",
                             GTK_TABLE (table), 0, 0,
                             _("Quality:"),
                             1, 10, 1,
                             FALSE, 0, 100);
                             
  factor = gimp_prop_scale_entry_new (config, "smoothing-factor",
                                      GTK_TABLE (table), 0, 1,
                                      _("Factor:"),
                                      1, 10, 1,
                                      FALSE, 0, 100);
  gimp_scale_entry_set_logarithmic (factor, TRUE);
  
  children = gtk_container_get_children (GTK_CONTAINER (table));
  gimp_tool_options_setup_popup_layout (children, FALSE);

  *result = table;
}

static void       
aspect_entry_create_view (GtkWidget *table, 
                         gint column, gint row,
                         GObject *config)
{
  GtkObject *adj;
  adj   = gimp_prop_scale_entry_new (config, "brush-aspect-ratio",
                                     GTK_TABLE (table),
                                     column, row,
                                     _("Aspect ratio:"),
                                     0.01, 0.1, 2,
                                     FALSE, 0.0, 0.0);
}

static void       
scale_entry_create_view (GtkWidget *table,
                         gint column, gint row,
                         GObject *config)
{
  GtkObject *adj;
  adj = gimp_prop_scale_entry_new (config, "brush-scale",
                                   GTK_TABLE (table), 
                                   column, row,
                                   _("Scale:"),
                                   0.01, 0.1, 2,
                                   FALSE, 0.0, 0.0);
  gimp_scale_entry_set_logarithmic (GTK_OBJECT (adj), TRUE);
}


static void       
angle_entry_create_view (GtkWidget *table,
                         gint column, gint row,
                         GObject *config)
{
  GtkObject *adj;
  adj   = gimp_prop_scale_entry_new (config, "brush-angle",
                                     GTK_TABLE (table), column, row,
                                     _("Angle:"),
                                     1.0, 5.0, 2,
                                     FALSE, 0.0, 0.0);
}
