/* GIMP - The GNU Image Manipulation Program
 * Copyright (C) 1995-1999 Spencer Kimball and Peter Mattis
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

extern "C" {
#include "config.h"

#include <glib.h>
#include <glib/gprintf.h>
#include <gtk/gtk.h>

#include "libgimpwidgets/gimpwidgets.h"

#include "tools-types.h"

#include "base/temp-buf.h"

#include "core/gimpmypaintbrush.h"
#include "core/gimptoolinfo.h"
#include "core/mypaintbrush-brushsettings.h"
#include "core/gimpmypaintbrush-private.hpp"
#include "core/gimpcurve.h"

#include "paint/gimpmypaintoptions.h"

#include "widgets/gimppropwidgets.h"
#include "widgets/gimpspinscale.h"
#include "widgets/gimpviewablebox.h"
#include "widgets/gimpwidgets-constructors.h"
#include "widgets/gimppopupbutton.h"
#include "widgets/gimpcurveview.h"

#include "widgets/gimpwidgets-utils.h"

#include "gimpairbrushtool.h"
#include "gimpclonetool.h"
#include "gimpconvolvetool.h"
#include "gimpdodgeburntool.h"
#include "gimperasertool.h"
#include "gimphealtool.h"
#include "gimpinktool.h"
#include "gimpmypaintoptions-gui.h"
#include "gimppenciltool.h"
#include "gimpperspectiveclonetool.h"
#include "gimpsmudgetool.h"
#include "gimptooloptions-gui.h"
#include "gimpmypaintbrushoptions-gui.h"
};

#include "gimptooloptions-gui-cxx.hpp"
#include "base/scopeguard.hpp"

#include "gimp-intl.h"

///////////////////////////////////////////////////////////////////////////////
class MypaintGUIPrivateBase {
protected:
  GimpMypaintOptions* options;

  class MypaintOptionsPropertyGUIPrivate {
    typedef MypaintOptionsPropertyGUIPrivate Class;
    GimpMypaintOptions* options;
    MyPaintBrushSettings* setting;
    
    GtkWidget* widget;
    gchar*   property_name;
    gchar*   internal_name;
    GClosure* notify_closure;
    GClosure* value_changed_closure;

  public:
    MypaintOptionsPropertyGUIPrivate(GimpMypaintOptions* opts,
				     GHashTable* dict,
				     const gchar* name) 
    {
      ScopeGuard<gchar, void(gpointer)> name_replaced(
        mypaint_brush_signal_name_to_internal_name(name), g_free);

      setting = reinterpret_cast<MyPaintBrushSettings*>(g_hash_table_lookup(dict, name_replaced.ptr()));
      internal_name = (gchar*)g_strdup(name);
      property_name = (gchar*)g_strdup_printf("notify::%s", name);
      options = opts;
      g_object_add_weak_pointer(G_OBJECT(options), (void**)&options);
      notify_closure = value_changed_closure = NULL;
    }
      
    ~MypaintOptionsPropertyGUIPrivate() {
      if (property_name)
        g_free(property_name);

      if (internal_name)
	g_free(internal_name);
      
/*
      if (widget && value_changed_closure) {
	  gulong handler_id = g_signal_handler_find(gpointer(widget), G_SIGNAL_MATCH_CLOSURE, 0, 0, value_changed_closure, NULL, NULL);
	  g_signal_handler_disconnect(gpointer(widget), handler_id);
	  value_changed_closure = NULL;
      }

      if (options && notify_closure) {
        gulong handler_id = g_signal_handler_find(gpointer(optios), G_SIGNAL_MATCH_CLOSURE, 0, 0, notify_closure, NULL, NULL);
	g_signal_handler_disconnect(gpointer(options), handler_id);
	value_changed_closure = NULL;
      }*/
    }
      
    void notify(GObject* object) {
      gdouble value;
      g_object_get(G_OBJECT(options), internal_name, &value, NULL);
    }

    void value_changed(GObject* object) {
      gdouble value = gtk_spin_button_get_value(GTK_SPIN_BUTTON(object));
      g_object_set(G_OBJECT(options), property_name, value, NULL);
    }
      
    GtkWidget* create() {
      gdouble range = setting->maximum - setting->minimum;
      widget  = gimp_prop_spin_scale_new (G_OBJECT(options), 
                                          internal_name,
					  _(setting->displayed_name),
                                          range / 1000.0, range / 100.0, 2);
      g_object_add_weak_pointer(G_OBJECT(widget), (void**)&widget);
      gimp_spin_scale_set_scale_limits (GIMP_SPIN_SCALE (widget), 
					setting->minimum, 
					setting->maximum);

/*      value_changed_closure = 
        g_signal_connect_delegator(G_OBJECT(widget),
				   "value-changed",
				   Delegator::delegator(this, &Class::value_changed));
      notify_closure = 
        g_signal_connect_delegator(G_OBJECT(options),
				   property_name,
				   Delegator::delegator(this, &Class::notify));*/
      g_object_set_cxx_object (G_OBJECT(widget), "behavior", this);
//      gtk_widget_set_size_request (widget, 200, -1);
      return widget;
    }
  };

public:
  MypaintGUIPrivateBase(GimpToolOptions* opts);
  virtual ~MypaintGUIPrivateBase() {}
  virtual GtkWidget* create() = 0;
};

MypaintGUIPrivateBase::
MypaintGUIPrivateBase(GimpToolOptions* opts)
{
  options = GIMP_MYPAINT_OPTIONS(opts);
}

///////////////////////////////////////////////////////////////////////////////
class MypaintOptionsGUIPrivate : public MypaintGUIPrivateBase {
  bool is_toolbar;
  
public:
  MypaintOptionsGUIPrivate(GimpToolOptions* options, bool toolbar);
  GtkWidget* create();

  void destroy(GObject* o);
  void reset_size(GObject *o);
};


MypaintOptionsGUIPrivate::
MypaintOptionsGUIPrivate(GimpToolOptions* opts, bool toolbar) :
  MypaintGUIPrivateBase(opts), is_toolbar(toolbar)
{
}

GtkWidget *
MypaintOptionsGUIPrivate::create ()
{
  GObject            *config  = G_OBJECT (options);
  GtkWidget          *vbox    = gimp_tool_options_gui_full (GIMP_TOOL_OPTIONS(options), is_toolbar);
  GtkWidget          *hbox;
  GtkWidget          *frame;
  GtkWidget          *table;
  GtkWidget          *menu;
  GtkWidget          *scale;
  GtkWidget          *label;
  GtkWidget          *button;
  GtkWidget          *incremental_toggle = NULL;
  GType             tool_type;
  GList            *children;
  GimpToolOptionsTableIncrement inc = gimp_tool_options_table_increment (is_toolbar);  

  tool_type = GIMP_TOOL_OPTIONS(options)->tool_info->tool_type;
  ScopeGuard<GHashTable, void(GHashTable*)> brush_settings_dict(mypaint_brush_get_brush_settings_dict(), g_hash_table_unref);

  /*  the main table  */
  table = gimp_tool_options_table (3, is_toolbar);
  gtk_table_set_col_spacings (GTK_TABLE (table), 2);
  gtk_box_pack_start (GTK_BOX (vbox), table, FALSE, FALSE, 0);
  gtk_widget_show (table);

  /*  the opacity scale  */
  scale = gimp_prop_opacity_spin_scale_new (config, "opacity",
                                            _("Opacity"));
//  gtk_widget_set_size_request (scale, 200, -1);
  gtk_box_pack_start (GTK_BOX (vbox), scale, FALSE, FALSE, 0);
  gtk_widget_show (scale);

  /*  the brush  */
    {
      GtkWidget *button;
      MypaintOptionsPropertyGUIPrivate* scale_obj;
      
      if (is_toolbar)
        button = gimp_mypaint_brush_button_with_popup (config);
      else {
        button = gimp_mypaint_brush_button_with_popup (config);
/*        button = gimp_prop_brush_box_new (NULL, GIMP_CONTEXT(options),
                                          _("MypaintBrush"), 2,
                                          "mypaint-brush-view-type", "mypaint-brush-view-size",
                                          "gimp-mypaint-brush-editor");*/
      }
      gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 0);
      gtk_widget_show (button);

      /* brush size */
      if (is_toolbar)
        hbox = vbox;
      else {
        hbox = gtk_box_new (GTK_ORIENTATION_HORIZONTAL, 2);
        gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
        gtk_widget_show (hbox);
      }

      scale_obj = 
        new MypaintOptionsPropertyGUIPrivate(options, brush_settings_dict.ptr(), "radius-logarithmic");
      scale = scale_obj->create();
      gtk_box_pack_start (GTK_BOX (hbox), scale, TRUE, TRUE, 0);
      gtk_widget_show (scale);

      button = gimp_stock_button_new (GIMP_STOCK_RESET, NULL);
      gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
      gtk_image_set_from_stock (GTK_IMAGE (gtk_bin_get_child (GTK_BIN (button))),
                                GIMP_STOCK_RESET, GTK_ICON_SIZE_MENU);
      gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
      gtk_widget_show (button);

      g_signal_connect_delegator (G_OBJECT(button), "clicked",
                                  Delegator::delegator(this, &MypaintOptionsGUIPrivate::reset_size));

      gimp_help_set_help_data (button,
                               _("Reset size to brush's native size"), NULL);

      if (is_toolbar)
        hbox = vbox;
      else {
        hbox = gtk_box_new (GTK_ORIENTATION_HORIZONTAL, 2);
        gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
        gtk_widget_show (hbox);
      }

      scale_obj = 
        new MypaintOptionsPropertyGUIPrivate(options, brush_settings_dict.ptr(), "slow-tracking");
      scale = scale_obj->create();
      gtk_box_pack_start (GTK_BOX (hbox), scale, TRUE, TRUE, 0);
      gtk_widget_show (scale);

      if (is_toolbar)
        hbox = vbox;
      else {
        hbox = gtk_box_new (GTK_ORIENTATION_HORIZONTAL, 2);
        gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
        gtk_widget_show (hbox);
      }
      scale_obj = 
        new MypaintOptionsPropertyGUIPrivate(options, brush_settings_dict.ptr(), "hardness");
      scale = scale_obj->create();
      gtk_box_pack_start (GTK_BOX (hbox), scale, TRUE, TRUE, 0);
      gtk_widget_show (scale);

//      frame = dynamics_options_gui (options, tool_type, is_toolbar);
//      gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
//      gtk_widget_show (frame);
    }

  if (is_toolbar)
    {
      children = gtk_container_get_children (GTK_CONTAINER (vbox));  
      gimp_tool_options_setup_popup_layout (children, FALSE);
    }

  g_object_set_cxx_object(G_OBJECT(vbox), "behavior-options-private", this);
  gtk_widget_show(vbox);
  return vbox;
}

void
MypaintOptionsGUIPrivate::reset_size (GObject* o)
{
/*
 GimpMypaintBrush *brush = gimp_context_get_brush (GIMP_CONTEXT (paint_options));

 if (brush)
   {
     g_object_set (paint_options,
                   "brush-size", (gdouble) MAX (brush->mask->width,
                                                brush->mask->height),
                   NULL);
   }
*/
}

///////////////////////////////////////////////////////////////////////////////

class ComplexBindAction {
protected:
  GObject* receiver;

  template<typename Delegator>
  void bind_to_full(GObject* target, const gchar* signal_name, Delegator* delegator) {
    ScopeGuard<gchar, void(gpointer)> object_id(g_strdup_printf("behavior-%s-action", signal_name), g_free);
    g_signal_connect_delegator (G_OBJECT(target), signal_name, delegator);
    g_object_set_cxx_object(G_OBJECT(target), object_id, this);
  };

public:
  ComplexBindAction(GObject* receiver_) : receiver(receiver_) {
  };
  virtual ~ComplexBindAction() {}
  
};

class SimpleBindAction : public ComplexBindAction {
protected:
  virtual void emit(GObject* event_target) {};

  void bind_to(GObject* target, const gchar* signal_name) {
    bind_to_full(target, signal_name, Delegator::delegator(this, &SimpleBindAction::emit));
  };

public:
  SimpleBindAction(GObject* target_, GObject* receiver_, const gchar* signal_name) : ComplexBindAction(receiver_) {
    bind_to(target_, signal_name);
  };
  ~SimpleBindAction() {}
  
};

class ToggleWidgetAction : public SimpleBindAction {
public:
  ToggleWidgetAction(GObject* target, GObject* receiver) : SimpleBindAction(target, receiver, "clicked") {};
  ~ToggleWidgetAction() {}

  void emit(GObject* event_target) {
    gboolean visible = gtk_widget_get_visible(GTK_WIDGET(receiver));
    gtk_widget_set_visible(GTK_WIDGET(receiver), !visible);
  };
};


class MypaintBrushEditorPrivate : public MypaintGUIPrivateBase {

  class CurveViewActions {
    GObject*                   receiver;
    GimpMypaintOptions*        options;
    int                        setting_index;
    MyPaintBrushInputSettings* input_setting;
    GtkTreeModel*              model;
    GtkAdjustment*             x_min_adj;
    GtkAdjustment*             x_max_adj;
    GtkAdjustment*             y_scale_adj;
    GClosure*                  options_notify_closure;

  public:

    CurveViewActions(GObject* tree_view, 
                     GObject* receiver_, 
                     GimpMypaintOptions* opts, 
                     int setting_index_, 
                     GObject* toggle, 
                     GtkAdjustment* x_min_adj_,
                     GtkAdjustment* x_max_adj_,
                     GtkAdjustment* y_scale_adj_) {
      receiver      = receiver_;
      options       = opts;
      setting_index = setting_index_;
      model         = GTK_TREE_MODEL( gtk_tree_view_get_model( GTK_TREE_VIEW(tree_view) ) );
      x_min_adj     = x_min_adj_;
      x_max_adj     = x_max_adj_;
      y_scale_adj   = y_scale_adj_; 
      GtkTreeSelection* tree_sel = gtk_tree_view_get_selection(GTK_TREE_VIEW (tree_view));
      input_setting = NULL;

      g_object_set_cxx_object(G_OBJECT(tree_view), "behavior", this);

      g_signal_connect_delegator (G_OBJECT(tree_sel), "changed", Delegator::delegator(this, &CurveViewActions::selected));
      g_signal_connect_delegator (G_OBJECT(toggle), "toggled", Delegator::delegator(this, &CurveViewActions::toggled));
      options_notify_closure = NULL;
//      ScopeGuard<gchar, void(gpointer)> signal_name(mypaint_brush_internal_name_to_signal_name(s->name));
      options_notify_closure = g_signal_connect_delegator (G_OBJECT(options), "notify", Delegator::delegator(this, &CurveViewActions::notify_options));
      g_signal_connect_delegator (G_OBJECT(x_min_adj), "value-changed", Delegator::delegator(this, &CurveViewActions::value_changed));
      g_signal_connect_delegator (G_OBJECT(x_max_adj), "value-changed", Delegator::delegator(this, &CurveViewActions::value_changed));
      g_signal_connect_delegator (G_OBJECT(y_scale_adj), "value-changed", Delegator::delegator(this, &CurveViewActions::value_changed));

    };
    ~CurveViewActions() {
      if (options_notify_closure) {
	      gulong handler_id = g_signal_handler_find(gpointer(options), G_SIGNAL_MATCH_CLOSURE, 0, 0, options_notify_closure, NULL, NULL);
	      g_signal_handler_disconnect(gpointer(options), handler_id);
	      options_notify_closure = NULL;
      }
    }

    void selected(GObject* event_target) {
      GtkTreeModel *model;
      GtkTreeIter   iter;

      if (gtk_tree_selection_get_selected (GTK_TREE_SELECTION(event_target), &model, &iter)) {
        gchar* name = NULL;

        gtk_tree_model_get (model, &iter,
                            2, &name,
                            -1);

        activate_input (name);
        if (name)
          g_free(name);
      }
    };
    
    Mapping* get_mapping() {
      GimpMypaintBrush* myb = gimp_mypaint_options_get_current_brush(options);
      GimpMypaintBrushPrivate* priv = reinterpret_cast<GimpMypaintBrushPrivate*>(myb->p);
      Mapping* mapping = priv->get_setting(setting_index)->mapping;
      return mapping;
    }
    
    void activate_input(gchar* name) {
      ScopeGuard<GHashTable, void(GHashTable*)> input_settings_dict(mypaint_brush_get_input_settings_dict(), g_hash_table_unref);
      MyPaintBrushInputSettings* s = (MyPaintBrushInputSettings*)g_hash_table_lookup(input_settings_dict, name);
      input_setting = s;
      
      g_return_if_fail (s != NULL);
      
      GimpCurveView* curve_view = GIMP_CURVE_VIEW(receiver);

      GimpCurve* old_curve = GIMP_CURVE(gimp_curve_view_get_curve(curve_view));

      gimp_curve_view_set_curve (curve_view, NULL, NULL);
      gimp_curve_view_remove_all_backgrounds (curve_view);
      
      GimpCurve* curve = GIMP_CURVE(gimp_curve_new(s->displayed_name));
      gimp_curve_set_curve_type(curve, GIMP_CURVE_SMOOTH);
      g_object_set(G_OBJECT(curve), "n-points", 8, NULL); // "8" is the maximum number of control points of mypaint dynamics...
      Mapping* mapping = get_mapping();
      bool is_used = mapping && mapping->get_n(s->index) > 0;
      
      if (is_used) {
        if (s->hard_minimum == s->hard_maximum) {
          gtk_adjustment_set_lower(x_min_adj, -20);
          gtk_adjustment_set_upper(x_min_adj,  20 - 0.1);
          gtk_adjustment_set_lower(x_max_adj, -20 + 0.1);
          gtk_adjustment_set_upper(x_max_adj,  20);
        } else {
          gtk_adjustment_set_lower(x_min_adj, s->hard_minimum);
          gtk_adjustment_set_upper(x_min_adj, s->hard_maximum - 0.1);
          gtk_adjustment_set_lower(x_max_adj, s->hard_minimum + 0.1);
          gtk_adjustment_set_upper(x_max_adj, s->hard_maximum);
        }
        
        mapping_to_curve(curve);

        GimpRGB color = {1.0, 0., 0.};
        gimp_curve_view_set_curve (curve_view,
                                   curve, &color);

        update_range_parameters();
        
      } else {
      }
      gimp_curve_view_set_x_axis_label (curve_view,
                                        s->displayed_name);
      g_signal_connect_delegator(G_OBJECT(curve), "notify::points",
                                 Delegator::delegator(this, &CurveViewActions::notify_curve));      
    }
    
    
    void mapping_to_curve(GimpCurve* curve) {
      g_return_if_fail( input_setting != NULL );

      float xmin = input_setting->soft_maximum;
      float ymin = 1.0;
      float xmax = input_setting->soft_minimum;
      float ymax = 0;

      Mapping* mapping = get_mapping();
      if (mapping) {
        int n_points = mapping->get_n(input_setting->index);
        if (n_points > 0) {
          float x, y;

          mapping->get_point(input_setting->index, 0, &x, &ymin);
          ymax = ymin;
          xmin = MIN(xmin, x);
          xmax = MAX(xmax, x);
          for (int i = 1; i < n_points; i ++) {
            mapping->get_point(input_setting->index, i, &x, &y);
            if (x == -1.0 && y == -1.0)
              continue;
            if (xmin > x)
              xmin = x;
            if (xmax < x)
              xmax = x;
            if (ymin > y)
              ymin = y;
            if (ymax < y)
              ymax = y;
          }
          
          float xrange = xmax - xmin;
          float yrange = ymax - ymin;
          int i = 0;

          for (; i < n_points; i ++) {
            mapping->get_point(input_setting->index, i, &x, &y);
            float normalized_x = xrange != 0.0? (x - xmin) / xrange : 0.;
            float normalized_y = yrange != 0.0? (y - ymin) / yrange : 0.;
            gimp_curve_set_point(curve, i, normalized_x, normalized_y);
          }
          g_object_get(G_OBJECT(curve), "n-points", &n_points, NULL);
          for (; i < n_points; i ++) {
            gimp_curve_set_point(curve, i, -1.0, -1.0);
          }

        }
      }
    }
    
    
    void update_range_parameters() {
      g_return_if_fail( input_setting != NULL );
      GimpCurveView* curve_view = GIMP_CURVE_VIEW(receiver);
      g_return_if_fail( curve_view != NULL );

      float xmin = input_setting->soft_maximum;
      float ymin = 0;
      float xmax = input_setting->soft_minimum;
      float ymax;
      Mapping* mapping = get_mapping();
      if (mapping) {
        int n_points = mapping->get_n(input_setting->index);
        if (n_points > 0) {
          float x, y;
          mapping->get_point(input_setting->index, 0, &x, &y);
          ymin = ymax = y;
          xmin = MIN(xmin, x);
          xmax = MAX(xmax, x);
          for (int i = 1; i < n_points; i ++) {
            mapping->get_point(input_setting->index, i, &x, &y);
            if (x == -1.0 && y == -1.0)
              continue;
            if (xmin > x)
              xmin = x;
            if (xmax < x)
              xmax = x;
            if (ymin > y)
              ymin = y;
            if (ymax < y)
              ymax = y;
          }
        }
      }
      g_print("xmin,xmax=%3.2lf,%3.2lf: ymin,ymax=%3.2lf, %3.2lf\n", xmin, xmax, ymin, ymax);

      gdouble xmin_value = xmin;
      gdouble xmax_value = xmax;
      gdouble ymax_value = MAX(ABS(ymax), ABS(ymin));
      
      gtk_adjustment_set_upper(y_scale_adj,  7); // "7" is magic word...
      gtk_adjustment_set_lower(y_scale_adj, -7);

      gtk_adjustment_set_value(x_min_adj, xmin_value);
      gtk_adjustment_set_value(x_max_adj, xmax_value);
      gtk_adjustment_set_value(y_scale_adj, ymax_value);
      
      gimp_curve_view_set_range_x(curve_view,  xmin_value, xmax_value);
      gimp_curve_view_set_range_y(curve_view, -ymax_value, ymax_value);
    }
    
    
    void update_list() {
      ScopeGuard<GHashTable, void(GHashTable*)> input_settings_dict(mypaint_brush_get_input_settings_dict(), g_hash_table_unref);
      GtkTreeIter iter;
      g_return_if_fail (gtk_tree_model_get_iter_first(model, &iter));
      do {
        gchar* name = NULL;

        gtk_tree_model_get (model, &iter, 2, &name, -1);

        MyPaintBrushInputSettings* s = (MyPaintBrushInputSettings*)g_hash_table_lookup(input_settings_dict, name);
        Mapping* mapping = get_mapping();
        
        bool is_used = mapping->get_n(s->index) > 0;
        gtk_list_store_set (GTK_LIST_STORE(model), &iter,
                            0, is_used,
                            -1);
      } while (gtk_tree_model_iter_next(model, &iter));        
    }

    
    void toggled(GObject* event_target, gchar* path) {
      GtkTreeIter                      iter;

      if (gtk_tree_model_get_iter_from_string (model, &iter, path)) {
        gchar*   name;
        gboolean use;

        gtk_tree_model_get (model, &iter,
                            2, &name,
                            0, &use,
                            -1);
        g_print("%s is now %d\n", name, use);

        Mapping* mapping = get_mapping();
        
        if (mapping && input_setting) {
          bool is_used = mapping->get_n(input_setting->index) > 1;
          if (!is_used) {
            mapping->set_n(input_setting->index, 2);
            mapping->set_point(input_setting->index, 0, input_setting->hard_minimum, 0);
            mapping->set_point(input_setting->index, 1, input_setting->hard_maximum, input_setting->normal);
          } else {
            mapping->set_n(input_setting->index, 0);
          }
          gtk_list_store_set (GTK_LIST_STORE(model), &iter,
                              0, !is_used,
                              -1);
          activate_input(name);
        }
      }
    }
    
    
    void notify_curve(GObject* target, GParamSpec *pspec) {
      GimpCurve* curve = GIMP_CURVE(target);
      curve_to_mapping(curve);
    }
    
    void curve_to_mapping(GimpCurve* curve) {
      Mapping* mapping = get_mapping();
      GimpCurveView* curve_view = GIMP_CURVE_VIEW(receiver);
      
      g_return_if_fail(input_setting && mapping && curve);

      gint n_points;
      g_object_get(G_OBJECT(curve), "n-points", &n_points, NULL);
      
      int count = 0;
      struct { gdouble x, y; } points[n_points], tmp;

      for (int i = 0; i < n_points; i ++) {
        gimp_curve_get_point(curve, i, &(points[count].x), &(points[count].y));
        if (points[count].x == -1.0 && points[count].y == -1.0)
          continue;
          
        // insert by bubble sort method.
        for (int j = count; j > 0; j --) {
          if (points[j].x < points[j-1].x) {
            tmp = points[j];
            points[j] = points[j-1];
            points[j-1] = tmp;
          }
        };
        count ++;
      }
      
      gdouble xmin, xmax, ymin, ymax, xrange, yrange;
      gimp_curve_view_get_range_x(curve_view, &xmin, &xmax);
      gimp_curve_view_get_range_y(curve_view, &ymin, &ymax);
      
      xrange = xmax - xmin;
      yrange = ymax - ymin;
      
      if (count > 1 || count == 0) {
        mapping->set_n(input_setting->index, count);
        
        for (int i = 0; i < count; i ++) {
          mapping->set_point(input_setting->index, i, points[i].x * xrange + xmin, points[i].y * yrange + ymin);
        }
      }
    }
    
    
    void notify_options(GObject* config, GParamSpec *pspec) {
      update_list();
      if (input_setting)
        activate_input(input_setting->name);
    }
    
    
    void value_changed(GObject* target) {
      GimpCurveView* curve_view = GIMP_CURVE_VIEW(receiver);
      GimpCurve*     curve      = GIMP_CURVE(gimp_curve_view_get_curve(curve_view));

      gdouble xmin_value = gtk_adjustment_get_value(x_min_adj);
      gdouble xmax_value = gtk_adjustment_get_value(x_max_adj);
      gdouble ymax_value = gtk_adjustment_get_value(y_scale_adj);
      
      gimp_curve_view_set_range_x(curve_view, xmin_value, xmax_value);
      gimp_curve_view_set_range_y(curve_view, -ymax_value, ymax_value);

      curve_to_mapping(curve);
    };

  }; // class CurveViewActions
  
  
public:
  MypaintBrushEditorPrivate(GimpToolOptions* opts) :
    MypaintGUIPrivateBase(opts)
  {
  }
  GtkWidget* create();
  GtkWidget* create_input_editor(int setting_index);
};

GtkWidget*
MypaintBrushEditorPrivate::create() {
  GtkWidget* brushsetting_vbox = gtk_box_new(GTK_ORIENTATION_VERTICAL, 0);
  gtk_widget_show(brushsetting_vbox);

  GList* setting_groups;
  setting_groups = mypaint_brush_get_setting_group_list();

  for (GList* iter = setting_groups; iter; iter = iter->next) {
    MypaintBrushSettingGroup* group =
      (MypaintBrushSettingGroup*)iter->data;

    ScopeGuard<gchar, void(gpointer)> bold_title(
      g_strdup_printf("<b>%s</b>", group->display_name), g_free);    
    GtkWidget* group_expander = gtk_expander_new(bold_title.ptr());
    gtk_expander_set_use_markup(GTK_EXPANDER(group_expander), TRUE);
    gtk_widget_show(group_expander);

    GtkWidget* table = gtk_table_new(3, 2 * g_list_length(group->setting_list), FALSE);
    gtk_widget_show(table);

    if (strcmp(group->name, "basic") == 0)
      gtk_expander_set_expanded (GTK_EXPANDER(group_expander), TRUE);

    int count = 0;
    for (GList* iter2 = group->setting_list; iter2; iter2 = iter2->next, count += 2) {
      MyPaintBrushSettings* s = (MyPaintBrushSettings*)iter2->data;
      ScopeGuard<gchar,void(gpointer)> prop_name(
        mypaint_brush_internal_name_to_signal_name(s->internal_name), g_free);

      GtkWidget* h = gimp_prop_spin_scale_new(G_OBJECT(options), prop_name.ptr(),
                               s->displayed_name,
                               0.1, 1.0, 2);
      gtk_widget_set_tooltip_text(h, s->tooltip);
      gtk_widget_show(h);

      GtkWidget* button = gimp_stock_button_new (GIMP_STOCK_RESET, NULL);
      gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
      gtk_image_set_from_stock (GTK_IMAGE (gtk_bin_get_child (GTK_BIN (button))),
                                GIMP_STOCK_RESET, GTK_ICON_SIZE_MENU);
      gtk_widget_show (button);
      gtk_widget_set_tooltip_text(button, _("Reset to default value"));
      
      //g_signal_connect_...(button, "clicked,...);
      GtkWidget* button2;
      if (s->constant) {
        button2 = gtk_label_new("");
        gtk_widget_set_tooltip_text(button2, _("No additional configuration"));
        gtk_misc_set_alignment(GTK_MISC(button2), 0.5, 0.5);

      } else {
        button2 = gtk_button_new_with_label("...");
        gtk_widget_set_tooltip_text(button2, _("Add input value mapping"));
        //g_signal_connect...(button2, "clicked", self.details_clicked_cb, adj, s)
      }

      gtk_widget_show(button2);
      gtk_table_attach(GTK_TABLE(table), h, 0, 1, count, count+1, 
                       GtkAttachOptions(GTK_EXPAND | GTK_FILL), 
                       GtkAttachOptions(GTK_EXPAND | GTK_FILL), 0, 0);

      gtk_table_attach(GTK_TABLE(table), button, 1, 2, count, count+1, 
                       GtkAttachOptions(GTK_FILL), 
                       GtkAttachOptions(GTK_FILL), 0, 0);

      gtk_table_attach(GTK_TABLE(table), button2, 2, 3, count, count+1, 
                       GtkAttachOptions(GTK_FILL), 
                       GtkAttachOptions(GTK_FILL), 0, 0);

      GtkWidget* input_editor = create_input_editor(s->index);

      new ToggleWidgetAction(G_OBJECT(button2), G_OBJECT(input_editor));
      gtk_table_attach(GTK_TABLE(table), input_editor, 0, 3, count + 1, count + 2,
                       GtkAttachOptions(GTK_FILL),
                       GtkAttachOptions(GTK_FILL), 0, 0);
      
    }

    gtk_container_add(GTK_CONTAINER(group_expander), table);
    gtk_box_pack_start(GTK_BOX(brushsetting_vbox) ,group_expander,TRUE,TRUE,0);
  }
  
  g_object_set_cxx_object(G_OBJECT(brushsetting_vbox), "behavior-editor-private", this);
  return brushsetting_vbox;
}


GtkWidget*
MypaintBrushEditorPrivate::create_input_editor(int setting_index) {
  ScopeGuard<GList, void(GList*)> inputs(mypaint_brush_get_input_settings(), g_list_free);
  
  GtkWidget* result_box = gtk_box_new(GTK_ORIENTATION_HORIZONTAL, 2);

  // List view  
  GtkWidget* list_view  = gtk_tree_view_new();
  gtk_box_pack_start(GTK_BOX(result_box), list_view, FALSE, TRUE, 0);
  gtk_widget_show(list_view);
  
  gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(list_view), FALSE);

  {  
    ScopeGuard<GtkListStore, void(gpointer)> 
      list_store(GTK_LIST_STORE(gtk_list_store_new(3, G_TYPE_BOOLEAN, G_TYPE_STRING, G_TYPE_STRING)), g_object_unref);
    gtk_tree_view_set_model (GTK_TREE_VIEW(list_view), GTK_TREE_MODEL((GtkListStore*)list_store));
  }
  
  GtkListStore* store = GTK_LIST_STORE( gtk_tree_view_get_model( GTK_TREE_VIEW(list_view) ) );
  gtk_list_store_clear (store);
  
  for (GList* iter = inputs; iter; iter = iter->next) {
    
    GtkTreeIter tree_iter;
    MyPaintBrushInputSettings* input = reinterpret_cast<MyPaintBrushInputSettings*>(iter->data);
    gtk_list_store_append(store, &tree_iter);
    
    GimpMypaintBrush* myb         = gimp_mypaint_options_get_current_brush(options);
    GimpMypaintBrushPrivate* priv = reinterpret_cast<GimpMypaintBrushPrivate*>(myb->p);
    Mapping* mapping              = priv->get_setting(setting_index)->mapping;
    
    bool is_enabled = (mapping && mapping->get_n(input->index));
    gtk_list_store_set(store, &tree_iter, 0, is_enabled, 1, input->displayed_name, 2, input->name, -1);
  }

  GtkTreeViewColumn   *column;
  GtkCellRenderer *renderer;
  GtkCellRenderer *used;

  renderer = gtk_cell_renderer_toggle_new();
  g_object_set (G_OBJECT(renderer),
                "mode",        GTK_CELL_RENDERER_MODE_ACTIVATABLE,
                "activatable", TRUE,
                NULL);
  column = gtk_tree_view_column_new_with_attributes(_("Input"), renderer,
                                                    "active",  0,
                                                    NULL);
  gtk_tree_view_append_column(GTK_TREE_VIEW(list_view), column);
  used = renderer;

  renderer = gtk_cell_renderer_text_new();
  column = gtk_tree_view_column_new_with_attributes(_("Input"), renderer,
                                                    "text",  1,
                                                    NULL);
  gtk_tree_view_append_column(GTK_TREE_VIEW(list_view), column);

  // GimpCurveView
  GtkWidget* table = gtk_table_new(4, 4, FALSE);
  gtk_widget_show(table);
  gtk_box_pack_start(GTK_BOX(result_box), table, TRUE, TRUE, 0);

  GtkWidget* curve_view = gimp_curve_view_new();
  g_object_set (curve_view, "border-width", 1, NULL);
  g_object_set (curve_view, "y-axis-label", "", NULL);
  gtk_widget_show(curve_view);

  gtk_table_attach(GTK_TABLE(table), curve_view, 0, 3, 0, 3, 
                   GtkAttachOptions(GTK_EXPAND | GTK_FILL), 
                   GtkAttachOptions(GTK_EXPAND | GTK_FILL), 0, 0);
 
  GtkAdjustment* x_min_adj    = GTK_ADJUSTMENT(gtk_adjustment_new(0.0, 0, 0.1, 0.01, 0.1, 0));
  GtkAdjustment* x_max_adj    = GTK_ADJUSTMENT(gtk_adjustment_new(1.0, 0.9, 1.0, 0.01, 0.1, 0));
  GtkAdjustment* y_scale_adj  = GTK_ADJUSTMENT(gtk_adjustment_new(1.0/4.0, -1.0, 1.0, 0.01, 0.1, 0)); 
  GtkWidget*     x_min_edit   = gtk_spin_button_new(x_min_adj, 0.01, 2);
  GtkWidget*     x_max_edit   = gtk_spin_button_new(x_max_adj, 0.01, 2);
  GtkWidget*     y_scale_edit = gtk_spin_button_new(y_scale_adj, 0.01, 2);

  gtk_widget_show(x_min_edit);
  gtk_widget_show(x_max_edit);
  gtk_widget_show(y_scale_edit);

  gtk_table_attach(GTK_TABLE(table), x_min_edit, 0, 1, 3, 4, 
                   GtkAttachOptions(GTK_FILL), 
                   GtkAttachOptions(GTK_FILL), 0, 0);
  gtk_table_attach(GTK_TABLE(table), x_max_edit, 2, 3, 3, 4, 
                   GtkAttachOptions(GTK_FILL), 
                   GtkAttachOptions(GTK_FILL), 0, 0);
  gtk_table_attach(GTK_TABLE(table), y_scale_edit, 3, 4, 0, 1, 
                   GtkAttachOptions(GTK_FILL), 
                   GtkAttachOptions(GTK_FILL), 0, 0);
 
  
  // Event handler construction
  GtkTreeSelection* tree_sel = gtk_tree_view_get_selection(GTK_TREE_VIEW (list_view));
  gtk_tree_selection_set_mode(tree_sel, GTK_SELECTION_BROWSE);

  CurveViewActions* select_action = new CurveViewActions(G_OBJECT(list_view), 
                                                         G_OBJECT(curve_view), 
                                                         options, 
                                                         setting_index, 
                                                         G_OBJECT(used),
                                                         x_min_adj,
                                                         x_max_adj,
                                                         y_scale_adj);

  return result_box;
}

///////////////////////////////////////////////////////////////////////////////
static void
smoothing_options_create_view (GtkWidget *button, GtkWidget **result, GObject *config)
{
  GtkWidget *vbox;
  GtkWidget *scale;
  GList     *children;

  vbox = gtk_box_new (GTK_ORIENTATION_VERTICAL, 2);
  scale = gimp_prop_spin_scale_new (config, "smoothing-quality",
                                    _("Quality"),
                                    1, 20, 0);
  gtk_box_pack_start (GTK_BOX (vbox), scale, TRUE, TRUE, 0);
  gtk_widget_show (scale);

  scale = gimp_prop_spin_scale_new (config, "smoothing-factor",
                                    _("Weight"),
                                    1, 10, 1);
  gtk_box_pack_start (GTK_BOX (vbox), scale, TRUE, TRUE, 0);
  gtk_widget_show (scale);
  
  children = gtk_container_get_children (GTK_CONTAINER (vbox));
  gimp_tool_options_setup_popup_layout (children, FALSE);

  *result = vbox;
}


///////////////////////////////////////////////////////////////////////////////
/*  public functions  */
extern "C" {

GtkWidget *
gimp_mypaint_options_gui (GimpToolOptions *tool_options)
{
  g_print("MYPAINTOPTIONSGUI::create...\n");
  MypaintOptionsGUIPrivate* priv = new MypaintOptionsGUIPrivate(tool_options, false);
  MypaintBrushEditorPrivate* editor_priv = new MypaintBrushEditorPrivate(tool_options);
  GtkWidget* priv_widget = priv->create();
  GtkWidget* editor_widget = editor_priv->create();

  GtkWidget* options_box = gtk_notebook_new();
  gtk_widget_show (options_box);
//  gtk_notebook_set_show_tabs(GTK_NOTEBOOK(options_box), FALSE);

//  gtk_notebook_insert_page(GTK_NOTEBOOK(options_box), brushsetting_vbox, label, 0);
//  gtk_notebook_insert_page(GTK_NOTEBOOK(editor->options_box), brushinputs, label, 1);
//  gtk_notebook_insert_page(GTK_NOTEBOOK(editor->options_box), brushicon, label, 2);
  gtk_notebook_insert_page(GTK_NOTEBOOK(options_box), priv_widget, gtk_label_new(_("Basic")), 0);
  gtk_notebook_insert_page(GTK_NOTEBOOK(options_box), editor_widget, gtk_label_new("Details"), 1);
//  gtk_notebook_set_current_page (GTK_NOTEBOOK(options_box), 1);
#if 0
  GtkWidget*options_box = gtk_box_new(GTK_ORIENTATION_VERTICAL, 2);
  gtk_widget_show (options_box);
  g_print("MYPAINTOPTIONSGUI::add PRIV_WIDGET=%lx\n", (gulong)(gpointer)priv_widget);
  gtk_box_pack_start(GTK_BOX(options_box), priv_widget, TRUE, TRUE, 2);  
  g_print("MYPAINTOPTIONSGUI::add EDITOR_WIDGET=%lx\n", (gulong)(gpointer)editor_widget);
  gtk_box_pack_start(GTK_BOX(options_box), editor_widget, TRUE, TRUE, 2);  
#endif
  return options_box;
}

GtkWidget *
gimp_mypaint_options_gui_horizontal (GimpToolOptions *tool_options)
{
  MypaintOptionsGUIPrivate* priv = new MypaintOptionsGUIPrivate(tool_options, true);
  return priv->create();
}

}; // extern "C"
