/*
 * Galatea Dialog Studio
 * (c)2009 Takuya NISHIMOTO
 *
 * $Id: GrammarTransformer.java,v 1.9 2009/08/27 04:40:47 nishimoto Exp $
 */
package galatea.io.julius;

import galatea.dialog.DialogManager;
import galatea.logger.Logger;
import galatea.util.Property;
import galatea.util.SubProcess;
import galatea.util.Util;

import java.io.InputStream;
import java.io.StringReader;
import java.io.StringWriter;
import java.nio.charset.Charset;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

public class GrammarTransformer {
	private Logger dbg = new Logger(this.getClass());
	private static final String xslRes = "/res/grammar/xml2julgrm.xsl";
	private Transformer transformer_;
	private String grammar_ = "";
	private String voca_ = "";
	private String errors_ = "";
	private String outputs_ = "";
	private SubProcess subProcess_;
	private String mkdfaPerl_ = "perl";
	private String mkdfaScript_ = "./mkdfa.pl";
	private String mkdfaDir_ = ".";
	private boolean running_ = false;
	private String mkdfaExe_;
	private boolean useMkdfaExe_ = false;
	private static int count_ = 0;
	private Charset writingCharset_ = Util.getSystemDefaultCharset();
	private boolean verbose_ = false;
	
	public static void main(String argv[]) {
		String renrakuRes = "/res/grammar/renraku.xml";
		GrammarTransformer gt = new GrammarTransformer();
		String str = DialogManager.getResourceAsString(renrakuRes);
		String base = Util.getPlatformWorkDir() + "/renraku";
		if (gt.doTransformAndCompile(str, base) == false) {
			System.out.print("[error]\n");
			System.out.print("[E]\n" + gt.getErrors());
			System.out.print("[O]\n" + gt.getOutputs());
		} else {
			System.out.print("[ok]\n");
			System.out.print("[grammar]\n" + gt.getGrammar());
			System.out.print("[voca]\n" + gt.getVoca());
			System.out.print("[E]\n" + gt.getErrors());
			System.out.print("[O]\n" + gt.getOutputs());
		}
	}

	public GrammarTransformer() {
		grammar_ = "";
		voca_ = "";
		errors_ = "";
		outputs_ = "";
		
		mkdfaPerl_ = Property.getForPlatformAsStr(
				"Grammar.Mkdfa.Perl", 
				"perl");
		mkdfaDir_ = Property.getForPlatformAsStr(
				"Grammar.Mkdfa.Dir", 
				"C:/work/istc/SRM/bin"
				);
		mkdfaScript_ = Property.getForPlatformAsStr(
				"Grammar.Mkdfa.Script", 
				"mkdfa.pl");
		mkdfaExe_ = Property.getForPlatformAsStr(
				"Grammar.Mkdfa.Exe", 
				"mkdfa.exe");
		useMkdfaExe_  = Property.getForPlatformAsBoolean("Grammar.UseMkdfaExe", false);
		
		if (count_ == 0) {
			Runtime.getRuntime().addShutdownHook(new Thread() {
				public void run() {
					dbg.print("Shutting down");
					if (subProcess_ != null)
						try {
							subProcess_.destroy();
						} catch (Exception e) {
							// ignore
						}
				}
			});
			count_++;
		}
	}
	
	public boolean doTransform(String str, String base) {
		if (_transform(str) == false) {
			errors_ += "_transform(str) error\n";
			return false;
		}
		if (_makeSrcFiles(base) == false) {
			errors_ += "_makeSrcFiles(base) error\n";
			return false;
		}
		return true;
	}
	
	public boolean doCompile(String str, String base) {
		if (_compileGram(base) == false) {
			errors_ += "_makeSrcFiles(base) error\n";
			return false; 
		}
		return true;
	}
	
	public boolean doTransformAndCompile(String str, String base) {
		if (doTransform(str, base) == false) return false;
		if (doCompile(str, base) == false) return false;
		return true;
	}
	
	/**
	 * run mkdfa
	 * @param base
	 * @return false if error
	 */
	private boolean _compileGram(String base) {
		//errors_ = "";
		//outputs_ = "";
		String cmd = "";
		if (useMkdfaExe_) {
			cmd = mkdfaDir_ + "/" + mkdfaExe_ + " " + base;
		} else {
			cmd = mkdfaPerl_ + " " + mkdfaDir_ + "/" + mkdfaScript_ + " " + base;
		}
		String dir = mkdfaDir_;
		base = Util.getPlatformFileName(base);
		cmd = Util.getPlatformFileName(cmd); 
		dir = Util.getPlatformFileName(dir);
		errors_ += "generate SubProcess(dir:" + dir + ")\n" + cmd + "\n";
		subProcess_ = new SubProcess(cmd, dir);
		if (subProcess_ == null) {
			return false;
		}
		if (subProcess_.run() == false) {
			errors_ += subProcess_.getError();
			return false;
		}
		Thread th = new Thread(new Runnable() {
			public void run() {
				boolean working = true;
				while (working) {
					String er = subProcess_.receiveFromStderrNB();
					if (er != null && er.length() > 0) {
						if (verbose_) { System.out.println("[E] " + er); System.out.flush(); }
						if (er.startsWith("cannot open ")) {
							working = false;
						}
						errors_ += er + "\n";
					}
					String out = subProcess_.receiveFromStdoutNB();
					if (out != null && out.length() > 0) { 
						if (verbose_) { System.out.println("[O] " + out); System.out.flush(); }
						outputs_ += out + "\n";
						if (out.indexOf("generated:") >= 0) {
							working = false;
						}
						if (out.indexOf("no .dfa or .dict file generated") >= 0) {
							working = false;
						}
					}
					try {
						Thread.sleep(1);
					} catch (InterruptedException e) {
						e.printStackTrace();
					}
				}
				subProcess_.destroy();
				running_ = false;
			}
		});
		th.setName("GrammarTransformer(_compileGram thread)");
		running_ = true;
		th.start();
		while (running_) {
			try {
				Thread.sleep(100);
			} catch (InterruptedException e) {
				e.printStackTrace();
			}
		}
		if (outputs_.indexOf("generated:") >= 0) {
			return true;
		}
		return false;
	}
	
	/**
	 * write to file : grammar_ voca_ => hoge.grammar hoge.voca
	 * @param base
	 * @return false if error
	 */
	private boolean _makeSrcFiles(String base) {
		//errors_ = "";
		//outputs_ = "";
		try {
			Util.writeToFile(base + ".grammar", getGrammar(), writingCharset_);
			Util.writeToFile(base + ".voca", getVoca(), writingCharset_);
		} catch (Exception e) {
			errors_ += "makeSrcFiles() " + e.toString();
			return false;
		}
		return true;
	}
	
	/**
	 * generate as string : grammar_ voca_ 
	 * @param s : ipa-xml-grammar string
	 * @return false if error
	 */
	private boolean _transform(String s) {
		//errors_ = "";
		//outputs_ = "";
		TransformerFactory factory = TransformerFactory.newInstance();
		InputStream is = DialogManager.class.getResourceAsStream(xslRes);
		StreamSource xsl = new StreamSource(is);
		try {
			transformer_ = factory.newTransformer(xsl);
		} catch (TransformerConfigurationException e) {
			errors_ += "transform() " + e.toString();
			return false;
		}
		StringReader sr = new StringReader(s);
		StreamSource src = new StreamSource(sr);
		StringWriter writer = new StringWriter();
		StreamResult target = new StreamResult(writer);
		try {
			transformer_.transform(src, target);
		} catch (Exception e) {
			errors_ += "transform() " +  e.toString();
			return false;
		}
		String ws = new String(writer.getBuffer());
		String lines[] = ws.replaceAll("\r","").split("\n");
		Pattern pat = Pattern.compile("^[^\\s]+ :.*$");
		grammar_ = "";
		for (int i = 0, n = lines.length; i < n; i++) {
			Matcher mat = pat.matcher(lines[i]);
			if (mat.matches()) {
				grammar_ += lines[i] + "\n";
			}
		}
		voca_ = "";
		Pattern pat2 = Pattern.compile("^\\s*$");
		for (int i = 0, n = lines.length; i < n; i++) {
			Matcher mat = pat.matcher(lines[i]);
			Matcher mat2 = pat2.matcher(lines[i]);
			if (!mat.matches() 
					&& !mat2.matches() 
					&& lines[i].length() > 0 && lines[i].charAt(0) != '#') {
				voca_ += lines[i] + "\n";
			}
		}
		return true;
	}
	
	public String getGrammar() { 
		return grammar_; 
	}
	
	public String getVoca() {
		return voca_; 
	}

	public String getErrors() {
		return errors_; 
	}

	public String getOutputs() {
		return outputs_; 
	}

	public void setMkdfaDir(String mkdfaDir) {
		mkdfaDir_ = mkdfaDir;
	}

	public void setEncoding(String encoding) {
		writingCharset_ = Charset.forName(encoding);
	}

	public void setVerbose(boolean b) {
		verbose_ = true;
	}
	
}
