package jp.ne.motoki.android.fuelchecker;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;

public class FuelData {
    
    private long milliDatetime;
    private int decaQuantity;
    private int decaDistance;
    
    public FuelData() {
        this(Factory.getCurrentTimeInMillis(), 0, 0);
    }
    
    private FuelData(long milliDatetime, int decaQuantity, int decaDistance) {
        if (milliDatetime < 0) {
            throw new IllegalArgumentException("milliDatetime < 0");
        }
        if (decaQuantity < 0) {
            throw new IllegalArgumentException("decaQuantity < 0");
        }
        if (decaDistance < 0) {
            throw new IllegalArgumentException("dedaDistance < 0");
        }
        this.milliDatetime = milliDatetime;
        this.decaQuantity = decaQuantity;
        this.decaDistance = decaDistance;
    }
    
    public Calendar getDatetime() {
        Calendar cal = Calendar.getInstance();
        cal.setTimeInMillis(milliDatetime);
        return cal;
    }
    
    public void setDate(int year, int monthOfYear, int dayOfMonth) {
        Calendar cal = getDatetime();
        cal.set(year, monthOfYear, dayOfMonth);
        milliDatetime = cal.getTimeInMillis();
    }
    
    public void setTime(int hourOfDay, int minute) {
        Calendar cal = getDatetime();
        cal.set(Calendar.HOUR_OF_DAY, hourOfDay);
        cal.set(Calendar.MINUTE, minute);
        milliDatetime = cal.getTimeInMillis();
    }
    
    public float getQuantity() {
        return ((float) decaQuantity) / 10;
    }
    
    public void setQuantity(int integralValue, int decimalValue) {
        decaQuantity =
            Factory.toDecaNumber(new int[] {integralValue, decimalValue});
    }
    
    public float getDistance() {
        return ((float) decaDistance) / 10;
    }
    
    public void setDistance(int integralValue, int decimalValue) {
        decaDistance =
            Factory.toDecaNumber(new int[] {integralValue, decimalValue});
    }
    
    public float getFuelEfficiency() {
        return ((float) decaDistance) / decaQuantity;
    }

    public ContentValues toContentValues() {
        ContentValues values = new ContentValues();
        values.put(FuelDbAdapter.KEY_DATETIME, milliDatetime);
        values.put(FuelDbAdapter.KEY_QUANTITY, decaQuantity);
        values.put(FuelDbAdapter.KEY_DISTANCE, decaDistance);
        return values;
    }
    
    @Override
    public boolean equals(Object o) {
        if (!(o instanceof FuelData)) {
            return false;
        }
        if (o == this) {
            return true;
        }
        FuelData f = (FuelData) o;
        return milliDatetime == f.milliDatetime
                && decaQuantity == f.decaQuantity
                && decaDistance == f.decaDistance;
    }
    
    @Override
    public int hashCode() {
        int hash = 7;
        hash = hash * 11 + ((Long) milliDatetime).hashCode();
        hash = hash * 11 + decaQuantity;
        hash = hash * 11 + decaDistance;
        return hash;
    }
    
    public static class Factory {
        
        private Factory() {
            throw new AssertionError("Factory cannot be instantiated.");
        }
        
        public static FuelData createInstance(
                int[] date, int[] time, int[] quantity, int[] distance) {
            long milliDatetime = toMilliDatetime(date, time);
            int decaQuantity = toDecaNumber(quantity);
            int decaDistance = toDecaNumber(distance);
            return new FuelData(milliDatetime, decaQuantity, decaDistance);
        }
        
        public static List<FuelData> createDataListFromDB(Context context) {
            FuelDbAdapter adapter = new FuelDbAdapter(context);
            adapter.open();
            Cursor cursor = adapter.fetchAllData();
            List<FuelData> dataList = FuelData.Factory.createDataList(cursor);
            adapter.close();
            return dataList;
        }
        
        private static List<FuelData> createDataList(Cursor cursor) {
            List<FuelData> list = new ArrayList<FuelData>();
            
            int indexOfDatetime =
                cursor.getColumnIndex(FuelDbAdapter.KEY_DATETIME);
            int indexOfQuantity =
                cursor.getColumnIndex(FuelDbAdapter.KEY_QUANTITY);
            int indexOfDistance =
                cursor.getColumnIndex(FuelDbAdapter.KEY_DISTANCE);
            
            cursor.moveToPrevious();
            while(cursor.moveToNext()) {
                list.add(new FuelData(
                        cursor.getLong(indexOfDatetime),
                        cursor.getInt(indexOfQuantity),
                        cursor.getInt(indexOfDistance)));
            }
            
            return list;
        }
        
        private static long toMilliDatetime(int[] date, int[] time) {
            Calendar cal = Calendar.getInstance();
            cal.set(date[0], date[1], date[2], time[0], time[1], 0);
            cal.set(Calendar.MILLISECOND, 0);
            return cal.getTime().getTime();
        }
        
        private static int toDecaNumber(int[] numeric) {
            return Integer.parseInt("" + numeric[0] + numeric[1]);
        }
        
        private static long getCurrentTimeInMillis() {
            Calendar cal = Calendar.getInstance();
            cal.set(Calendar.SECOND, 0);
            cal.set(Calendar.MILLISECOND, 0);
            return cal.getTimeInMillis();
        }
    }
}
