package jp.sourceforge.freegantt.swing;

import java.awt.BorderLayout;
import java.awt.Point;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Calendar;

import javax.swing.BoundedRangeModel;
import javax.swing.JScrollPane;
import javax.swing.JViewport;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import jp.sourceforge.freegantt.util.CalendarUtil;

public class TaskLineRootPane extends JScrollPane {
	private static final long serialVersionUID = -8965586189659056103L;

	Application app;
	TaskLineHeaderPane headerPane;
	TaskLineDataPane taskLineDataPane;
	TaskLinePane taskLinePane;
	JViewport dataViewport;
	boolean viewportOffsetInitialized = false;

	public TaskLineRootPane(Application app) {
		super(JScrollPane.VERTICAL_SCROLLBAR_NEVER, JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
		this.app = app;
		
		taskLinePane = new TaskLinePane(app);
		setViewportView(taskLinePane);
		
		// 絶対座標レイアウト
		taskLinePane.setLayout(null);
		
		headerPane = new TaskLineHeaderPane(app, taskLinePane);
		headerPane.setBounds(0, 0, taskLinePane.getWidth(), taskLinePane.getHeaderHeight());
		taskLinePane.add(headerPane, BorderLayout.CENTER);

		taskLineDataPane = new TaskLineDataPane(app, app.getProject(), taskLinePane);
		taskLineDataPane.setBounds(0, 0, 1, 1);
		
		dataViewport = new JViewport();
		dataViewport.setBounds(0, 0, 1, 1);
		dataViewport.setView(taskLineDataPane);
		taskLinePane.add(dataViewport, BorderLayout.CENTER);

		getHorizontalScrollBar().getModel().addChangeListener(new ScrollChangeListener());
	}

	public void initializeViewportOffset() {
		Calendar firstDate = app.getProject().getFirstDate();
		if (firstDate == null) firstDate = CalendarUtil.toDateCalendar(Calendar.getInstance());
		firstDate.add(Calendar.DATE, -1);
		
		setViewportAtCalendar(firstDate);
		viewportOffsetInitialized = true;
	}
	
	public void setViewportAtCalendar(Calendar calendar) {
		int offsetDate = CalendarUtil.subDate(calendar, app.getProject().getChartFromDate());
		int x =offsetDate * app.getProject().getCellSize().width;
		
		// 自動スクロール位置調整のループを防ぐために1日分端からずらす
		int cellWidth = app.getProject().getCellSize().width;
		x = Math.max(cellWidth, x);
		x = Math.min(getViewport().getView().getWidth() - getViewport().getWidth() - cellWidth, x);
		
		getViewport().setViewPosition(new Point(x, 0));
	}
	
	public Calendar getViewingCalendar() {
		int offset = getViewport().getViewPosition().x;
		Calendar calendar = (Calendar)app.getProject().getChartFromDate().clone();
		calendar.add(Calendar.DATE, offset / app.getProject().getCellSize().width);
		return calendar;
	}
	
	public void addPreviousChartRange() {
		Calendar viewing = getViewingCalendar();
		app.getProject().getChartFromDate().add(Calendar.MONTH, -1);
		taskLinePane.updateWidth();
		headerPane.updateWidth();
		taskLineDataPane.updateWidth();
		setViewportAtCalendar(viewing);
		repaint();
	}

	public void addPostChartRange() {
		Calendar viewing = getViewingCalendar();
		app.getProject().getChartToDate().add(Calendar.MONTH, 1);
		taskLinePane.updateWidth();
		headerPane.updateWidth();
		taskLineDataPane.updateWidth();
		setViewportAtCalendar(viewing);
		repaint();
	}


	public TaskLinePane getTaskLinePane() {
		return taskLinePane;
	}
	
	public JViewport getDataViewport() {
		return dataViewport;
	}

	public TaskLineDataPane getTaskLineDataPane() {
		return taskLineDataPane;
	}

	public TaskLineHeaderPane getTaskLineHeaderPane() {
		return headerPane;
	}
	
	@Override
	public void doLayout() {
		super.doLayout();
		dataViewport.setBounds(0, taskLinePane.getHeaderHeight(), taskLinePane.getWidth(), taskLinePane.getHeight() - taskLinePane.getHeaderHeight());
	}
	
	public class ScrollChangeListener implements PropertyChangeListener, ChangeListener {

		@Override
		public void stateChanged(ChangeEvent e) {
			if (!(e.getSource() instanceof BoundedRangeModel)) return;
			BoundedRangeModel model = (BoundedRangeModel)e.getSource();
			
			if (model.getValueIsAdjusting()) return;
			if (!viewportOffsetInitialized) return;

			if (model.getValue() <= 0) {
				System.out.println("autoViewportAdjust <-");
				System.out.println("autoViewportAdjust value: " + model.getValue());
				System.out.println("autoViewportAdjust min: " + model.getMinimum());
				System.out.println("autoViewportAdjust max: " + model.getMaximum());
				addPreviousChartRange();
			} else if (model.getValue() >= model.getMaximum() - model.getExtent()) {
				System.out.println("autoViewportAdjust ->");
				System.out.println("autoViewportAdjust value: " + model.getValue());
				System.out.println("autoViewportAdjust min: " + model.getMinimum());
				System.out.println("autoViewportAdjust max: " + model.getMaximum());
				addPostChartRange();
			}
		}

		@Override
		public void propertyChange(PropertyChangeEvent evt) {
		}
		
	}
}
