package jp.sourceforge.freegantt.swing;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.image.BufferedImage;
import java.awt.print.PageFormat;
import java.awt.print.PrinterException;

import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPanel;

import jp.sourceforge.freegantt.locale.Resource;
import jp.sourceforge.freegantt.print.GanttChartPrintable;

public class PreviewDialog extends JDialog {
	private static final long serialVersionUID = 6695686076348590807L;
	
	Application app;
	PageFormat pageFormat;
	int page = 0;
	int pageCount = 0;
	BufferedImage image;
	JPanel buttonPanel;
	ImagePanel imagePanel;
	JButton prevPageButton;
	JButton postPageButton;
	JLabel pageLabel;
	
	public PreviewDialog(Application app) {
		super(app, true);
		this.app = app;
		setContentPane(new ContentPane());
		setTitle(Resource.get("previewTitle"));
		setSize(640, 480);
		
		pageCount = app.getTaskLineDataPane().getPrintRange().getPrintRanges().size();
		
		updatePageFormat(app.getProject().getPageFormat());
		updatePage();
		renderPage(page);
	}

	public void updatePageFormat(PageFormat pageFormat) {
		this.pageFormat = pageFormat;
		image = new BufferedImage((int)pageFormat.getWidth(), (int)pageFormat.getHeight(), BufferedImage.TYPE_INT_BGR);
		
		Graphics g = null;
		try {
			g = image.getGraphics();
			g.setColor(Color.white);
			g.fillRect(0, 0, image.getWidth(this), image.getHeight(this));
		} finally {
			if (g != null) g.dispose();
		}
		imagePanel.setImage(image);
		repaint();
	}
	
	private void renderPage(int pageIndex) {
		GanttChartPrintable printable = new GanttChartPrintable(app);
		Graphics g = null;
		try {
			g = image.getGraphics();
			g.setColor(Color.white);
			g.fillRect(0, 0, image.getWidth(this), image.getHeight(this));
			printable.print(g, pageFormat, pageIndex);
		} catch (PrinterException e) {
			e.printStackTrace();
		} finally {
			if (g != null) g.dispose();
		}
		imagePanel.repaint();
	}

	private void updatePage() {
		pageLabel.setText("" + page + " / " + pageCount);
		prevPageButton.setEnabled(page > 0);
		postPageButton.setEnabled(page < pageCount - 1);
	}

	class ContentPane extends JPanel {
		private static final long serialVersionUID = -4424554441476030096L;

		public ContentPane() {
			setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
			
			buttonPanel = new JPanel();
			buttonPanel.setLayout(new FlowLayout());
			buttonPanel.setMaximumSize(new Dimension(Integer.MAX_VALUE, 24));
			add(buttonPanel);
			
			JButton pageButton = new JButton(Resource.get("previewPage"));
			pageButton.addActionListener(new PageConfigureAction());
			buttonPanel.add(pageButton);
			
			JButton printButton = new JButton(Resource.get("previewPrint"));
			printButton.addActionListener(new PrintAction());
			buttonPanel.add(printButton);
			
			JPanel panel = new JPanel();
			panel.setSize(12, 1);
			buttonPanel.add(panel);
			
			prevPageButton = new JButton(Resource.get("previewPrevPage"));
			prevPageButton.addActionListener(new PrevPageAction());
			buttonPanel.add(prevPageButton);

			pageLabel = new JLabel();
			buttonPanel.add(pageLabel);
			
			postPageButton = new JButton(Resource.get("previewPostPage"));
			postPageButton.addActionListener(new PostPageAction());
			buttonPanel.add(postPageButton);			
			
			JButton closeButton = new JButton(Resource.get("previewClose"));
			closeButton.addActionListener(new CloseAction());
			buttonPanel.add(closeButton);

			imagePanel = new ImagePanel();
			imagePanel.setBounds(0, 0, 100, 100);
			imagePanel.setPreferredSize(new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE));
			add(imagePanel);
		}

		@Override
		public void doLayout() {
			super.doLayout();
			
			int height = getHeight();
			height -= buttonPanel.getHeight();
			imagePanel.setSize(imagePanel.getWidth(), height);
			imagePanel.setPreferredSize(getSize());
		}
	}
	
	class ImagePanel extends JPanel {
		private static final long serialVersionUID = 7267024727103791877L;
		
		Image image;

		public Image getImage() {
			return image;
		}

		public void setImage(Image image) {
			this.image = image;
		}

		@Override
		protected void paintComponent(Graphics g) {
			super.paintComponent(g);
			
			if (image == null) return;
			double scaleX = (double)getWidth() / (double)image.getWidth(this);
			double scaleY = (double)getHeight() / (double)image.getHeight(this);
			double scale = Math.min(scaleX, scaleY);
			int scaledWidth = (int)((double)image.getWidth(this) * scale);
			int scaledHeight = (int)((double)image.getHeight(this) * scale);
			g.drawImage(image, (getWidth() - scaledWidth)/2, (getHeight() - scaledHeight)/2, scaledWidth, scaledHeight, this);
		}
	}
	
	class PrintAction implements ActionListener {
		@Override
		public void actionPerformed(ActionEvent e) {
			app.print();
		}
	}
	
	class PageConfigureAction implements ActionListener {
		@Override
		public void actionPerformed(ActionEvent e) {
			app.configurePage();
			updatePageFormat(app.getProject().getPageFormat());
			renderPage(page);
		}
	}
	
	class CloseAction implements ActionListener {
		@Override
		public void actionPerformed(ActionEvent e) {
			dispose();
		}
	}
	
	class PrevPageAction implements ActionListener {
		@Override
		public void actionPerformed(ActionEvent e) {
			page = Math.max(0, page - 1);
			updatePage();
			renderPage(page);
		}
	}
	
	class PostPageAction implements ActionListener {
		@Override
		public void actionPerformed(ActionEvent e) {
			page = Math.min(pageCount - 1, page + 1);
			updatePage();
			renderPage(page);
		}
	}
}
