package jp.sourceforge.freegantt.data;

import java.awt.Dimension;
import java.awt.print.PageFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import jp.sourceforge.freegantt.print.GanttChartPrintable;
import jp.sourceforge.freegantt.swing.HolidayTableModel;
import jp.sourceforge.freegantt.swing.MemberTableModel;
import jp.sourceforge.freegantt.swing.TaskListTableModel;
import jp.sourceforge.freegantt.swing.TaskMemberComboBoxModel;
import jp.sourceforge.freegantt.util.CalendarUtil;

public class Project {
	/** ファイル名 */
	String filename;
	/** プロジェクト名 */
	String name = "";
	/** プロジェクト概要 */
	String summary = "";
	/** 担当者 */
	List<Member> members = new ArrayList<Member>();
	/** タスク */
	List<Task> tasks = new ArrayList<Task>();
	/** 固定休日 */
	List<Integer> fixedHolidays = new ArrayList<Integer>();
	/** 個別休日 */
	List<Calendar> additionalHolidays = new ArrayList<Calendar>();
	/** 印刷範囲（セルサイズ) */
	Dimension printCellSize = new Dimension(30, 20);
	/** セル描画サイズ */
	Dimension cellSize = new Dimension(14, 16);
	
	/** ページフォーマット */
	PageFormat pageFormat = GanttChartPrintable.getDefaultPageFormat();
	
	// 以下は自動計算される
	/** タスクレベル */
	int maxLevel;
	/**　描画範囲開始日 */
	Calendar chartFromDate;
	/** 描画範囲終了日 */
	Calendar chartToDate;
	
	private TaskListTableModel taskTableModel;
	private MemberTableModel memberTableModel;
	private HolidayTableModel holidayTableModel;
	private TaskMemberComboBoxModel taskMemberComboBoxModel;
	
	
	public Dimension getCellSize() {
		return cellSize;
	}
	public String getFilename() {
		return filename;
	}
	public void setFilename(String filename) {
		this.filename = filename;
	}
	public Dimension getPrintCellSize() {
		return printCellSize;
	}
	public void setPrintCellSize(Dimension printCellSize) {
		this.printCellSize = printCellSize;
	}
	public String getName() {
		return name;
	}
	public void setName(String name) {
		this.name = name;
	}
	public String getSummary() {
		return summary;
	}
	public void setSummary(String summary) {
		this.summary = summary;
	}
	public List<Member> getMembers() {
		return members;
	}
	public void setMembers(List<Member> members) {
		this.members = members;
	}
	public List<Task> getTasks() {
		return tasks;
	}
	public void setTasks(List<Task> tasks) {
		this.tasks = tasks;
	}
	public List<Integer> getFixedHolidays() {
		return fixedHolidays;
	}
	public void setFixedHolidays(List<Integer> fixedHolidays) {
		this.fixedHolidays = fixedHolidays;
	}
	public List<Calendar> getAdditionalHolidays() {
		return additionalHolidays;
	}
	public void setAdditionalHolidays(List<Calendar> additionalHolidays) {
		this.additionalHolidays = additionalHolidays;
	}
	public int getMaxLevel() {
		return maxLevel;
	}
	public void setMaxLevel(int maxLevel) {
		this.maxLevel = maxLevel;
	}
	public TaskListTableModel getTaskTableModel() {
		return taskTableModel;
	}
	public MemberTableModel getMemberTableModel() {
		return memberTableModel;
	}
	public HolidayTableModel getHolidayTableModel() {
		return holidayTableModel;
	}
	public TaskMemberComboBoxModel getTaskMemberComboBoxModel() {
		return taskMemberComboBoxModel;
	}
	public PageFormat getPageFormat() {
		return pageFormat;
	}
	public void setPageFormat(PageFormat pageFormat) {
		this.pageFormat = pageFormat;
	}
	public Calendar getChartFromDate() {
		return chartFromDate;
	}
	public Calendar getChartToDate() {
		return chartToDate;
	}
	
	public Project() {
		this.taskTableModel = new TaskListTableModel(this);
		this.memberTableModel = new MemberTableModel(this);
		this.holidayTableModel = new HolidayTableModel(this);
		this.taskMemberComboBoxModel = new TaskMemberComboBoxModel(this);
	}
	
	public boolean isHoliday(Calendar calendar) {
		int week = calendar.get(Calendar.DAY_OF_WEEK);
		if (fixedHolidays.contains(week)) return true;
		if (additionalHolidays.contains(calendar)) return true;
		return false;
	}
	
	public boolean isAdditionalHoliday(Calendar calendar) {
		return additionalHolidays.contains(calendar);
	}
	
	public void update() {

		// タスクを階層順にソートする
		List<Task> sortedTasks = new ArrayList<Task>(tasks);
		Collections.sort(sortedTasks, new Comparator<Task>() {
			@Override
			public int compare(Task lhs, Task rhs) {
				if (lhs.getLevel() == rhs.getLevel()) return 0;
				return (lhs.getLevel() > rhs.getLevel()) ? 1 : -1;
			}
		});
		
		
		// 階層の深いものから実質工期を求める
		for (Task task: sortedTasks) {
			updateChildrenPeriod(task);
			updateParentPeriod(task);
		}
		
		// 最深階層を更新する
		maxLevel = 0;
		if (!sortedTasks.isEmpty()) {
			maxLevel = sortedTasks.get(sortedTasks.size()-1).getLevel();
		}
	}
	
	public void updateChildrenPeriod(Task task) {
		if (task.isParent()) return;
		
		if (task.getStartDate() == null) return;
		if (task.getPeriod() == null) task.setPeriod(0);
		
		int leftPeriod = task.getPeriod();
		int realPeriod = 0;
		Calendar now = (Calendar)task.getStartDate().clone();
		
		// 無限ループを防ぐためにリミッタを使用する
		for (int i=0; i<365 && leftPeriod > 0.0; i++) {
			if (!isHoliday(now)) {
				leftPeriod -= 1.0;
			}
			realPeriod += 1.0;
			now.add(Calendar.DATE, 1);
		}
		
		task.setRealPeriod(realPeriod);
	}
	
	/**
	 * 子タスクを集めて返す
	 * @param targetTask
	 */
	private List<Task> getChildTasks(Task targetTask) {
		List<Task> childTasks = new ArrayList<Task>();
		if (targetTask.getLevel() == 0) return childTasks;
		
		int rangeIndex = Integer.MAX_VALUE;
		for (Task task: tasks) {
			if (task.getIndex() > targetTask.getIndex() &&
					task.getLevel() >= targetTask.getLevel()) {
				rangeIndex = Math.min(rangeIndex, task.getIndex());
			}
		}
		
		for (Task task: tasks) {
			if (task.getLevel() < targetTask.getLevel() &&
					task.getIndex() > targetTask.getIndex() &&
					task.getIndex() < rangeIndex) {
				childTasks.add(task);
			}
		}
		
		return childTasks;
	}
	
	/**
	 * 親タスクの開始日と工期は子タスクの全日程を含む長さとする。
	 * 開始日と工期が無い子タスクは無視され、
	 * 全ての子タスクが無視された場合は親タスクに工期が設定されないものとする。
	 * @param task
	 */
	public void updateParentPeriod(Task task) {
		if (!task.isParent()) return;
		task.setStartDate(null);
		task.setPeriod(null);
		task.setRealPeriod(null);
		
		long from = Long.MAX_VALUE;
		long to = Long.MIN_VALUE;
		
		List<Task> children = getChildTasks(task);
		for (Task child: children) {
			if (child.getStartDate() == null) continue;
			long childFrom = child.getStartDate().getTimeInMillis();
			if (childFrom < from) from = childFrom;
			if (childFrom > to) to = childFrom;
			
			if (child.getRealPeriod() == null) continue;
			long childTo = childFrom + child.getRealPeriod().longValue() * 86400000;
			if (childTo < from) from = childTo;
			if (childTo > to) to = childTo;
		}
		
		if (to < from) return;
		
		Calendar startDate = Calendar.getInstance();
		startDate.setTimeInMillis(from);
		task.setStartDate(startDate);
		task.setPeriod((int)((to - from) / 86400000));
		task.setRealPeriod((int)((to - from) / 86400000));
	}
	
	/**
	 * 行番号に対応したタスクを返す
	 * @param index
	 * @return
	 */
	public Task findTask(int index) {
		for (Task task: tasks) {
			if (index == task.getIndex()) {
				return task;
			}
		}
		return null;
	}
	
	/**
	 * 行を削除する
	 * @param index
	 */
	public void removeRow(int index) {
		Task targetTask = findTask(index);
		for (Task task: tasks) {
			if (targetTask != null) task.getRestrictions().remove(targetTask);
			if (task.getIndex() > index) task.setIndex(task.getIndex() - 1);
		}
		if (targetTask != null) tasks.remove(targetTask);
			
		update();
		taskTableModel.fireTableChanged();
	}
	
	/**
	 * 行を挿入する
	 * @param index
	 */
	public void insertRow(int index) {
		for (Task task: tasks) {
			if (task.getIndex() >= index) task.setIndex(task.getIndex() + 1);
		}
		taskTableModel.fireTableChanged();
	}
	
	public Calendar getFirstDate() {
		Calendar firstDate = Calendar.getInstance();
		boolean found = false;
		firstDate.setTimeInMillis(Long.MAX_VALUE);
		for (Task task: tasks) {
			if (task.getStartDate() != null) {
				if (firstDate.getTimeInMillis() > task.getStartDate().getTimeInMillis()) {
					found = true;
					firstDate = (Calendar)(task.getStartDate().clone());
				}
			}
		}
		if (!found) return null;
		return firstDate;
	}
	
	public int getWholePeriod() {
		Calendar firstDate = getFirstDate();
		Calendar lastDate = Calendar.getInstance();
		boolean found = false;
		lastDate.setTimeInMillis(Long.MIN_VALUE);
		if (firstDate == null) return 0;
		for (Task task: tasks) {
			Calendar endDate = task.getEndDate();
			if (endDate != null) {
				if (lastDate.getTimeInMillis() < endDate.getTimeInMillis()) {
					found = true;
					lastDate = endDate;
				}
			}
		}
		if (!found) return 0;
		return CalendarUtil.subDate(lastDate, firstDate);
	}
	
	public int getMaxIndex() {
		int index = 0;
		for (Task task: tasks) {
			if (task.getIndex() > index) index = task.getIndex();
		}
		return index;
	}
	
	/**
	 * リソースを削除し、関連したタスクの担当者をなしにする
	 * @param index
	 */
	public void removeMember(int index) {
		if (index >= members.size()) return;

		Member member = members.get(index);
		for (Task task: tasks) {
			if (task.getMember() == member) {
				task.setMember(null);
			}
		}
		members.remove(index);
		getMemberTableModel().fireTableChanged();
		getTaskTableModel().fireTableChanged();
	}
	
	/**
	 * 個別休日を削除し、関連したタスクの実質期間を再計算する
	 * @param index
	 */
	public void removeAdditionalHoliday(int index) {
		if (index >= additionalHolidays.size()) return;
		
		additionalHolidays.remove(index);
		update();
		getHolidayTableModel().fireTableChanged();
		getTaskTableModel().fireTableChanged();
	}
	
	/**
	 * 現在のタスクデータから描画するべき範囲の初期値を決める
	 * タスクが空であれば本日を基準とする。
	 */
	public void updateChartRange() {
		chartFromDate = getFirstDate();
		if (chartFromDate == null) chartFromDate = CalendarUtil.toDateCalendar(Calendar.getInstance());
		chartToDate = getFirstDate();
		int period = getWholePeriod();
		if (chartToDate != null) chartToDate.add(Calendar.DATE, period);
		if (chartToDate == null) chartToDate = CalendarUtil.toDateCalendar(Calendar.getInstance());

		chartFromDate.add(Calendar.MONTH, -1);
		chartToDate.add(Calendar.MONTH, 2);
	}
}
