﻿//#define TEST_ASYNC

using System;
using System.IO;
using System.Collections.Generic;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading;
using System.Threading.Tasks;
using System.Linq;

namespace FooEditEngine
{
    /// <summary>
    /// 進行状況を表す列挙体
    /// </summary>
    public enum ProgressState
    {
        /// <summary>
        /// 操作が開始したことを表す
        /// </summary>
        Start,
        /// <summary>
        /// 操作が終了したことを表す
        /// </summary>
        Complete,
    }
    /// <summary>
    /// 非同期操作の状態を表す
    /// </summary>
    public enum AsyncState
    {
        /// <summary>
        /// 非同期操作は行われていないことを表す
        /// </summary>
        None,
        /// <summary>
        /// 読み出し中であることを表す
        /// </summary>
        Loading,
        /// <summary>
        /// 書き込み中であることを表す
        /// </summary>
        Saving
    }
    /// <summary>
    /// 進行状況を表すためのイベントデータ
    /// </summary>
    public sealed class ProgressEventArgs : EventArgs
    {
        /// <summary>
        /// 進行状況
        /// </summary>
        public ProgressState state;
        /// <summary>
        /// コンストラクター
        /// </summary>
        /// <param name="state">ProgressStateオブジェクト</param>
        public ProgressEventArgs(ProgressState state)
        {
            this.state = state;
        }
    }

    /// <summary>
    /// 進行状況を通知するためのデリゲート
    /// </summary>
    /// <param name="sender">送信元クラス</param>
    /// <param name="e">イベントデータ</param>
    public delegate void ProgressEventHandler(object sender, ProgressEventArgs e);

    /// <summary>
    /// 更新タイプを表す列挙体
    /// </summary>
    public enum UpdateType
    {
        /// <summary>
        /// ドキュメントが置き換えられたことを表す
        /// </summary>
        Replace,
        /// <summary>
        /// ドキュメント全体が削除されたことを表す
        /// </summary>
        Clear,
    }

    /// <summary>
    /// 更新タイプを通知するためのイベントデータ
    /// </summary>
    public sealed class DocumentUpdateEventArgs : EventArgs
    {
        /// <summary>
        /// 更新タイプ
        /// </summary>
        public UpdateType type;
        /// <summary>
        /// 開始位置
        /// </summary>
        public int startIndex;
        /// <summary>
        /// 削除された長さ
        /// </summary>
        public int removeLength;
        /// <summary>
        /// 追加された長さ
        /// </summary>
        public int insertLength;
        /// <summary>
        /// コンストラクター
        /// </summary>
        /// <param name="type">更新タイプ</param>
        /// <param name="startIndex">開始インデックス</param>
        /// <param name="removeLength">削除された長さ</param>
        /// <param name="insertLength">追加された長さ</param>
        public DocumentUpdateEventArgs(UpdateType type, int startIndex, int removeLength, int insertLength)
        {
            this.type = type;
            this.startIndex = startIndex;
            this.removeLength = removeLength;
            this.insertLength = insertLength;
        }
    }

    /// <summary>
    /// ドキュメントに更新があったことを伝えるためのデリゲート
    /// </summary>
    /// <param name="sender">送信元クラス</param>
    /// <param name="e">イベントデータ</param>
    public delegate void DocumentUpdateEventHandler(object sender, DocumentUpdateEventArgs e);

    /// <summary>
    /// ドキュメントの管理を行う
    /// </summary>
    /// <remarks>この型のすべてのメソッド・プロパティはスレッドセーフです</remarks>
    public sealed class Document : IEnumerable<char>, IRandomEnumrator<char>
    {
        const int ProgressNotifyCount = 100;
        Regex regex;
        Match match;
        StringBuffer buffer = new StringBuffer();
        CancellationTokenSource cancleTokenSource;
        Progress<ProgressEventArgs> _Progress = new Progress<ProgressEventArgs>();

        /// <summary>
        /// コンストラクター
        /// </summary>
        public Document()
        {
            this.buffer.Update += new DocumentUpdateEventHandler(buffer_Update);
            this.LoadProgress += new ProgressEventHandler((s, e) => { });
            this.SaveProgress += new ProgressEventHandler((s, e) => { });
            this.Update += new DocumentUpdateEventHandler((s, e) => { });
            this.ChangeFireUpdateEvent += new EventHandler((s,e)=>{});
            this.Markers = new MarkerCollection(this);
            this.UndoManager = new UndoManager();
        }

        /// <summary>
        /// 進捗処理を表す
        /// </summary>
        public event EventHandler<ProgressEventArgs> Progress
        {
            add
            {
                this._Progress.ProgressChanged += value;
            }
            remove
            {
                this._Progress.ProgressChanged -= value;
            }
        }

        /// <summary>
        /// 読み出し中に呼び出されるイベント
        /// </summary>
        [Obsolete]
        public event ProgressEventHandler LoadProgress;

        /// <summary>
        /// 保存中に呼び出されるイベント
        /// </summary>
        [Obsolete]
        public event ProgressEventHandler SaveProgress;

        /// <summary>
        /// ドキュメントが更新された時に呼ばれるイベント
        /// </summary>
        public event DocumentUpdateEventHandler Update;

        /// <summary>
        /// FireUpdateEventの値が変わったときに呼び出されるイベント
        /// </summary>
        public event EventHandler ChangeFireUpdateEvent;

        /// <summary>
        /// 改行コードの内部表現
        /// </summary>
        public const char NewLine = '\n';

        /// <summary>
        /// EOFの内部表現
        /// </summary>
        public const char EndOfFile = '\u001a';

        /// <summary>
        /// アンドゥ管理クラスを表す
        /// </summary>
        public UndoManager UndoManager
        {
            get;
            private set;
        }

        /// <summary>
        /// 非同期操作の状態を表す
        /// </summary>
        public AsyncState State
        {
            get;
            private set;
        }

        /// <summary>
        /// 文字列の長さ
        /// </summary>
        public int Length
        {
            get
            {
                return this.buffer.Length;
            }
        }

        /// <summary>
        /// 変更のたびにUpdateイベントを発生させるかどうか
        /// </summary>
        /// <remarks>非同期操作中はこの値を変更することはできません</remarks>
        public bool FireUpdateEvent
        {
            get
            {
                return this.buffer.EnableFireUpdateEvent;
            }
            set
            {
                if (this.State == AsyncState.Loading)
                    throw new InvalidOperationException();
                this.buffer.EnableFireUpdateEvent = value; this.ChangeFireUpdateEvent(this, null);
            }
        }

        /// <summary>
        /// インデクサー
        /// </summary>
        /// <param name="i">インデックス（自然数でなければならない）</param>
        /// <returns>Char型</returns>
        public char this[int i]
        {
            get
            {
                return this.buffer[i];
            }
        }

        /// <summary>
        /// 選択範囲コレクション
        /// </summary>
        internal SelectCollection Selections
        {
            get{return this.buffer.Selections;}
        }

        /// <summary>
        /// マーカーコレクション
        /// </summary>
        public MarkerCollection Markers
        {
            get;
            private set;
        }

        /// <summary>
        /// DocumentReaderを作成します
        /// </summary>
        /// <returns></returns>
        public DocumentReader CreateReader()
        {
            return new DocumentReader(this.buffer);
        }

        /// <summary>
        /// マーカーを設定する
        /// </summary>
        /// <param name="m">設定したいマーカー</param>
        public void SetMarker(Marker m)
        {
            if (m.start < 0 || m.start + m.length > this.Length)
                throw new ArgumentOutOfRangeException("startもしくはendが指定できる範囲を超えています");

            this.Markers.Add(m);
        }

        /// <summary>
        /// マーカーを削除する
        /// </summary>
        /// <param name="start">開始インデックス</param>
        /// <param name="length">削除する長さ</param>
        public void RemoveMarker(int start, int length)
        {
            if (start < 0 || start + length > this.Length)
                throw new ArgumentOutOfRangeException("startもしくはendが指定できる範囲を超えています");

            this.Markers.RemoveAll(start, length);
        }

        /// <summary>
        /// マーカーを削除する
        /// </summary>
        /// <param name="type">削除したいマーカーのタイプ</param>
        public void RemoveMarker(HilightType type)
        {
            this.Markers.RemoveAll(type);
        }

        /// <summary>
        /// インデックスに対応するマーカーを得る
        /// </summary>
        /// <param name="index">インデックス</param>
        /// <returns>Marker構造体の列挙子</returns>
        public IEnumerable<Marker> GetMarkers(int index)
        {
            if (index < 0 || index > this.Length)
                throw new ArgumentOutOfRangeException("indexが範囲を超えています");
            return this.Markers.Get(index);
        }

        /// <summary>
        /// 部分文字列を取得する
        /// </summary>
        /// <param name="index">開始インデックス</param>
        /// <param name="length">長さ</param>
        /// <returns>Stringオブジェクト</returns>
        public string ToString(int index, int length)
        {
            return this.buffer.ToString(index, length);
        }

        /// <summary>
        /// インデックスを開始位置とする文字列を返す
        /// </summary>
        /// <param name="index">開始インデックス</param>
        /// <returns>Stringオブジェクト</returns>
        public string ToString(int index)
        {
            return this.ToString(index, this.buffer.Length - index);
        }

        /// <summary>
        /// 行を取得する
        /// </summary>
        /// <param name="startIndex">開始インデックス</param>
        /// <param name="endIndex">終了インデックス</param>
        /// <param name="maxCharCount">最大長</param>
        /// <returns>行イテレーターが返される</returns>
        public IEnumerable<string> GetLines(int startIndex, int endIndex, int maxCharCount = -1)
        {
            return this.buffer.GetLines(startIndex, endIndex, maxCharCount);
        }

        /// <summary>
        /// 文字列を追加する
        /// </summary>
        /// <param name="s">追加したい文字列</param>
        /// <remarks>非同期操作中はこのメソッドを実行することはできません</remarks>
        public void Append(string s)
        {
            this.Replace(this.buffer.Length, 0, s);
        }

        /// <summary>
        /// 文字列を挿入する
        /// </summary>
        /// <param name="index">開始インデックス</param>
        /// <param name="s">追加したい文字列</param>
        /// <remarks>読み出し操作中はこのメソッドを実行することはできません</remarks>
        public void Insert(int index, string s)
        {
            this.Replace(index, 0, s);
        }

        /// <summary>
        /// 文字列を削除する
        /// </summary>
        /// <param name="index">開始インデックス</param>
        /// <param name="length">長さ</param>
        /// <remarks>読み出し操作中はこのメソッドを実行することはできません</remarks>
        public void Remove(int index, int length)
        {
            this.Replace(index, length, "");
        }

        /// <summary>
        /// ドキュメントを置き換える
        /// </summary>
        /// <param name="index">開始インデックス</param>
        /// <param name="length">長さ</param>
        /// <param name="s">文字列</param>
        /// <remarks>読み出し操作中はこのメソッドを実行することはできません</remarks>
        public void Replace(int index, int length, string s)
        {
            if (this.State == AsyncState.Loading)
                throw new InvalidOperationException();
            if (index < 0 || index > this.buffer.Length || index + length > this.buffer.Length || length < 0)
                throw new ArgumentOutOfRangeException();
            if (length == 0 && (s == string.Empty || s == null))
                return;

            this.RemoveMarker(index, length);

            ReplaceCommand cmd = new ReplaceCommand(this.buffer, index, length, s);
            this.UndoManager.push(cmd);
            cmd.redo();
        }

        /// <summary>
        /// 物理行をすべて削除する
        /// </summary>
        /// <remarks>Dirtyフラグも同時にクリアーされます</remarks>
        /// <remarks>非同期操作中はこのメソッドを実行することはできません</remarks>
        public void Clear()
        {
            if (this.State == AsyncState.Loading)
                throw new InvalidOperationException();
            this.buffer.Clear();
        }

        /// <summary>
        /// 非同期操作をキャンセルします
        /// </summary>
        public void Cancle()
        {
            if(this.cancleTokenSource != null)
                this.cancleTokenSource.Cancel();
        }

#if !METRO
        /// <summary>
        /// ファイルからドキュメントを構築します
        /// </summary>
        /// <param name="filepath">読み取り先のファイル</param>
        /// <param name="enc">エンコーディング</param>
        /// <remarks>非同期操作中はこのメソッドを実行することはできません</remarks>
        public void Load(string filepath, Encoding enc)
        {
            if (this.State != AsyncState.None)
                throw new InvalidOperationException();
            using (StreamReader sr = new StreamReader(filepath, enc))
            {
                this.Load(sr);
            }
        }

        /// <summary>
        /// ファイルからドキュメントを非同期的に構築します
        /// </summary>
        /// <param name="filepath">読み取り先のファイル</param>
        /// <param name="enc">エンコーディング</param>
        /// <remarks>読み取り操作は別スレッドで行われます。また、非同期操作中にこのメソッドを読みだすことはできません</remarks>
        public async Task LoadAsync(string filepath, Encoding enc)
        {
            if (this.State != AsyncState.None)
                throw new InvalidOperationException();
            this.cancleTokenSource = new CancellationTokenSource();
            using (StreamReader sr = new StreamReader(filepath, enc))
            {
                await this.LoadAsync(sr, this.cancleTokenSource);
            }
        }
#endif

        /// <summary>
        /// ストリームからドキュメントを構築します
        /// </summary>
        /// <param name="sr">読み取り先のストリーム</param>
        /// <remarks>非同期操作中はこのメソッドを実行することはできません</remarks>
        public void Load(TextReader sr)
        {
            Task t = this.LoadAsync(sr, null);
            t.Wait();
        }


        /// <summary>
        /// ストリームからドキュメントを非同期的に構築します
        /// </summary>
        /// <param name="sr">読み取り先のストリーム</param>
        /// <param name="tokenSource">キャンセルトークン</param>
        /// <remarks>非同期操作中はこのメソッドを実行することはできません</remarks>
        public async Task LoadAsync(TextReader sr,CancellationTokenSource tokenSource = null)
        {
            if (this.State != AsyncState.None)
                throw new InvalidOperationException();
            if (sr.Peek() == -1)
            {
                this.LoadProgress(this, new ProgressEventArgs(ProgressState.Complete));
                return;
            }

            IProgress<ProgressEventArgs> progress = this._Progress;
            try
            {
                this.State = AsyncState.Loading;
                progress.Report(new ProgressEventArgs(ProgressState.Start));
                this.LoadProgress(this, new ProgressEventArgs(ProgressState.Start));
                this.buffer.Clear();
                this.UndoManager.BeginLock();
                string str;
                for (int i = 0; (str = await sr.ReadLineAsync().ConfigureAwait(false)) != null; i++)
                {
                    int index = this.buffer.Length;
                    if (index < 0)
                        index = 0;

                    this.buffer.Replace(index, 0, str + Document.NewLine);

                    if(tokenSource != null)
                        tokenSource.Token.ThrowIfCancellationRequested();
#if TEST_ASYNC
                    System.Threading.Thread.Sleep(10);
#endif
                }
            }
            finally
            {
                this.UndoManager.EndLock();
                this.State = AsyncState.None;
                progress.Report(new ProgressEventArgs(ProgressState.Complete));
                this.LoadProgress(this, new ProgressEventArgs(ProgressState.Complete));
            }
        }

#if !METRO
        /// <summary>
        /// ドキュメントをファイルに保存します
        /// </summary>
        /// <param name="filepath">保存先のファイル</param>
        /// <param name="enc">保存したいエンコード</param>
        /// <param name="newline">改行を表す文字列</param>
        /// <remarks>非同期操作中はこのメソッドを実行することはできません</remarks>
        public void Save(string filepath, Encoding enc, string newline)
        {
            if (this.State != AsyncState.None)
                throw new InvalidOperationException();
            using (StreamWriter sw = new StreamWriter(filepath, false, enc))
            {
                sw.NewLine = newline;
                this.Save(sw);
            }
        }

        /// <summary>
        /// ドキュメントをファイルに保存します
        /// </summary>
        /// <param name="filepath">保存先のファイル</param>
        /// <param name="enc">保存したいエンコード</param>
        /// <param name="newline">改行を表す文字列</param>
        /// <remarks>非同期操作中はこのメソッドを実行することはできません</remarks>
        public async Task SaveAsync(string filepath, Encoding enc, string newline)
        {
            if (this.State != AsyncState.None)
                throw new InvalidOperationException();
            this.cancleTokenSource = new CancellationTokenSource();
            using (StreamWriter sw = new StreamWriter(filepath, false, enc))
            {
                sw.NewLine = newline;
                await this.SaveAsync(sw,this.cancleTokenSource);
            }
        }
#endif

        /// <summary>
        /// ストリームに保存します
        /// </summary>
        /// <param name="sw">保存先のストリーム</param>
        /// <param name="tokenSource">キャンセルトークン</param>
        /// <remarks>非同期操作中はこのメソッドを実行することはできません</remarks>
        public void Save(StreamWriter sw, CancellationTokenSource tokenSource = null)
        {
            Task t = this.SaveAsync(sw, null);
            t.Wait();
        }

        /// <summary>
        /// ストリームに非同期モードで保存します
        /// </summary>
        /// <param name="sw">保存先のストリーム</param>
        /// <param name="tokenSource">キャンセルトークン</param>
        /// <remarks>非同期操作中はこのメソッドを実行することはできません</remarks>
        public async Task SaveAsync(StreamWriter sw, CancellationTokenSource tokenSource = null)
        {
            if (this.State != AsyncState.None)
                throw new InvalidOperationException();
            IProgress<ProgressEventArgs> progress = this._Progress;
            try
            {
                progress.Report(new ProgressEventArgs(ProgressState.Start));
                this.SaveProgress(this, new ProgressEventArgs(ProgressState.Start));
                this.State = AsyncState.Saving;
                StringBuilder line = new StringBuilder();
                for (int i = 0; i < this.Length; i++)
                {
                    char c = this.buffer[i];
                    line.Append(c);
                    if (c == Document.NewLine || i == this.Length - 1)
                    {
                        string str = line.ToString();
                        str = str.Replace(Document.NewLine.ToString(), sw.NewLine);
                        await sw.WriteAsync(str).ConfigureAwait(false);
                        line.Clear();
                        if(tokenSource != null)
                            tokenSource.Token.ThrowIfCancellationRequested();
#if TEST_ASYNC
                    System.Threading.Thread.Sleep(10);
#endif
                    }
                }
            }
            finally
            {
                this.State = AsyncState.None;
                progress.Report(new ProgressEventArgs(ProgressState.Complete));
                this.SaveProgress(this, new ProgressEventArgs(ProgressState.Complete));
            }
        }

        /// <summary>
        /// Find()で使用するパラメーターをセットします
        /// </summary>
        /// <param name="pattern">検索したい文字列</param>
        /// <param name="UseRegex">正規表現を使用するなら真</param>
        /// <param name="opt">RegexOptions列挙体</param>
        public void SetFindParam(string pattern, bool UseRegex, RegexOptions opt)
        {
            this.match = null;
            if (UseRegex)
                this.regex = new Regex(pattern, opt);
            else
                this.regex = new Regex(Regex.Escape(pattern), opt);
        }

        /// <summary>
        /// 指定した文字列を検索します
        /// </summary>
        /// <returns>見つかった場合はSearchResult列挙子を返却します</returns>
        /// <remarks>見つかったパターン以外を置き換えた場合、正常に動作しないことがあります</remarks>
        public IEnumerator<SearchResult> Find()
        {
            return this.Find(0, this.Length);
        }

        /// <summary>
        /// 指定した文字列を検索します
        /// </summary>
        /// <returns>見つかった場合はSearchResult列挙子を返却します</returns>
        /// <param name="start">開始インデックス</param>
        /// <param name="length">検索する長さ</param>
        /// <remarks>見つかったパターン以外を置き換えた場合、正常に動作しないことがあります</remarks>
        public IEnumerator<SearchResult> Find(int start, int length)
        {
            if (this.State == AsyncState.Loading)
                throw new InvalidOperationException();
            if (this.regex == null)
                throw new InvalidOperationException();
            if (start < 0 || start >= this.Length)
                throw new ArgumentOutOfRangeException();

            int end = start + length - 1;

            if(end > this.Length - 1)
                throw new ArgumentOutOfRangeException();

            StringBuilder line = new StringBuilder();
            int oldLength = this.Length;
            for (int i = start; i <= end; i++)
            {
                char c = this[i];
                line.Append(c);
                if (c == Document.NewLine || i == end)
                {
                    this.match = this.regex.Match(line.ToString());
                    while (this.match.Success)
                    {
                        int startIndex = i - line.Length + 1 + this.match.Index;
                        int endIndex = startIndex + this.match.Length - 1;

                        yield return new SearchResult(this.match, startIndex, endIndex);

                        if (this.Length != oldLength)   //長さが変わった場合は置き換え後のパターンの終点＋１まで戻る
                        {
                            int delta = this.Length - oldLength;
                            i = endIndex + delta;
                            end = end + delta;
                            oldLength = this.Length;
                            break;
                        }

                        this.match = this.match.NextMatch();
                    }
                    line.Clear();
                }
            }
        }

        /// <summary>
        /// 任意のパターンですべて置き換えます
        /// </summary>
        /// <param name="replacePattern">置き換え後のパターン</param>
        /// <param name="groupReplace">グループ置き換えを行うなら真。そうでないなら偽</param>
        public void ReplaceAll(string replacePattern,bool groupReplace)
        {
            if (this.regex == null)
                throw new InvalidOperationException();
            ReplaceAllCommand cmd = new ReplaceAllCommand(this.buffer, this.regex, replacePattern, groupReplace);
            this.UndoManager.push(cmd);
            cmd.redo();
        }

        #region IEnumerable<char> メンバー

        /// <summary>
        /// 列挙子を返します
        /// </summary>
        /// <returns>IEnumeratorオブジェクトを返す</returns>
        public IEnumerator<char> GetEnumerator()
        {
            return this.buffer.GetEnumerator();
        }

        #endregion

        #region IEnumerable メンバー

        System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator()
        {
            throw new NotImplementedException();
        }

        #endregion

        void buffer_Update(object sender, DocumentUpdateEventArgs e)
        {
            this.Update(this, e);
        }
    }

    /// <summary>
    /// 検索結果を表す
    /// </summary>
    public class SearchResult
    {
        private Match Match;

        /// <summary>
        /// 一致した場所の開始位置を表す
        /// </summary>
        public int Start;

        /// <summary>
        /// 一致した場所の終了位置を表す
        /// </summary>
        public int End;

        /// <summary>
        /// 見つかった文字列を返す
        /// </summary>
        public string Value
        {
            get { return this.Match.Value; }
        }

        /// <summary>
        /// 指定したパターンを置き換えて返す
        /// </summary>
        /// <param name="replacement">置き換える文字列</param>
        /// <returns>置き換え後の文字列</returns>
        public string Result(string replacement)
        {
            return this.Match.Result(replacement);
        }

        /// <summary>
        /// コンストラクター
        /// </summary>
        /// <param name="m">Matchオブジェクト</param>
        /// <param name="start">開始インデックス</param>
        /// <param name="end">終了インデックス</param>
        public SearchResult(Match m, int start,int end)
        {
            this.Match = m;
            this.Start = start;
            this.End = end;
        }
    }

    /// <summary>
    /// ドキュメントリーダー
    /// </summary>
    public class DocumentReader : TextReader
    {
        StringBuffer document;      
        int currentIndex;

        /// <summary>
        /// コンストラクター
        /// </summary>
        /// <param name="doc"></param>
        internal DocumentReader(StringBuffer doc)
        {
            if (doc == null)
                throw new ArgumentNullException();
            this.document = doc;
        }

        /// <summary>
        /// 文字を取得する
        /// </summary>
        /// <returns>文字。取得できない場合は-1</returns>
        public override int Peek()
        {
            if (this.document == null)
                throw new InvalidOperationException();
            if (this.currentIndex >= this.document.Length)
                return -1;
            return this.document[this.currentIndex];
        }

        /// <summary>
        /// 文字を取得し、イテレーターを一つ進める
        /// </summary>
        /// <returns>文字。取得できない場合は-1</returns>
        public override int Read()
        {
            int c = this.Peek();
            if(c != -1)
                this.currentIndex++;
            return c;
        }

        /// <summary>
        /// 文字列を読み取りバッファーに書き込む
        /// </summary>
        /// <param name="buffer">バッファー</param>
        /// <param name="index">開始インデックス</param>
        /// <param name="count">カウント</param>
        /// <returns>読み取られた文字数</returns>
        public override int Read(char[] buffer, int index, int count)
        {
            if (this.document == null)
                throw new InvalidOperationException();

            if (buffer == null)
                throw new ArgumentNullException();

            if (this.document.Length < count)
                throw new ArgumentException();

            if (index < 0 || count < 0)
                throw new ArgumentOutOfRangeException();

            if (this.document.Length == 0)
                return 0;

            int actualCount = count;
            if (index + count - 1 > this.document.Length - 1)
                actualCount = this.document.Length - index;

            string str = this.document.ToString(index, actualCount);

            for (int i = 0; i < str.Length; i++)    //ToCharArray()だと戻った時に消えてしまう
                buffer[i] = str[i];

            this.currentIndex = index + actualCount;
            
            return actualCount;
        }

        /// <summary>
        /// オブジェクトを破棄する
        /// </summary>
        /// <param name="disposing">真ならアンマネージドリソースを解放する</param>
        protected override void Dispose(bool disposing)
        {
            this.document = null;
        }
    }
}
