﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Drawing;
using DWriteWarpper;
using FooEditEngine.Windows.Properties;

namespace FooEditEngine.Windows
{
    internal static class EffectResourceConverter
    {
        public static int GetID(TokenType t)
        {
            switch (t)
            {
                case TokenType.None:
                    return 0;
                case TokenType.Literal:
                    return 2;
                case TokenType.Keyword1:
                    return 3;
                case TokenType.Keyword2:
                    return 4;
                case TokenType.Control:
                    return 0;
                case TokenType.Comment:
                    return 5;
            }
            return -1;
        }
        public static int GetID(HilightType t)
        {
            switch (t)
            {
                case HilightType.None:
                    return 0;
                case HilightType.Sold:
                    return 0;
                case HilightType.Url:
                    return 6;
                case HilightType.Select:
                    return 7;
            }
            return -1;
        }
    }
    internal class D2DTextRender : ITextRender,IDisposable
    {
        const int LineNumberFiledLength = 6;    //行番号表示時の最大桁数

        int _TabWidthChar;

        DFactory Factory = new DFactory();
        DWindowRender render;
        FooTextBox TextBox;
        ColorTable _ColorTable = new ColorTable();
        ResourceManager<int,DDrawingEffect> effects = new ResourceManager<int,DDrawingEffect>();
        ResourceManager<string, DColorBrush> brushs = new ResourceManager<string,DColorBrush>();
        CacheManager<string, DTextLayout> cache = new CacheManager<string, DTextLayout>();
        DTextFormat Format;
        DBitmap bitmap;
        bool hasCache = false;
        
        public D2DTextRender(FooTextBox textbox)
        {
            InitRenderResource(Factory,textbox);

            textbox.SizeChanged += new EventHandler(textbox_SizeChanged);
            textbox.FontChanged += new EventHandler(textbox_FontChanged);
            textbox.HandleCreated += new EventHandler(textbox_HandleCreated);
            textbox.HandleDestroyed += new EventHandler(textbox_HandleDestroyed);
            textbox.Document.Markers.Updated += new EventHandler(Markers_Updated);
        }

        Color2F ToColor2F(Color color)
        {
            return new Color2F(color.R, color.G, color.B, 255);
        }

        void InitRenderResource(DFactory Factory,FooTextBox textbox)
        {
            if (textbox == null || Factory == null)
                throw new ArgumentNullException();

            this.render = Factory.CreateWindowRender(textbox.Handle,true);
            DColorBrush defalut = this.render.CreateBrush(ToColor2F(this.TokenToColor.Fore));
            DColorBrush control = this.render.CreateBrush(ToColor2F(this.TokenToColor.Control));
            Color2F color = ToColor2F(this.TokenToColor.Hilight);
            color.A = 0.5f;
            DColorBrush hilight = this.render.CreateBrush(color);
            this.brushs.Add("default", defalut);
            this.brushs.Add("control",control);
            this.brushs.Add("hilight", hilight);
            this.render.InitTextRender(defalut, control);
            this.render.AddControlSymbol('\n',UInt32.Parse(Resources.NewLineSymbol));
            this.render.AddControlSymbol('\t', UInt32.Parse(Resources.TabSymbol));
            this.render.AddControlSymbol('　', UInt32.Parse(Resources.FullSpaceSymbol));
            this.Format = Factory.CreateTextFormat(textbox.Font.Name, textbox.Font.Size);
            this.Format.WordWrapping = DWordWrapping.NoWrapping;
            this.TextBox = textbox;
            //EffectResourceConverterの返す値とキーを同じにしないと妙なことになる
            this.effects.Add(2, new DDrawingEffect(this.render, ToColor2F(this.TokenToColor.Literal)));
            this.effects.Add(3, new DDrawingEffect(this.render, ToColor2F(this.TokenToColor.Keyword1)));
            this.effects.Add(4,new DDrawingEffect(this.render, ToColor2F(this.TokenToColor.Keyword2)));
            this.effects.Add(5, new DDrawingEffect(this.render, ToColor2F(this.TokenToColor.Comment)));
            this.effects.Add(6, new DDrawingEffect(this.render, ToColor2F(this.TokenToColor.URL)));
            this.effects.Add(7, new DDrawingEffect(this.render, ToColor2F(this.TokenToColor.Fore), ToColor2F(this.TokenToColor.Hilight)));
            this.TabWidthChar = this.TabWidthChar;
            this.bitmap = render.CreateBitmap(new SizeU((uint)textbox.ClientSize.Width, (uint)textbox.ClientSize.Height));
            this.hasCache = false;
        }

        void DestructRenderResource()
        {
            this.cache.Clear();
            this.brushs.Clear();
            this.effects.Clear();
            this.Format.Dispose();
            this.bitmap.Dispose();
            if(this.render != null)
                this.render.Dispose();
            this.render = null;
        }

        void textbox_HandleDestroyed(object sender, EventArgs e)
        {
            DestructRenderResource();
        }

        void textbox_HandleCreated(object sender, EventArgs e)
        {
            FooTextBox textbox = (FooTextBox)sender;
            InitRenderResource(this.Factory, textbox);
        }

        void textbox_FontChanged(object sender, EventArgs e)
        {
            FooTextBox textbox = (FooTextBox)sender;
            DestructRenderResource();
            InitRenderResource(this.Factory, this.TextBox);
            if (this.ChangedRenderResource != null)
                ChangedRenderResource(this, null);
        }

        void textbox_SizeChanged(object sender, EventArgs e)
        {
            FooTextBox textbox = (FooTextBox)sender;
            if (render != null)
            {
                render.Resize((uint)textbox.ClientSize.Width, (uint)textbox.ClientSize.Height);
                this.bitmap.Dispose();
                this.bitmap = render.CreateBitmap(new SizeU((uint)textbox.ClientSize.Width, (uint)textbox.ClientSize.Height));
                this.hasCache = false;
            }
        }

        void Markers_Updated(object sender, EventArgs e)
        {
            this.PurgeLayoutCache();
        }

        public void BeginDraw()
        {
            render.BeginDraw();
        }

        public void EndDraw()
        {
            if (render.EndDraw())
            {
                DestructRenderResource();
                InitRenderResource(this.Factory, this.TextBox);
            }
        }

        public double LineNemberWidth
        {
            get
            {
                int length = Int32.Parse(Resources.LineNumberFiledLength);
                if(length < 0)
                    throw new InvalidOperationException("LineNumberFiledLengthは正の値を指定してください");
                length++;   //余白を確保する
                DTextLayout layout = Factory.CreateTextLayout(this.Format, "0", (float)this.ClipRect.Size.Width, (float)this.ClipRect.Size.Height);
                return (int)layout.metrics.widthIncludingTrailingWhitespace * length;
            }
        }

        public event ChangedRenderResourceEventHandler ChangedRenderResource;

        public int TabWidthChar
        {
            get { return this._TabWidthChar; }
            set
            {
                if (value == 0)
                    return;

                this._TabWidthChar = value;

                string str = Util.GetExpandedTab(this._TabWidthChar,0);
                DTextLayout layout = Factory.CreateTextLayout(this.Format, str, (float)this.ClipRect.Size.Width, (float)this.ClipRect.Size.Height);
                this.Format.TabWidth = (int)layout.metrics.widthIncludingTrailingWhitespace;
                layout.Dispose();
            }
        }

        public ColorTable TokenToColor
        {
            get { return _ColorTable; }
            set
            {
                _ColorTable = value;
                DestructRenderResource();
                InitRenderResource(this.Factory, this.TextBox);
            }
        }

        public Rectangle ClipRect
        {
            get;
            set;
        }

        public void PurgeLayoutCache()
        {
            this.cache.Clear();
        }

        public void FillBackground(Rectangle rect)
        {
            DColorBrush brush = this.render.CreateBrush(ToColor2F(this.TokenToColor.Back));
            System.Drawing.Rectangle background = this.TextBox.Bounds;
            this.render.FillRectangle(new RectF((float)background.X, (float)background.Y, (float)background.Right, (float)background.Bottom), brush);
            brush.Dispose();
        }

        public void DrawCachedBitmap(Rectangle rect)
        {
            RectF dstRect = new RectF((float)rect.X, (float)rect.Y, (float)rect.Right, (float)rect.Bottom);    //描く範囲は同じ
            render.DrawBitmap(this.bitmap, dstRect, 1.0f, dstRect);
        }

        public void CacheContent()
        {
            render.Flush();
            this.bitmap.CopyFromRenderTarget(new Point2U(), this.render, new RectU(0, 0, (uint)this.ClipRect.Width, (uint)this.ClipRect.Height));
            this.hasCache = true;
        }

        public bool IsVaildCache()
        {
            return this.hasCache;
        }

        public void DrawLineNumber(int lineNumber, double x, double y)
        {
            string lineNumberFormat = "{0," + Resources.LineNumberFiledLength +  "}";
            DTextLayout layout = Factory.CreateTextLayout(this.Format, string.Format(lineNumberFormat, lineNumber), (float)this.ClipRect.Size.Width, (float)this.ClipRect.Size.Height);
            render.DrawTextLayout(layout, (float)x, (float)y);
            layout.Dispose();
        }

        public double GetXFromIndex(string str, int index)
        {
            DTextLayout layout;
            this.CreateTextLayout(this.Format, str, this.ClipRect.Size, out layout);
            float x, y;
            DHitTestMetrics metrics;
            metrics = layout.HitTestTextPostion(index, false, out x, out y);
            return Util.RoundUp(x);
        }

        public double GetWidthFromIndex(string str, int index)
        {
            DTextLayout layout;
            this.CreateTextLayout(this.Format, str, this.ClipRect.Size,out layout);
            float x, y;
            DHitTestMetrics metrics;
            metrics = layout.HitTestTextPostion(index, false, out x, out y);
            float x2;
            layout.HitTestTextPostion(index, true, out x2, out y);

            return Util.RoundUp(x2 - x);
        }

        public double GetWidth(string str)
        {
            DTextLayout layout;
            this.CreateTextLayout(this.Format, str, this.ClipRect.Size, out layout);
            return Util.RoundUp(layout.metrics.widthIncludingTrailingWhitespace);
        }

        public int AlignIndexToNearestCluster(string str, int index, AlignDirection flow)
        {
            DTextLayout layout;
            this.CreateTextLayout(this.Format, str, this.ClipRect.Size, out layout);
            float x, y;
            DHitTestMetrics metrics;
            metrics = layout.HitTestTextPostion(index, false, out x, out y);

            if (flow == AlignDirection.Forward)
                return Util.RoundUp(metrics.textPosition + metrics.length);
            else if (flow == AlignDirection.Back) 
                return Util.RoundUp(metrics.textPosition);
            throw new ArgumentOutOfRangeException();
        }


        public int GetIndexFromX(string str, double x)
        {
            if (str == string.Empty || str == null || str[0] == Document.EndOfFile)
                return 0;

            DTextLayout layout;
            this.CreateTextLayout(this.Format, str, this.ClipRect.Size, out layout);
            bool isTrailing, isInsed;
            DHitTestMetrics metrics;
            metrics = layout.HitTextPoint((float)x, 0, out isTrailing, out isInsed);
            if (isTrailing)
                return Util.RoundUp(metrics.textPosition + metrics.length);
            else
                return Util.RoundUp(metrics.textPosition);
        }

        public double GetHeight(string str)
        {
            DTextLayout layout;
            this.CreateTextLayout(this.Format, str, this.ClipRect.Size,out layout);
            int height = 0;
            DLineMetrics[] metrics = layout.LineMetrics;
            if(metrics != null && metrics.Length > 0)
                height = Util.RoundUp(metrics[0].height);
            return height;
        }

        public List<LineToIndexTableData> BreakLine(Document doc, int startIndex, int endIndex, double wrapwidth)
        {
            List<LineToIndexTableData> output = new List<LineToIndexTableData>();

            this.Format.WordWrapping = DWordWrapping.Wrapping;

            foreach (string str in Util.GetLines(doc, startIndex, endIndex))
            {
                DTextLayout layout = Factory.CreateTextLayout(this.Format, str, (float)wrapwidth, Int32.MaxValue);

                int i = startIndex;
                foreach (DLineMetrics metrics in layout.LineMetrics)
                {
                    if (metrics.length == 0 && metrics.newlineLength == 0)
                        continue;

                    bool lineend = false;
                    if (metrics.newlineLength > 0 || doc[i + (int)metrics.length - 1] == Document.EndOfFile)
                        lineend = true;

                    output.Add(new LineToIndexTableData(i, (int)metrics.length, lineend, null));
                    i += Util.RoundUp(metrics.length);
                }

                layout.Dispose();

                startIndex += str.Length;
            }

            this.Format.WordWrapping = DWordWrapping.NoWrapping;

            return output;
        }

        public void DrawLine(Point from,Point to)
        {
            this.render.AntialiasMode = DAntialias.Alias;

            this.render.DrawLine(new Point2F((float)from.X, (float)from.Y), new Point2F((float)to.X, (float)to.Y), 1, this.brushs["default"], null);

            this.render.AntialiasMode = DAntialias.Antialias;
        }

        public void DrawCaret(Rectangle rect)
        {
            this.render.AntialiasMode = DAntialias.Alias;

            this.render.FillRectangle(new RectF((float)rect.X, (float)rect.Y, (float)rect.Right, (float)rect.Bottom), this.brushs["default"]);

            this.render.AntialiasMode = DAntialias.Antialias;
        }

        public void DrawOneLine(LineToIndexTable lti, int row, double x, double y, IEnumerable<Selection> SelectRanges, IEnumerable<Marker> MarkerRanges)
        {
            string str = lti[row];

            if (str == string.Empty || str == null)
                return;

            DTextLayout layout;
            this.CreateTextLayout(this.Format, str, this.ClipRect.Size, out layout);
            if ((bool)layout.Tag == false)
            {
                LineToIndexTableData lineData = lti.GetData(row);
                if (lineData.syntax != null)
                {
                    foreach (SyntaxInfo s in lineData.syntax)
                    {
                        int id = EffectResourceConverter.GetID(s.type);
                        if (id != 0)
                            layout.SetTextEffect(s.index, s.length, this.effects[id]);
                    }
                }

                foreach (Marker m in MarkerRanges)
                {
                    if (m.start == -1 || m.length == 0)
                        continue;
                    switch (m.hilight)
                    {
                        case HilightType.Url:
                            layout.SetUnderline(m.start, m.length, true);
                            break;
                        case HilightType.Sold:
                            layout.SetUnderline(m.start, m.length, true);
                            break;
                    }
                    int id = EffectResourceConverter.GetID(m.hilight);
                    if(id > 0)
                        layout.SetTextEffect(m.start, m.length, this.effects[id]);
                }
                layout.Tag = true;
            }

            foreach (Selection sel in SelectRanges)
            {
                if (sel.length == 0 || sel.start == -1)
                    continue;

                this.FillSelectArea(layout, sel.start, sel.length, (float)x , (float)y);
            }

            this.render.PushAxisAlignedClip(new RectF((float)this.ClipRect.X, (float)this.ClipRect.Y, (float)this.ClipRect.Right, (float)this.ClipRect.Bottom), DAntialias.Alias);

            render.DrawTextLayout(layout, (float)x, (float)y);

            this.render.PopAxisAlignedClip();
        }

        public void Dispose()
        {
            DestructRenderResource();
            Factory.Dispose();
        }

        void FillSelectArea(DTextLayout layout, int start, int length,float x,float y)
        {
            this.render.AntialiasMode = DAntialias.Alias;

            DHitTestMetrics[] metrics = layout.HitTestTextRange(start, length, x, y);
            foreach (DHitTestMetrics metric in metrics)
            {
                this.render.FillRectangle(new RectF(metric.left, metric.top, metric.left + metric.width, metric.top + metric.height), this.brushs["hilight"]);
            }

            this.render.AntialiasMode = DAntialias.Antialias;
        }

        bool CreateTextLayout(DTextFormat format, string str, Size size,out DTextLayout layout)
        {
            bool has = this.cache.TryGetValue(str, out layout);
            if (has == false)
            {
                layout = this.Factory.CreateGdiTextLayout(format, str, (float)size.Width, (float)size.Height, true);
                layout.Tag = false;
                this.cache.Add(str, layout);
            }
            return has;
        }
    }

}
