﻿using System;
using System.Collections.Generic;

namespace FooEditEngine
{
    struct Point
    {
        public double X;
        public double Y;
        public Point(double x, double y)
        {
            this.X = x;
            this.Y = y;
        }
#if WINFORM
        public static implicit operator Point(System.Drawing.Point p)
        {
            return new Point(p.X, p.Y);
        }
        public static implicit operator System.Drawing.Point(Point p)
        {
            return new System.Drawing.Point((int)p.X, (int)p.Y);
        }
#endif
#if WPF
        public static implicit operator Point(System.Windows.Point p)
        {
            return new Point(p.X, p.Y);
        }
        public static implicit operator System.Windows.Point(Point p)
        {
            return new System.Windows.Point(p.X, p.Y);
        }
#endif
    }
    struct Size
    {
        public double Width;
        public double Height;
        public Size(double width, double height)
        {
            this.Width = width;
            this.Height = height;
        }
#if WINFORM
        public static implicit operator Size(System.Drawing.Size p)
        {
            return new Size(p.Width, p.Height);
        }
        public static implicit operator System.Drawing.Size(Size p)
        {
            return new System.Drawing.Size((int)p.Width, (int)p.Height);
        }
#endif
#if WPF
        public static implicit operator Size(System.Windows.Size p)
        {
            return new Size(p.Width, p.Height);
        }
        public static implicit operator System.Windows.Size(Size p)
        {
            return new System.Windows.Size(p.Width, p.Height);
        }
#endif
    }
    struct Rectangle
    {
        public Point Location;
        public Size Size;
        public Point TopLeft
        {
            get { return this.Location; }
        }
        public Point TopRight
        {
            get { return new Point(this.Right, this.Location.Y); }
        }
        public Point BottomLeft
        {
            get { return new Point(this.Location.X, this.Bottom); }
        }
        public Point BottomRight
        {
            get { return new Point(this.Right, this.Bottom); }
        }
        public double Right
        {
            get { return this.X + this.Width; }
        }
        public double Bottom
        {
            get { return this.Y + this.Height; }
        }
        public double Height
        {
            get { return this.Size.Height; }
            set { this.Size.Height = value; }
        }
        public double Width
        {
            get { return this.Size.Width; }
            set { this.Size.Width = value; }
        }
        public double X
        {
            get { return this.Location.X; }
        }
        public double Y
        {
            get { return this.Location.Y; }
        }
        public Rectangle(double x, double y, double width, double height)
        {
            this.Location = new Point(x, y);
            this.Size = new Size(width, height);
        }
#if WINFORM
        public static implicit operator Rectangle(System.Drawing.Rectangle p)
        {
            return new Rectangle(p.X,p.Y,p.Width,p.Height);
        }
        public static implicit operator System.Drawing.Rectangle(Rectangle p)
        {
            return new System.Drawing.Rectangle((int)p.X, (int)p.Y, (int)p.Width, (int)p.Height);
        }
        public static implicit operator SharpDX.RectangleF(Rectangle p)
        {
            return new SharpDX.RectangleF((float)p.X, (float)p.Y, (float)p.Right, (float)p.Bottom);
        }
#endif
#if WPF
        public static implicit operator Rectangle(System.Windows.Rect p)
        {
            return new Rectangle(p.X,p.Y,p.Width,p.Height);
        }
        public static implicit operator System.Windows.Rect(Rectangle p)
        {
            return new System.Windows.Rect(p.X, p.Y, p.Width, p.Height);
        }
        public static implicit operator SharpDX.RectangleF(Rectangle p)
        {
            return new SharpDX.RectangleF((float)p.X, (float)p.Y, (float)p.Right, (float)p.Bottom);
        }
#endif
    }
    enum AlignDirection
    {
        Forward,
        Back,
    }
    enum ResourceType
    {
        Font,
        Brush,
        Antialias,
        InlineChar,
    }
    class ChangedRenderRsourceEventArgs : EventArgs
    {
        public ResourceType type;
        public ChangedRenderRsourceEventArgs(ResourceType type)
        {
            this.type = type;
        }
    }
    delegate void ChangedRenderResourceEventHandler(object sender, ChangedRenderRsourceEventArgs e);
    interface ITextRender
    {
        bool RightToLeft { get; set; }

        /// <summary>
        /// ドキュメントを表示する領域
        /// </summary>
        Rectangle ClipRect { get; set; }
        
        /// <summary>
        /// 行番号の幅
        /// </summary>
        double LineNemberWidth { get; }
        
        /// <summary>
        /// タブの文字数
        /// </summary>
        int TabWidthChar { get; set; }

        /// <summary>
        /// 保持しているリソースに変化があったことを通知する
        /// </summary>
        event ChangedRenderResourceEventHandler ChangedRenderResource;

        /// <summary>
        /// RightToLeftの値が変わったことを通知する
        /// </summary>
        event EventHandler ChangedRightToLeft;

        /// <summary>
        /// キャッシュされたビットマップを描写する
        /// </summary>
        /// <param name="rect">描く領域</param>
        void DrawCachedBitmap(Rectangle rect);

        /// <summary>
        /// 描写したものをキャッシュする
        /// </summary>
        void CacheContent();

        /// <summary>
        /// キャッシュが存在するなら真を返し、そうでないなら偽を返す
        /// </summary>
        bool IsVaildCache();

        /// <summary>
        /// 行番号を表示する
        /// </summary>
        /// <param name="lineNumber">表示すべき番号</param>
        /// <param name="x">行番号の左上を示すX座標</param>
        /// <param name="y">行番号の左上を示すY座標</param>
        void DrawLineNumber(int lineNumber, double x, double y);

        /// <summary>
        /// ラインマーカーを描く
        /// </summary>
        /// <param name="rect">文字列と外接する四角形</param>
        void DrawLineMarker(Rectangle rect);

        /// <summary>
        /// キャレットを描く
        /// </summary>
        /// <param name="rect"></param>
        /// <param name="transparent"></param>
        void DrawCaret(Rectangle rect,bool transparent);

        /// <summary>
        /// 背景を塗りつぶす
        /// </summary>
        /// <param name="rect">塗りつぶすべき領域</param>
        void FillBackground(Rectangle rect);

        /// <summary>
        /// 行を表示する
        /// </summary>
        /// <param name="lti">LineToIndexオブジェクト</param>
        /// <param name="row">行</param>
        /// <param name="x">行の左上を表すX座標</param>
        /// <param name="y">行の左上を表すY座標</param>
        /// <param name="SelectRanges">選択領域を保持しているコレクション。選択領域の開始位置は行の先頭を０とする相対位置としてください（位置が-1の場合表示されません）</param>
        void DrawOneLine(LineToIndexTable lti, int row, double x, double y, IEnumerable<Selection> SelectRanges);

        /// <summary>
        /// 行を折り返す
        /// </summary>
        /// <param name="doc">ドキュメント</param>
        /// <param name="startIndex">開始インデックス</param>
        /// <param name="endIndex">終了インデックス</param>
        /// <param name="wrapwidth">折り返しの幅</param>
        /// <returns>行のリスト</returns>
        List<LineToIndexTableData> BreakLine(Document doc, int startIndex, int endIndex, double wrapwidth);

        /// <summary>
        /// レイアウトを生成する
        /// </summary>
        /// <param name="str">文字列</param>
        /// <returns>ITextLayoutオブジェクト</returns>
        /// <param name="MarkerRanges">マーカーを保持しているコレクション。マーカーの開始位置は行の先頭を０とする相対位置としてください（位置が-1の場合表示しないこと）</param>
        ITextLayout CreateLaytout(string str,SyntaxInfo[] syntaxCollection, IEnumerable<Marker> MarkerRanges);
    }

    enum StringAlignment
    {
        Left,
        Center,
        Right,
    }
    interface IPrintableTextRender : ITextRender
    {
        /// <summary>
        /// 文字列を表示する
        /// </summary>
        /// <param name="str">文字列</param>
        /// <param name="x">x座標</param>
        /// <param name="y">y座標</param>
        /// <param name="align">書式方向</param>
        void DrawString(string str, double x, double y,StringAlignment align);

        /// <summary>
        /// ヘッダーの高さ
        /// </summary>
        float HeaderHeight { get; }

        /// <summary>
        /// フッターの高さ
        /// </summary>
        float FooterHeight { get; }
    }
}
