﻿using System;
using System.IO;
using System.Collections.Generic;
using System.Text;
using System.Text.RegularExpressions;
using System.Linq;

namespace FooEditEngine
{
    /// <summary>
    /// 進行状況を表す列挙体
    /// </summary>
    public enum ProgressState
    {
        /// <summary>
        /// 操作が開始したことを表す
        /// </summary>
        Start,
        /// <summary>
        /// 操作が終了したことを表す
        /// </summary>
        Complete,
    }

    /// <summary>
    /// 進行状況を表すためのイベントデータ
    /// </summary>
    public class ProgressEventArgs : EventArgs
    {
        /// <summary>
        /// 進行状況
        /// </summary>
        public ProgressState state;
        /// <summary>
        /// コンストラクター
        /// </summary>
        /// <param name="state">ProgressStateオブジェクト</param>
        public ProgressEventArgs(ProgressState state)
        {
            this.state = state;
        }
    }

    /// <summary>
    /// 進行状況を通知するためのデリゲート
    /// </summary>
    /// <param name="sender">送信元クラス</param>
    /// <param name="e">イベントデータ</param>
    public delegate void ProgressEventHandler(object sender, ProgressEventArgs e);

    /// <summary>
    /// 更新タイプを表す列挙体
    /// </summary>
    public enum UpdateType
    {
        /// <summary>
        /// ドキュメントが置き換えられたことを表す
        /// </summary>
        Replace,
        /// <summary>
        /// ドキュメント全体が削除されたことを表す
        /// </summary>
        Clear,
    }

    /// <summary>
    /// 更新タイプを通知するためのイベントデータ
    /// </summary>
    public class DocumentUpdateEventArgs : EventArgs
    {
        /// <summary>
        /// 更新タイプ
        /// </summary>
        public UpdateType type;
        /// <summary>
        /// 開始位置
        /// </summary>
        public int startIndex;
        /// <summary>
        /// 削除された長さ
        /// </summary>
        public int removeLength;
        /// <summary>
        /// 追加された長さ
        /// </summary>
        public int insertLength;
        /// <summary>
        /// コンストラクター
        /// </summary>
        /// <param name="type">更新タイプ</param>
        /// <param name="startIndex">開始インデックス</param>
        /// <param name="removeLength">削除された長さ</param>
        /// <param name="insertLength">追加された長さ</param>
        public DocumentUpdateEventArgs(UpdateType type, int startIndex, int removeLength, int insertLength)
        {
            this.type = type;
            this.startIndex = startIndex;
            this.removeLength = removeLength;
            this.insertLength = insertLength;
        }
    }

    /// <summary>
    /// ドキュメントに更新があったことを伝えるためのデリゲート
    /// </summary>
    /// <param name="sender">送信元クラス</param>
    /// <param name="e">イベントデータ</param>
    public delegate void DocumentUpdateEventHandler(object sender, DocumentUpdateEventArgs e);

    /// <summary>
    /// ドキュメントの管理を行う
    /// </summary>
    public class Document : IEnumerable<char>, IRandomEnumrator<char>
    {
        const int ProgressNotifyCount = 100;
        Regex regex;
        Match match;
        StringBuffer buffer = new StringBuffer();

        /// <summary>
        /// コンストラクター
        /// </summary>
        public Document()
        {
            this.buffer.Update += new DocumentUpdateEventHandler(buffer_Update);
            this.LoadProgress += new ProgressEventHandler((s, e) => { });
            this.SaveProgress += new ProgressEventHandler((s, e) => { });
            this.Update += new DocumentUpdateEventHandler((s, e) => { });
            this.ChangeFireUpdateEvent += new EventHandler((s,e)=>{});
            this.Markers = new MarkerCollection(this);
            this.UndoManager = new UndoManager();
        }

        /// <summary>
        /// 読み出し中に呼び出されるイベント
        /// </summary>
        public event ProgressEventHandler LoadProgress;

        /// <summary>
        /// 保存中に呼び出されるイベント
        /// </summary>
        public event ProgressEventHandler SaveProgress;

        /// <summary>
        /// ドキュメントが更新された時に呼ばれるイベント
        /// </summary>
        public event DocumentUpdateEventHandler Update;

        /// <summary>
        /// FireUpdateEventの値が変わったときに呼び出されるイベント
        /// </summary>
        public event EventHandler ChangeFireUpdateEvent;

        /// <summary>
        /// 改行コードの内部表現
        /// </summary>
        public const char NewLine = '\n';

        /// <summary>
        /// EOFの内部表現
        /// </summary>
        public const char EndOfFile = '\u001a';

        /// <summary>
        /// アンドゥ管理クラスを表す
        /// </summary>
        public UndoManager UndoManager
        {
            get;
            private set;
        }

        /// <summary>
        /// 文字列の長さ
        /// </summary>
        public int Length
        {
            get
            {
                return this.buffer.Length;
            }
        }

        /// <summary>
        /// 変更のたびにUpdateイベントを発生させるかどうか
        /// </summary>
        public bool FireUpdateEvent
        {
            get { return this.buffer.EnableFireUpdateEvent; }
            set { this.buffer.EnableFireUpdateEvent = value; this.ChangeFireUpdateEvent(this, null); }
        }

        /// <summary>
        /// ドキュメントの変更が加えられたかどうか
        /// </summary>
        public bool Dirty
        {
            get;
            private set;
        }

        /// <summary>
        /// インデクサー
        /// </summary>
        /// <param name="i">インデックス（自然数でなければならない）</param>
        /// <returns>Char型</returns>
        public char this[int i]
        {
            get
            {
                return this.buffer[i];
            }
        }

        /// <summary>
        /// 選択範囲コレクション
        /// </summary>
        internal SelectCollection Selections
        {
            get{return this.buffer.Selections;}
        }

        /// <summary>
        /// マーカーコレクション
        /// </summary>
        public MarkerCollection Markers
        {
            get;
            private set;
        }

        /// <summary>
        /// マーカーを設定する
        /// </summary>
        /// <param name="m">設定したいマーカー</param>
        public void SetMarker(Marker m)
        {
            if (m.start < 0 || m.start + m.length > this.Length)
                throw new ArgumentOutOfRangeException("startもしくはendが指定できる範囲を超えています");

            this.Markers.Add(m);
        }

        /// <summary>
        /// マーカーを削除する
        /// </summary>
        /// <param name="start">開始インデックス</param>
        /// <param name="length">削除する長さ</param>
        public void RemoveMarker(int start, int length)
        {
            if (start < 0 || start + length > this.Length)
                throw new ArgumentOutOfRangeException("startもしくはendが指定できる範囲を超えています");

            int end = start + length - 1;
            this.Markers.RemoveAll((m) =>
            {
                int markerEnd = m.start + m.length - 1;
                if (m.start >= start && markerEnd <= end ||
                    markerEnd >= start && markerEnd <= end ||
                    m.start >= start && m.start <= end ||
                    m.start < start && markerEnd > end)
                    return true;
                else
                    return false;
            });
        }

        /// <summary>
        /// マーカーを削除する
        /// </summary>
        /// <param name="type">削除したいマーカーのタイプ</param>
        public void RemoveMarker(HilightType type)
        {
            this.Markers.RemoveAll((e) =>
            {
                if (e.hilight == type)
                    return true;
                else
                    return false;
            });
        }

        /// <summary>
        /// インデックスに対応するマーカーを得る
        /// </summary>
        /// <param name="index">インデックス</param>
        /// <returns>Marker構造体の列挙子</returns>
        public IEnumerable<Marker> GetMarkers(int index)
        {
            if (index < 0 || index > this.Length)
                throw new ArgumentOutOfRangeException("indexが範囲を超えています");
            var result = from s in this.Markers
                         where s.start <= index && index <= s.start + s.length
                         select s;
            return result;
        }

        /// <summary>
        /// 部分文字列を取得する
        /// </summary>
        /// <param name="index">開始インデックス</param>
        /// <param name="length">長さ</param>
        /// <returns>Stringオブジェクト</returns>
        public string ToString(int index, int length)
        {
            return this.buffer.ToString(index, length);
        }

        /// <summary>
        /// インデックスを開始位置とする文字列を返す
        /// </summary>
        /// <param name="index">開始インデックス</param>
        /// <returns>Stringオブジェクト</returns>
        public string ToString(int index)
        {
            return this.ToString(index, this.buffer.Length - index);
        }

        /// <summary>
        /// 文字列を追加する
        /// </summary>
        /// <param name="s">追加したい文字列</param>
        public void Append(string s)
        {
            this.Replace(this.buffer.Length, 0, s);
        }

        /// <summary>
        /// 文字列を挿入する
        /// </summary>
        /// <param name="index">開始インデックス</param>
        /// <param name="s">追加したい文字列</param>
        public void Insert(int index, string s)
        {
            this.Replace(index, 0, s);
        }

        /// <summary>
        /// 文字列を削除する
        /// </summary>
        /// <param name="index">開始インデックス</param>
        /// <param name="length">長さ</param>
        public void Remove(int index, int length)
        {
            this.Replace(index, length, "");
        }

        /// <summary>
        /// ドキュメントを置き換える
        /// </summary>
        /// <param name="index">開始インデックス</param>
        /// <param name="length">長さ</param>
        /// <param name="s">文字列</param>
        public void Replace(int index, int length, string s)
        {
            if (index < 0 || index > this.buffer.Length || index + length > this.buffer.Length || length < 0)
                throw new ArgumentOutOfRangeException();
            if (length == 0 && (s == string.Empty || s == null))
                return;

            this.RemoveMarker(index, length);

            this.Dirty = true;

            ReplaceCommand cmd = new ReplaceCommand(this.buffer, index, length, s);
            this.UndoManager.push(cmd);
            cmd.redo();
        }

        /// <summary>
        /// 物理行をすべて削除する
        /// </summary>
        /// <remarks>Dirtyフラグも同時にクリアーされます</remarks>
        public void Clear()
        {
            this.buffer.Clear();
            
            this.Dirty = false;
        }

        /// <summary>
        /// ファイルからドキュメントを構築します
        /// </summary>
        /// <param name="filepath">読み取り先のファイル</param>
        /// <param name="enc">エンコーディング</param>
        public void Load(string filepath, Encoding enc)
        {
            using (StreamReader sr = new StreamReader(filepath, enc))
            {
                this.Load(sr);
            }
        }

        /// <summary>
        /// ストリームからドキュメントを構築します
        /// </summary>
        /// <param name="sr">読み取り先のストリーム</param>
        /// <remarks>読み取り時にドキュメントの内容は削除されます</remarks>
        public void Load(TextReader sr)
        {
            if (sr.Peek() == -1)
            {
                this.LoadProgress(this, new ProgressEventArgs(ProgressState.Complete));
                return;
            }

            try
            {
                this.Clear();

                this.LoadProgress(this, new ProgressEventArgs(ProgressState.Start));
                this.FireUpdateEvent = false;
                this.UndoManager.BeginLock();
                string str;
                for (int i = 0; (str = sr.ReadLine()) != null; i++)
                {
                    int index = this.Length;
                    if (index < 0)
                        index = 0;

                    this.buffer.Replace(index, 0, str + Document.NewLine);
                }
            }
            finally
            {
                this.UndoManager.EndLock();
                this.FireUpdateEvent = true;
                this.LoadProgress(this, new ProgressEventArgs(ProgressState.Complete));
                this.Dirty = false;
            }
        }

        /// <summary>
        /// ドキュメントをファイルに保存します
        /// </summary>
        /// <param name="filepath">保存先のファイル</param>
        /// <param name="enc">保存したいエンコード</param>
        /// <param name="newline">改行を表す文字列</param>
        public void Save(string filepath, Encoding enc, string newline)
        {
            using (StreamWriter sw = new StreamWriter(filepath, false, enc))
            {
                sw.NewLine = newline;
                this.Save(sw);
            }
        }

        /// <summary>
        /// ストリームに保存します
        /// </summary>
        /// <param name="sw">保存先のストリーム</param>
        public void Save(StreamWriter sw)
        {
            try
            {
                this.SaveProgress(this, new ProgressEventArgs(ProgressState.Start));
                StringBuilder line = new StringBuilder();
                for (int i = 0; i < this.Length; i++)
                {
                    char c = this[i];
                    line.Append(c);
                    if (c == Document.NewLine || i == this.Length - 1)
                    {
                        string str = line.ToString();
                        str = str.Replace(Document.NewLine.ToString(), sw.NewLine);
                        sw.Write(str);
                        line.Clear();
                    }
                }
            }
            finally
            {
                this.SaveProgress(this, new ProgressEventArgs(ProgressState.Complete));
                this.Dirty = false;
            }
        }

        /// <summary>
        /// Find()で使用するパラメーターをセットします
        /// </summary>
        /// <param name="pattern">検索したい文字列</param>
        /// <param name="UseRegex">正規表現を使用するなら真</param>
        /// <param name="opt">RegexOptions列挙体</param>
        public void SetFindParam(string pattern, bool UseRegex, RegexOptions opt)
        {
            this.match = null;
            if (UseRegex)
                this.regex = new Regex(pattern.Normalize(), opt);
            else
                this.regex = new Regex(Regex.Escape(pattern.Normalize()), opt);
        }

        /// <summary>
        /// 指定した文字列を検索します
        /// </summary>
        /// <returns>見つかった場合はSearchResult列挙子を返却します</returns>
        /// <remarks>見つかったパターン以外を置き換えた場合、正常に動作しないことがあります</remarks>
        public IEnumerator<SearchResult> Find()
        {
            if (this.regex == null)
                throw new InvalidOperationException();

            StringBuilder line = new StringBuilder();
            int oldLength = this.Length;
            for (int i = 0; i < this.Length; i++)
            {
                char c = this[i];
                line.Append(c);
                if (c == Document.NewLine || i == this.Length - 1)
                {
                    this.match = this.regex.Match(line.ToString().Normalize());
                    while (this.match.Success)
                    {
                        int start = i - line.Length + 1 + this.match.Index;
                        int end = start + this.match.Length - 1;
                        
                        yield return new SearchResult(this.match, start, end);

                        if (this.Length != oldLength)   //長さが変わった場合は置き換え後のパターンの終点＋１まで戻る
                        {
                            i = end - (oldLength - this.Length);
                            oldLength = this.Length;
                            break;
                        }
                        
                        this.match = this.match.NextMatch();
                    }
                    line.Clear();
                }
            }
        }

        /// <summary>
        /// 任意のパターンですべて置き換えます
        /// </summary>
        /// <param name="replacePattern">置き換え後のパターン</param>
        /// <param name="groupReplace">グループ置き換えを行うなら真。そうでないなら偽</param>
        public void ReplaceAll(string replacePattern,bool groupReplace)
        {
            if (this.regex == null)
                throw new InvalidOperationException();
            ReplaceAllCommand cmd = new ReplaceAllCommand(this.buffer, this.regex, replacePattern, groupReplace);
            this.UndoManager.push(cmd);
            cmd.redo();
        }

        #region IEnumerable<char> メンバー

        /// <summary>
        /// 列挙子を返します
        /// </summary>
        /// <returns>IEnumeratorオブジェクトを返す</returns>
        public IEnumerator<char> GetEnumerator()
        {
            for (int i = 0; i < this.Length; i++)
                yield return this[i];
        }

        #endregion

        #region IEnumerable メンバー

        System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator()
        {
            throw new NotImplementedException();
        }

        #endregion

        void buffer_Update(object sender, DocumentUpdateEventArgs e)
        {
            this.Update(this, e);
        }
    }

    /// <summary>
    /// 検索結果を表す
    /// </summary>
    public class SearchResult
    {
        private Match Match;

        /// <summary>
        /// 一致した場所の開始位置を表す
        /// </summary>
        public int Start;

        /// <summary>
        /// 一致した場所の終了位置を表す
        /// </summary>
        public int End;

        /// <summary>
        /// 見つかった文字列を返す
        /// </summary>
        public string Value
        {
            get { return this.Match.Value; }
        }

        /// <summary>
        /// 指定したパターンを置き換えて返す
        /// </summary>
        /// <param name="replacement">置き換える文字列</param>
        /// <returns>置き換え後の文字列</returns>
        public string Result(string replacement)
        {
            return this.Match.Result(replacement);
        }

        /// <summary>
        /// コンストラクター
        /// </summary>
        /// <param name="m">Matchオブジェクト</param>
        /// <param name="start">開始インデックス</param>
        /// <param name="end">終了インデックス</param>
        public SearchResult(Match m, int start,int end)
        {
            this.Match = m;
            this.Start = start;
            this.End = end;
        }
    }
}
