﻿/*
 * Copyright (C) 2013 FooProject
 * * This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or (at your option) any later version.

 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
using System;
using System.ComponentModel;
using System.Threading.Tasks;
using Windows.ApplicationModel.Resources.Core;
using Windows.Devices.Input;
using Windows.System;
using Windows.ApplicationModel.DataTransfer;
using Windows.Foundation;
using Windows.UI;
using Windows.UI.Input;
using Windows.UI.Core;
using Windows.UI.Popups;
using Windows.UI.Text;
using Windows.UI.Xaml.Media;
using Windows.UI.Xaml;
using Windows.UI.Xaml.Controls;
using Windows.UI.Xaml.Controls.Primitives;
using Windows.UI.Xaml.Input;
using DotNetTextStore;
using DotNetTextStore.UnmanagedAPI.TSF;
using DotNetTextStore.UnmanagedAPI.WinDef;

// テンプレート コントロールのアイテム テンプレートについては、http://go.microsoft.com/fwlink/?LinkId=234235 を参照してください

namespace FooEditEngine.Metro
{
    /// <summary>
    /// テキストボックスコントロール
    /// </summary>
    public sealed class FooTextBox : Control,IDisposable
    {
        EditView View;
        Controller _Controller;
        D2DRender Render;
        ScrollBar horizontalScrollBar, verticalScrollBar;
        Windows.UI.Xaml.Shapes.Rectangle rectangle;
        GestureRecognizer gestureRecongnizer = new GestureRecognizer();
        TextStore2 textStore;
        FooTextBoxAutomationPeer peer;
        bool nowCaretMove = false;

        /// <summary>
        /// コンストラクター
        /// </summary>
        public FooTextBox()
        {
            this.DefaultStyleKey = typeof(FooTextBox);

            this.textStore = new TextStore2();
            this.textStore.IsLoading += textStore_IsLoading;
            this.textStore.IsReadOnly += textStore_IsReadOnly;
            this.textStore.GetStringLength += () => this.Document.Length;
            this.textStore.GetString += _textStore_GetString;
            this.textStore.GetSelectionIndex += _textStore_GetSelectionIndex;
            this.textStore.SetSelectionIndex += _textStore_SetSelectionIndex;
            this.textStore.InsertAtSelection += _textStore_InsertAtSelection;
            this.textStore.GetScreenExtent += _textStore_GetScreenExtent;
            this.textStore.GetStringExtent += _textStore_GetStringExtent;
            this.textStore.CompositionStarted += textStore_CompositionStarted;
            this.textStore.CompositionUpdated += textStore_CompositionUpdated;
            this.textStore.CompositionEnded += textStore_CompositionEnded;

            this.Document = new Document();
            DocumentExtend.Progress += Document_Progress;

            this.rectangle = new Windows.UI.Xaml.Shapes.Rectangle();
            this.rectangle.Margin = this.Padding;
            this.Render = new D2DRender(this,this.rectangle,this.textStore);

            this.View = new EditView(this.Document, this.Render, new Margin(Gripper.GripperWidth / 2, Gripper.GripperWidth, Gripper.GripperWidth / 2, Gripper.GripperWidth));
            this.View.SrcChanged += View_SrcChanged;
            this.View.InsertMode = this.InsertMode;
            this.View.DrawLineNumber = this.DrawLineNumber;
            this.View.HideCaret = !this.DrawCaret;
            this.View.HideLineMarker = !this.DrawCaretLine;
            this.View.HideRuler = !this.DrawRuler;
            this.View.UrlMark = this.MarkURL;
            this.View.TabStops = this.TabChars;

            this._Controller = new Controller(this.Document, this.View);
            this._Controller.SelectionChanged += Controller_SelectionChanged;

            this.FirstGripper = new Gripper(this._Controller, this.View, this.Render, GripperPostion.Upper);
            this.SecondGripper = new Gripper(this._Controller, this.View, this.Render, GripperPostion.Bottom);

            this.gestureRecongnizer.GestureSettings = GestureSettings.Drag | 
                GestureSettings.RightTap | 
                GestureSettings.Tap | 
                GestureSettings.ManipulationTranslateX | 
                GestureSettings.ManipulationTranslateY |
                GestureSettings.ManipulationScale;
            this.gestureRecongnizer.RightTapped += gestureRecongnizer_RightTapped;
            this.gestureRecongnizer.Tapped += gestureRecongnizer_Tapped;
            this.gestureRecongnizer.Dragging += gestureRecongnizer_Dragging;
            this.gestureRecongnizer.ManipulationStarted += gestureRecongnizer_ManipulationStarted;
            this.gestureRecongnizer.ManipulationUpdated += gestureRecongnizer_ManipulationUpdated;
            this.gestureRecongnizer.ManipulationCompleted += gestureRecongnizer_ManipulationCompleted;

            //Viewの初期化が終わった直後に置かないと例外が発生する
            this.Document.Update += Document_Update;

            Window.Current.CoreWindow.CharacterReceived += CoreWindow_CharacterReceived;

            this.SizeChanged += FooTextBox_SizeChanged;

            this.Loaded += FooTextBox_Loaded;
        }

        /// <summary>
        /// ファイナライザー
        /// </summary>
        ~FooTextBox()
        {
            this.Dispose(false);
        }

        /// <summary>
        /// アンマネージドリソースを解放する
        /// </summary>
        public void Dispose()
        {
            this.Dispose(true);
            GC.SuppressFinalize(this);
        }

        bool Disposed = false;
        private void Dispose(bool disposing)
        {
            if (this.Disposed)
                return;
            if (disposing)
            {
                this.textStore.Dispose();
                this.View.Dispose();
                this.Render.Dispose();
            }
            DocumentExtend.Progress -= this.Document_Progress;
        }

        /// <summary>
        /// ドキュメントを選択する
        /// </summary>
        /// <param name="start">開始インデックス</param>
        /// <param name="length">長さ</param>
        public void Select(int start, int length)
        {
            if (this.Document.State == AsyncState.Loading)
                throw new InvalidOperationException();
            this._Controller.Select(start, length);
        }

        /// <summary>
        /// キャレットを指定した行に移動させます
        /// </summary>
        /// <param name="index">インデックス</param>
        /// <remarks>このメソッドを呼び出すと選択状態は解除されます</remarks>
        public void JumpCaret(int index)
        {
            if (this.Document.State == AsyncState.Loading)
                throw new InvalidOperationException();
            this._Controller.JumpCaret(index);
        }
        /// <summary>
        /// キャレットを指定した行と桁に移動させます
        /// </summary>
        /// <param name="row">行番号</param>
        /// <param name="col">桁</param>
        /// <remarks>このメソッドを呼び出すと選択状態は解除されます</remarks>
        public void JumpCaret(int row, int col)
        {
            if (this.Document.State == AsyncState.Loading)
                throw new InvalidOperationException();
            this._Controller.JumpCaret(row, col);
        }

        /// <summary>
        /// 選択中のテキストをクリップボードにコピーします
        /// </summary>
        public void Copy()
        {
            if (this.Document.State == AsyncState.Loading)
                throw new InvalidOperationException();
            string text = this._Controller.SelectedText;
            if (text != null && text != string.Empty)
            {
                DataPackage dataPackage = new DataPackage();
                dataPackage.RequestedOperation = DataPackageOperation.Copy;
                dataPackage.SetText(text);

                Clipboard.SetContent(dataPackage); 
            }
        }

        /// <summary>
        /// 選択中のテキストをクリップボードに切り取ります
        /// </summary>
        public void Cut()
        {
            if (this.Document.State == AsyncState.Loading)
                throw new InvalidOperationException();
            string text = this._Controller.SelectedText;
            if (text != null && text != string.Empty)
            {
                DataPackage dataPackage = new DataPackage();
                dataPackage.RequestedOperation = DataPackageOperation.Copy;
                dataPackage.SetText(text);

                Clipboard.SetContent(dataPackage);
                
                this._Controller.SelectedText = "";
            }
        }

        /// <summary>
        /// 選択中のテキストを貼り付けます
        /// </summary>
        public async Task PasteAsync()
        {
            if (this.Document.State == AsyncState.Loading)
                throw new InvalidOperationException();

            var dataPackageView = Clipboard.GetContent();
            if (dataPackageView.Contains(StandardDataFormats.Text))
            {
                this._Controller.SelectedText = await dataPackageView.GetTextAsync();
            }
        }

        /// <summary>
        /// すべて選択する
        /// </summary>
        public void SelectAll()
        {
            if (this.Document.State == AsyncState.Loading)
                throw new InvalidOperationException();
            this._Controller.Select(0, this.Document.Length);
        }

        /// <summary>
        /// 選択を解除する
        /// </summary>
        public void DeSelectAll()
        {
            if (this.Document.State == AsyncState.Loading)
                throw new InvalidOperationException();
            this._Controller.DeSelectAll();
        }

        /// <summary>
        /// 対応する座標を返します
        /// </summary>
        /// <param name="tp">テキストポイント</param>
        /// <returns>座標</returns>
        /// <remarks>テキストポイントがクライアント領域の原点より外にある場合、返される値は原点に丸められます</remarks>
        public Windows.Foundation.Point GetPostionFromTextPoint(TextPoint tp)
        {
            if (this.Document.State == AsyncState.Loading)
                throw new InvalidOperationException();
            if (this.Document.FireUpdateEvent == false)
                throw new InvalidOperationException("");
            return this.View.GetPostionFromTextPoint(tp);
        }

        /// <summary>
        /// 対応するテキストポイントを返します
        /// </summary>
        /// <param name="p">クライアント領域の原点を左上とする座標</param>
        /// <returns>テキストポイント</returns>
        public TextPoint GetTextPointFromPostion(Windows.Foundation.Point p)
        {
            if (this.Document.State == AsyncState.Loading)
                throw new InvalidOperationException();
            if (this.Document.FireUpdateEvent == false)
                throw new InvalidOperationException("");
            return this.View.GetTextPointFromPostion(p);
        }

        /// <summary>
        /// 行の高さを取得します
        /// </summary>
        /// <param name="row">レイアウト行</param>
        /// <returns>行の高さ</returns>
        public double GetLineHeight(int row)
        {
            if (this.Document.State == AsyncState.Loading)
                throw new InvalidOperationException();
            if (this.Document.FireUpdateEvent == false)
                throw new InvalidOperationException("");
            return this.View.LayoutLines.GetLayout(row).Height; ;
        }

        /// <summary>
        /// インデックスに対応する座標を得ます
        /// </summary>
        /// <param name="index">インデックス</param>
        /// <returns>座標を返す</returns>
        public Windows.Foundation.Point GetPostionFromIndex(int index)
        {
            if (this.Document.State == AsyncState.Loading)
                throw new InvalidOperationException();
            if (this.Document.FireUpdateEvent == false)
                throw new InvalidOperationException("");
            TextPoint tp = this.View.GetLayoutLineFromIndex(index);
            return this.View.GetPostionFromTextPoint(tp);
        }

        /// <summary>
        /// 座標からインデックスに変換します
        /// </summary>
        /// <param name="p">座標</param>
        /// <returns>インデックスを返す</returns>
        public int GetIndexFromPostion(Windows.Foundation.Point p)
        {
            if (this.Document.State == AsyncState.Loading)
                throw new InvalidOperationException();
            if (this.Document.FireUpdateEvent == false)
                throw new InvalidOperationException("");
            TextPoint tp = this.View.GetTextPointFromPostion(p);
            return this.View.GetIndexFromLayoutLine(tp);
        }

        /// <summary>
        /// 再描写する
        /// </summary>
        public void Refresh()
        {
            this.Refresh(this.View.PageBound);
        }

        /// <summary>
        /// レイアウト行をすべて破棄し、再度レイアウトを行う
        /// </summary>
        public void PerfomLayouts()
        {
            this.View.PerfomLayouts();
        }

        #region command
        void CopyCommand()
        {
            this.Copy();
        }

        void CutCommand()
        {
            this.Cut();
            this.Refresh();
        }

        async Task PasteCommand()
        {
            await this.PasteAsync();
            this.Refresh();
        }

        #endregion

        #region event
        /// <inheritdoc/>
        protected override Windows.UI.Xaml.Automation.Peers.AutomationPeer OnCreateAutomationPeer()
        {
            this.peer = new FooTextBoxAutomationPeer(this);
            return this.peer;
        }

        /// <inheritdoc/>
        protected override void OnApplyTemplate()
        {
            base.OnApplyTemplate();

            Grid grid = this.GetTemplateChild("PART_Grid") as Grid;
            if (grid != null)
            {
                Grid.SetRow(this.rectangle, 0);
                Grid.SetColumn(this.rectangle, 0);
                grid.Children.Add(this.rectangle);
            }

            this.horizontalScrollBar = this.GetTemplateChild("PART_HorizontalScrollBar") as ScrollBar;
            if (this.horizontalScrollBar != null)
            {
                this.horizontalScrollBar.SmallChange = 10;
                this.horizontalScrollBar.LargeChange = 100;
                this.horizontalScrollBar.Maximum = this.horizontalScrollBar.LargeChange + 1;
                this.horizontalScrollBar.Scroll += new ScrollEventHandler(horizontalScrollBar_Scroll);
            }
            this.verticalScrollBar = this.GetTemplateChild("PART_VerticalScrollBar") as ScrollBar;
            if (this.verticalScrollBar != null)
            {
                this.verticalScrollBar.SmallChange = 1;
                this.verticalScrollBar.LargeChange = 10;
                this.verticalScrollBar.Maximum = this.View.LayoutLines.Count;
                this.verticalScrollBar.Scroll += new ScrollEventHandler(verticalScrollBar_Scroll);
            }
        }

        /// <inheritdoc/>
        protected override void OnGotFocus(RoutedEventArgs e)
        {
            base.OnGotFocus(e);
            this.textStore.SetFocus();
            this.View.IsFocused = true;
            this.Refresh();
        }

        /// <inheritdoc/>
        protected override void OnLostFocus(RoutedEventArgs e)
        {
            base.OnLostFocus(e);
            this.View.IsFocused = false;
            this.Refresh();
        }

        /// <inheritdoc/>
        protected override async void OnKeyDown(KeyRoutedEventArgs e)
        {
            if (this.Document.State == AsyncState.Loading)
                return;
            bool isControlPressed = this.IsModiferKeyPressed(VirtualKey.Control);
            bool isShiftPressed = this.IsModiferKeyPressed(VirtualKey.Shift);
            bool isMovedCaret = false;
            switch (e.Key)
            {
                case VirtualKey.Up:
                    this._Controller.MoveCaretVertical(-1, isShiftPressed);
                    this.Refresh();
                    e.Handled = true;
                    isMovedCaret = true;
                    break;
                case VirtualKey.Down:
                    this._Controller.MoveCaretVertical(+1, isShiftPressed);
                    this.Refresh();
                    e.Handled = true;
                    isMovedCaret = true;
                    break;
                case VirtualKey.Left:
                    this._Controller.MoveCaretHorizontical(-1, isShiftPressed, isControlPressed);
                    this.Refresh();
                    e.Handled = true;
                    isMovedCaret = true;
                    break;
                case VirtualKey.Right:
                    this._Controller.MoveCaretHorizontical(1, isShiftPressed, isControlPressed);
                    this.Refresh();
                    e.Handled = true;
                    isMovedCaret = true;
                    break;
                case VirtualKey.PageUp:
                    this._Controller.Scroll(ScrollDirection.Up, this.View.LineCountOnScreen, isShiftPressed, true);
                    this.Refresh();
                    isMovedCaret = true;
                    break;
                case VirtualKey.PageDown:
                    this._Controller.Scroll(ScrollDirection.Down, this.View.LineCountOnScreen, isShiftPressed, true);
                    this.Refresh();
                    isMovedCaret = true;
                    break;
                case VirtualKey.Home:
                    if (isControlPressed)
                        this._Controller.JumpToHead(isShiftPressed);
                    else
                        this.Controller.JumpToLineHead(this.View.CaretPostion.row,isShiftPressed);
                    this.Refresh();
                    isMovedCaret = true;
                    break;
                case VirtualKey.End:
                    if (isControlPressed)
                        this._Controller.JumpToEnd(isShiftPressed);
                    else
                        this.Controller.JumpToLineEnd(this.View.CaretPostion.row,isShiftPressed);
                    this.Refresh();
                    isMovedCaret = true;
                    break;
                case VirtualKey.Tab:
                    if (!isControlPressed)
                    {
                        if (this._Controller.SelectionLength == 0)
                            this._Controller.DoInputChar('\t');
                        else if (isShiftPressed)
                            this._Controller.DownIndent();
                        else
                            this._Controller.UpIndent();
                        this.Refresh();
                        e.Handled = true;
                    }
                    break;
                case VirtualKey.Enter:
                    this._Controller.DoEnterAction();
                    this.Refresh();
                    e.Handled = true;
                    break;
                case VirtualKey.Insert:
                    if(this.View.InsertMode)
                        this.View.InsertMode = false;
                    else
                        this.View.InsertMode = true;
                    this.Refresh();
                    e.Handled = true;
                    break;
                case VirtualKey.A:
                    if (isControlPressed)
                    {
                        this.SelectAll();
                        this.Refresh();
                        e.Handled = true;
                    }
                    break;
                case VirtualKey.B:
                    if (isControlPressed)
                    {
                        if (this._Controller.RectSelection)
                            this._Controller.RectSelection = false;
                        else
                            this._Controller.RectSelection = true;
                        this.Refresh();
                        e.Handled = true;
                    }
                    break;
                case VirtualKey.C:
                    if (isControlPressed)
                    {
                        this.CopyCommand();
                        e.Handled = true;
                    }
                    break;
                case VirtualKey.X:
                    if (isControlPressed)
                    {
                        this.CutCommand();
                        e.Handled = true;
                    }
                    break;
                case VirtualKey.V:
                    if (isControlPressed)
                    {
                        await this.PasteCommand();
                        e.Handled = true;
                    }
                    break;
                case VirtualKey.Y:
                    if (isControlPressed)
                    {
                        this.Document.UndoManager.redo();
                        this.Refresh();
                        e.Handled = true;
                    }
                    break;
                case VirtualKey.Z:
                    if (isControlPressed)
                    {
                        this.Document.UndoManager.undo();
                        this.Refresh();
                        e.Handled = true;
                    }
                    break;
                case VirtualKey.Back:
                    this._Controller.DoBackSpaceAction();
                    this.Refresh();
                    e.Handled = true;
                    break;
                case VirtualKey.Delete:
                    this._Controller.DoDeleteAction();
                    this.Refresh();
                    e.Handled = true;
                    break;
            }
            if (isMovedCaret && this.peer != null)
                this.peer.OnNotifyCaretChanged();
            base.OnKeyDown(e);
        }

        /// <inheritdoc/>
        protected override void OnPointerPressed(PointerRoutedEventArgs e)
        {
            this.CapturePointer(e.Pointer);
            this.gestureRecongnizer.ProcessDownEvent(e.GetCurrentPoint(this));
            e.Handled = true;
        }

        /// <inheritdoc/>
        protected override void OnPointerMoved(PointerRoutedEventArgs e)
        {
            this.gestureRecongnizer.ProcessMoveEvents(e.GetIntermediatePoints(this));
            e.Handled = true;

            if (e.Pointer.PointerDeviceType == PointerDeviceType.Mouse && this.Document.State != AsyncState.Loading)
            {
                Point p = e.GetCurrentPoint(this).Position;
                if (this.View.HitTextArea(p.X, p.Y))
                {
                    TextPoint tp = this.View.GetTextPointFromPostion(p);
                    if (this._Controller.IsMarker(tp, HilightType.Url))
                        Window.Current.CoreWindow.PointerCursor = new CoreCursor(CoreCursorType.Hand, 101);
                    else
                        Window.Current.CoreWindow.PointerCursor = new CoreCursor(CoreCursorType.IBeam, 101);
                }
                else
                {
                    Window.Current.CoreWindow.PointerCursor = new CoreCursor(CoreCursorType.Arrow, 101);
                }
            }
        }

        /// <inheritdoc/>
        protected override void OnPointerReleased(PointerRoutedEventArgs e)
        {
            this.gestureRecongnizer.ProcessUpEvent(e.GetCurrentPoint(this));
            this.ReleasePointerCapture(e.Pointer);
            e.Handled = true;
        }

        /// <inheritdoc/>
        protected override void OnPointerCanceled(PointerRoutedEventArgs e)
        {
            this.gestureRecongnizer.CompleteGesture();
            this.ReleasePointerCapture(e.Pointer);
            e.Handled = true;
        }

        /// <inheritdoc/>
        protected override void OnPointerCaptureLost(PointerRoutedEventArgs e)
        {
            this.gestureRecongnizer.CompleteGesture();
            this.ReleasePointerCapture(e.Pointer);
            e.Handled = true;
        }

        /// <inheritdoc/>
        protected override void OnPointerWheelChanged(PointerRoutedEventArgs e)
        {
            bool shift = (e.KeyModifiers & Windows.System.VirtualKeyModifiers.Shift) ==
                Windows.System.VirtualKeyModifiers.Shift;
            bool ctrl = (e.KeyModifiers & Windows.System.VirtualKeyModifiers.Control) ==
                Windows.System.VirtualKeyModifiers.Control;
            this.gestureRecongnizer.ProcessMouseWheelEvent(e.GetCurrentPoint(this), shift, ctrl);
            e.Handled = true;
        }

        void CoreWindow_CharacterReceived(CoreWindow sender, CharacterReceivedEventArgs args)
        {
            if (this.FocusState == FocusState.Unfocused || this.Document.State == AsyncState.Loading)
                return;
            if (args.KeyCode >= 00 && args.KeyCode <= 0x1f)
                return;
            this._Controller.DoInputString(Char.ConvertFromUtf32((int)args.KeyCode));
            this.Refresh();
        }

        bool textStore_IsReadOnly()
        {
            return false;
        }

        bool textStore_IsLoading()
        {
            return this.Document.State == AsyncState.Loading;
        }

        void textStore_CompositionEnded()
        {
            TextStoreHelper.EndCompostion(this.Document);
            this.Refresh();
        }

        void textStore_CompositionUpdated(int start, int end)
        {
            if (TextStoreHelper.ScrollToCompstionUpdated(this.textStore, this.View, start, end))
                this.Refresh();
        }
        bool textStore_CompositionStarted()
        {
            return TextStoreHelper.StartCompstion(this.Document);
        }

        string _textStore_GetString(int start, int length)
        {
            return this.Document.ToString(start, length);
        }

        void _textStore_GetStringExtent(
            int i_startIndex,
            int i_endIndex,
            out POINT o_topLeft,
            out POINT o_bottomRight
        )
        {
            Point startPos, endPos;
            TextStoreHelper.GetStringExtent(this.Document, this.View, i_startIndex, i_endIndex, out startPos, out endPos);

            var gt = this.TransformToVisual(Window.Current.Content);
            startPos = gt.TransformPoint(startPos);
            endPos = gt.TransformPoint(endPos);

            o_topLeft = new POINT((int)startPos.X, (int)startPos.Y);
            o_bottomRight = new POINT((int)endPos.X, (int)endPos.Y);
        }

        void _textStore_GetScreenExtent(out POINT o_topLeft, out POINT o_bottomRight)
        {
            var pointTopLeft = new Point(0, 0);
            var pointBottomRight = new Point(this.RenderSize.Width, this.RenderSize.Height);

            var gt = this.TransformToVisual(Window.Current.Content);
            pointTopLeft = gt.TransformPoint(pointTopLeft);
            pointBottomRight = gt.TransformPoint(pointBottomRight);

            o_topLeft = new POINT((int)pointTopLeft.X, (int)pointTopLeft.Y);
            o_bottomRight = new POINT((int)pointBottomRight.X, (int)pointBottomRight.Y);
        }

        void _textStore_GetSelectionIndex(out int o_startIndex, out int o_endIndex)
        {
            TextStoreHelper.GetSelection(this._Controller, this.View.Selections, out o_startIndex, out o_endIndex);
        }

        void _textStore_SetSelectionIndex(int i_startIndex, int i_endIndex)
        {
            if (this.Document.State == AsyncState.Loading)
                return;
            TextStoreHelper.SetSelectionIndex(this._Controller, this.View, i_startIndex, i_endIndex);
            this.Refresh();
        }

        void _textStore_InsertAtSelection(string i_value)
        {
            TextStoreHelper.InsertTextAtSelection(this._Controller, i_value);
            this.Refresh();
        }

        void Controller_SelectionChanged(object sender, EventArgs e)
        {
            //こうしないと選択できなくなってしまう
            this.nowCaretMove = true;
            SetValue(SelectionProperty, new TextRange(this._Controller.SelectionStart, this._Controller.SelectionLength));
            SetValue(CaretPostionPropertyKey, this.View.CaretPostion);
            this.nowCaretMove = false;
            if (this.textStore.IsLocked() == false)
                this.textStore.NotifySelectionChanged();
        }

        Gripper FirstGripper, SecondGripper;
        bool HittedCaret;
        Gripper hittedGripper;
        void gestureRecongnizer_ManipulationStarted(GestureRecognizer sender, ManipulationStartedEventArgs e)
        {
            this.HittedCaret = false;
            this.hittedGripper = null;

            Point p = e.Position;
            TextPoint tp = this.View.GetTextPointFromPostion(p);
            if (tp == this.View.CaretPostion)
            {
                HittedCaret = true;
            }
            
            if (this.FirstGripper.IsHit(p))
            {
                hittedGripper = this.FirstGripper;
                HittedCaret = true;
            }
            
            else if (this.SecondGripper.IsHit(p))
            {
                hittedGripper = this.SecondGripper;
                HittedCaret = true;
            }
        }

        void gestureRecongnizer_ManipulationUpdated(GestureRecognizer sender, ManipulationUpdatedEventArgs e)
        {
            if (this.Document.State == AsyncState.Loading)
                return;

            if (HittedCaret)
            {
                Point p;
                if (this.hittedGripper == null)
                     p = e.Position;
                else
                    p = this.hittedGripper.AdjustPoint(e.Position);

                if (hittedGripper != null)
                {
                    TextPoint tp = this.View.GetTextPointFromPostion(p);
                    if (this._Controller.IsReverseSelect())
                    {
                        if (Object.ReferenceEquals(hittedGripper,this.SecondGripper))
                            this._Controller.MoveSelectBefore(tp);
                        else
                            this._Controller.MoveCaretAndSelect(tp);
                    }
                    else
                    {
                        if (Object.ReferenceEquals(hittedGripper,this.FirstGripper))
                            this._Controller.MoveSelectBefore(tp);
                        else
                            this._Controller.MoveCaretAndSelect(tp);
                    }
                    if (this.peer != null)
                        this.peer.OnNotifyCaretChanged();
                    this.Refresh();
                    return;
                }
            }
            if (e.Delta.Scale < 1)
            {
                double newSize = this.Render.FontSize - 1;
                if (newSize < 1)
                    newSize = 1;
                this.Render.FontSize = newSize;
                this.Refresh();
                SetValue(MagnificationPowerPropertyKey, this.Render.FontSize / this.FontSize);
                return;
            }
            
            if (e.Delta.Scale > 1)
            {
                double newSize = this.Render.FontSize + 1;
                if (newSize > 72)
                    newSize = 72;
                this.Render.FontSize = newSize;
                this.Refresh();
                SetValue(MagnificationPowerPropertyKey, this.Render.FontSize / this.FontSize);
                return;
            }

            Point translation = e.Delta.Translation;

            int scrollCount = (int)Math.Abs(translation.Y);
            if (scrollCount > 0)
            {
                if (scrollCount > this.View.LineCountOnScreen)  //ホイール対策
                    scrollCount = this.View.LineCountOnScreen;
                if (translation.Y > 0)
                    this._Controller.Scroll(ScrollDirection.Up, scrollCount, false, false);
                else
                    this._Controller.Scroll(ScrollDirection.Down, scrollCount, false, false);
                this.FirstGripper.Enabled = false;
                this.SecondGripper.Enabled = false;
                this.Refresh();
                return;
            }

            int deltax = (int)Math.Abs(translation.X);
            if (deltax != 0)
            {
                if (translation.X < 0)
                    this._Controller.Scroll(ScrollDirection.Left, deltax, false, false);
                else
                    this._Controller.Scroll(ScrollDirection.Right, deltax, false, false);
                this.FirstGripper.Enabled = false;
                this.SecondGripper.Enabled = false;
                this.Refresh();
            }
        }

        void gestureRecongnizer_ManipulationCompleted(GestureRecognizer sender, ManipulationCompletedEventArgs e)
        {
        }

        async void gestureRecongnizer_RightTapped(GestureRecognizer sender, RightTappedEventArgs e)
        {
            ResourceMap map = ResourceManager.Current.MainResourceMap.GetSubtree("FooEditEngine.Metro/Resources");
            if (this.View.HitTextArea(e.Position.X, e.Position.Y))
            {
                FooContextMenuEventArgs args = new FooContextMenuEventArgs(e.Position);
                if (this.ContextMenuOpening != null)
                    this.ContextMenuOpening(this, args);
                if (!args.Handled)
                {
                    PopupMenu ContextMenu = new PopupMenu();
                    ContextMenu.Commands.Add(new UICommand(map.GetValue("CopyMenuName").ValueAsString, (command) =>
                    {
                        this.CopyCommand();
                    }));
                    ContextMenu.Commands.Add(new UICommand(map.GetValue("CutMenuName").ValueAsString, (command) =>
                    {
                        this.CutCommand();
                    }));
                    ContextMenu.Commands.Add(new UICommand(map.GetValue("PasteMenuName").ValueAsString, async (command) =>
                    {
                        await this.PasteCommand();
                    }));
                    if (this._Controller.RectSelection)
                    {
                        ContextMenu.Commands.Add(new UICommand(map.GetValue("LineSelectMenuName").ValueAsString, (command) =>
                        {
                            this._Controller.RectSelection = false;
                        }));
                    }
                    else
                    {
                        ContextMenu.Commands.Add(new UICommand(map.GetValue("RectSelectMenuName").ValueAsString, (command) =>
                        {
                            this._Controller.RectSelection = true;
                        }));
                    }
                    await ContextMenu.ShowAsync(Util.GetScreentPoint(e.Position,this));
                }
            }
        }

        void gestureRecongnizer_Tapped(GestureRecognizer sender, TappedEventArgs e)
        {
            bool touched = e.PointerDeviceType == PointerDeviceType.Touch;
            this.FirstGripper.Enabled = touched;
            this.SecondGripper.Enabled = touched;
            this.JumpCaret(e.Position);
        }

        void JumpCaret(Point p)
        {
            if (this.Document.State == AsyncState.Loading)
                return;
            
            TextPoint tp = this.View.GetTextPointFromPostion(p);
            if (tp == TextPoint.Null)
                return;

            int index = this.View.LayoutLines.GetIndexFromTextPoint(tp);

            FoldingItem foldingData = this.View.HitFoldingData(p.X, tp.row);
            if (foldingData != null)
            {
                if (foldingData.Expand)
                    this.View.LayoutLines.FoldingCollection.Collapse(foldingData);
                else
                    this.View.LayoutLines.FoldingCollection.Expand(foldingData);
                this._Controller.JumpCaret(foldingData.Start, false);
            }
            else
            {
                this._Controller.JumpCaret(tp.row, tp.col, false);
            }
            if (this.peer != null)
                this.peer.OnNotifyCaretChanged();
            this.Focus(FocusState.Pointer);
            this.Refresh();
        }

        void gestureRecongnizer_Dragging(GestureRecognizer sender, DraggingEventArgs e)
        {
            if (this.Document.State == AsyncState.Loading)
                return;
            Point p = e.Position;
            if (this.View.HitTextArea(p.X, p.Y))
            {
                TextPoint tp = this.View.GetTextPointFromPostion(p);
                this._Controller.MoveCaretAndSelect(tp);
                if (this.peer != null)
                    this.peer.OnNotifyCaretChanged();
                this.Refresh();
            }
        }

        bool IsModiferKeyPressed(VirtualKey key)
        {
            CoreVirtualKeyStates state = Window.Current.CoreWindow.GetKeyState(key);
            return (state & CoreVirtualKeyStates.Down) == CoreVirtualKeyStates.Down;
        }
        void Refresh(Rectangle updateRect)
        {
            if (this.rectangle.ActualWidth == 0 || this.rectangle.ActualHeight == 0/* || !this.Render.IsCanDraw()*/)
                return;

            this.Render.BeginDraw();
            if (this.Document.State != AsyncState.Loading)
                this.View.Draw(updateRect);
            else
                this.Render.FillBackground(updateRect);
            this.FirstGripper.Draw();
            this.SecondGripper.Draw();
            this.Render.EndDraw();
        }


        bool Resize(double width, double height)
        {
            if (width == 0 || height == 0)
                throw new ArgumentOutOfRangeException();
            if (this.Document.State != AsyncState.Loading && this.Render.Resize(this.rectangle, width, height))
            {
                this.View.PageBound = new Rectangle(0, 0, width, height);

                if (this.horizontalScrollBar != null)
                {
                    this.horizontalScrollBar.LargeChange = this.View.PageBound.Width;
                    this.horizontalScrollBar.Maximum = this.View.LongestWidth + this.horizontalScrollBar.LargeChange + 1;
                }
                if (this.verticalScrollBar != null)
                {
                    this.verticalScrollBar.LargeChange = this.View.LineCountOnScreen;
                    this.verticalScrollBar.Maximum = this.View.LayoutLines.Count + this.verticalScrollBar.LargeChange + 1;
                }
                return true;
            }
            return false;
        }

        void View_SrcChanged(object sender, EventArgs e)
        {
            if (this.horizontalScrollBar == null || this.verticalScrollBar == null)
                return;
            EditView view = this.View;
            if (view.Src.Row > this.verticalScrollBar.Maximum)
                this.verticalScrollBar.Maximum = view.Src.Row + view.LineCountOnScreen + 1;
            double absoulteX = Math.Abs(view.Src.X);
            if (absoulteX > this.horizontalScrollBar.Maximum)
                this.horizontalScrollBar.Maximum = absoulteX + view.PageBound.Width + 1;
            if (view.Src.Row != this.verticalScrollBar.Value)
                this.verticalScrollBar.Value = view.Src.Row;
            if (view.Src.X != this.horizontalScrollBar.Value)
                this.horizontalScrollBar.Value = Math.Abs(view.Src.X);
        }

        void FooTextBox_SizeChanged(object sender, SizeChangedEventArgs e)
        {
            if (this.Document.State == AsyncState.Loading || this.Resize(this.rectangle.ActualWidth, this.rectangle.ActualHeight))
            {
                this.Refresh();
                return;
            }
        }

        void horizontalScrollBar_Scroll(object sender, ScrollEventArgs e)
        {
            if (this.Document.State == AsyncState.Loading)
                return;
            if (this.horizontalScrollBar == null)
                return;
            double toX;
            if (this.FlowDirection == FlowDirection.LeftToRight)
                toX = this.horizontalScrollBar.Value;
            else
                toX = -this.horizontalScrollBar.Value;
            this._Controller.Scroll(toX, this.View.Src.Row, false, false);
            this.FirstGripper.Enabled = false;
            this.SecondGripper.Enabled = false;
            this.Refresh();
        }

        void verticalScrollBar_Scroll(object sender, ScrollEventArgs e)
        {
            if (this.Document.State == AsyncState.Loading)
                return;
            if (this.verticalScrollBar == null)
                return;
            int newRow = (int)this.verticalScrollBar.Value;
            if (newRow >= this.View.LayoutLines.Count)
                return;
            this._Controller.Scroll(this.View.Src.X, newRow, false, false);
            this.FirstGripper.Enabled = false;
            this.SecondGripper.Enabled = false;
            this.Refresh();
        }

        void Document_Update(object sender, DocumentUpdateEventArgs e)
        {
            if (this.textStore.IsLocked() || this.Document.State == AsyncState.Loading || e.type == UpdateType.Clear)
                return;
            if (e.removeLength == 0)
                this.textStore.NotifyTextChanged(e.startIndex, e.startIndex, e.startIndex + e.insertLength);
            else
                this.textStore.NotifyTextChanged(e.startIndex, e.startIndex + 1, e.startIndex + e.insertLength - e.removeLength);
        }

        void Document_Progress(object sender, ProgressEventArgs e)
        {
            if (this.Document.State == AsyncState.Loading)
                return;
            switch (e.state)
            {
                case ProgressState.Start:
                    this.textStore.NotifyTextChanged(0, this.Document.Length, 0);
                    break;
                case ProgressState.Complete:
                    this.textStore.NotifyTextChanged(0, 0, this.Document.Length);
                    if (this.verticalScrollBar != null)
                        this.verticalScrollBar.Maximum = this.View.LayoutLines.Count;
                    this.View.CalculateLineCountOnScreen();
                    break;
            }
        }

        void FooTextBox_Loaded(object sender, RoutedEventArgs e)
        {
            this.Focus(FocusState.Programmatic);
        }

        /// <inheritdoc/>
        public static void OnPropertyChanged(object sender, DependencyPropertyChangedEventArgs e)
        {
            FooTextBox source = (FooTextBox)sender;
            if (e.Property.Equals(SelectionProperty) && !source.nowCaretMove)
                source._Controller.Select(source.Selection.Index,source.Selection.Length);
            if (e.Property.Equals(CaretPostionPropertyKey) && !source.nowCaretMove)
                source.JumpCaret(source.CaretPostion.row, source.CaretPostion.col);
            if (e.Property.Equals(InsertModeProperty))
                source.View.InsertMode = source.InsertMode;
            if (e.Property.Equals(TabCharsProperty))
                source.View.TabStops = source.TabChars;
            if (e.Property.Equals(RectSelectModeProperty))
                source._Controller.RectSelection = source.RectSelectMode;
            if (e.Property.Equals(DrawCaretProperty))
                source.View.HideCaret = !source.DrawCaret;
            if (e.Property.Equals(DrawCaretLineProperty))
                source.View.HideLineMarker = !source.DrawCaretLine;
            if (e.Property.Equals(DrawLineNumberProperty))
            {
                source.View.DrawLineNumber = source.DrawLineNumber;
                source._Controller.JumpCaret(source.View.CaretPostion.row, source.View.CaretPostion.col);
            }
            if(e.Property.Equals(MagnificationPowerPropertyKey))
                source.Render.FontSize = source.FontSize * source.MagnificationPower;
            if (e.Property.Equals(FontFamilyProperty))
                source.Render.FontFamily = source.FontFamily;
            if (e.Property.Equals(FontStyleProperty))
                source.Render.FontStyle = source.FontStyle;
            if (e.Property.Equals(FontWeightProperty))
                source.Render.FontWeigth = source.FontWeight;
            if (e.Property.Equals(FontSizeProperty))
                source.Render.FontSize = source.FontSize;
            if (e.Property.Equals(ForegroundProperty))
                source.Render.Foreground = source.Foreground;
            if (e.Property.Equals(BackgroundProperty))
                source.Render.Background = source.Background;
            if (e.Property.Equals(ControlCharProperty))
                source.Render.ControlChar = source.ControlChar;
            if (e.Property.Equals(HilightProperty))
                source.Render.Hilight = source.Hilight;
            if (e.Property.Equals(Keyword1Property))
                source.Render.Keyword1 = source.Keyword1;
            if (e.Property.Equals(Keyword2Property))
                source.Render.Keyword2 = source.Keyword2;
            if (e.Property.Equals(CommentProperty))
                source.Render.Comment = source.Comment;
            if (e.Property.Equals(LiteralProperty))
                source.Render.Literal = source.Literal;
            if (e.Property.Equals(URLProperty))
                source.Render.Url = source.URL;
            if (e.Property.Equals(InsertCaretProperty))
                source.Render.InsertCaret = source.InsertCaret;
            if (e.Property.Equals(OverwriteCaretProperty))
                source.Render.OverwriteCaret = source.OverwriteCaret;
            if (e.Property.Equals(LineMarkerProperty))
                source.Render.LineMarker = source.LineMarker;
            if (e.Property.Equals(MarkURLProperty))
                source.View.UrlMark = source.MarkURL;
            if (e.Property.Equals(ShowFullSpaceProperty))
                source.Render.ShowFullSpace = source.ShowFullSpace;
            if (e.Property.Equals(ShowHalfSpaceProperty))
                source.Render.ShowHalfSpace = source.ShowHalfSpace;
            if (e.Property.Equals(ShowTabProperty))
                source.Render.ShowTab = source.ShowTab;
            if (e.Property.Equals(ShowLineBreakProperty))
                source.Render.ShowLineBreak = source.ShowLineBreak;
            if (e.Property.Equals(LineBreakProperty))
                source.View.LineBreak = source.LineBreakMethod;
            if (e.Property.Equals(LineBreakCharCountProperty))
                source.View.LineBreakCharCount = source.LineBreakCharCount;
            if (e.Property.Equals(FlowDirectionProperty))
            {
                source.Render.RightToLeft = source.FlowDirection == Windows.UI.Xaml.FlowDirection.RightToLeft;
                if(source.horizontalScrollBar != null)
                    source.horizontalScrollBar.FlowDirection = source.FlowDirection;
            }
            if (e.Property.Equals(DrawRulerProperty))
            {
                source.View.HideRuler = !source.DrawRuler;
                source._Controller.JumpCaret(source.View.CaretPostion.row, source.View.CaretPostion.col);
            }            
        }
        #endregion

        #region event
        
        /// <summary>
        /// コンテキストメニューが表示されるときに呼び出されます
        /// </summary>
        public event EventHandler<FooContextMenuEventArgs> ContextMenuOpening;

        #endregion

        #region property

        internal Controller Controller
        {
            get
            {
                return this._Controller;
            }
        }

        /// <summary>
        /// 文字列の描写に使用されるアンチエイリアシング モードを表します
        /// </summary>
        public TextAntialiasMode TextAntialiasMode
        {
            get
            {
                return this.Render.TextAntialiasMode;
            }
            set
            {
                this.Render.TextAntialiasMode = value;
            }
        }

        /// <summary>
        /// シンタックスハイライターを表す
        /// </summary>
        public IHilighter Hilighter
        {
            get
            {
                return this.View.Hilighter;
            }
            set
            {
                this.View.Hilighter = value;
                this.View.LayoutLines.ClearLayoutCache();
            }
        }

        /// <summary>
        /// フォールティングを作成するインターフェイスを表す
        /// </summary>
        public IFoldingStrategy FoldingStrategy
        {
            get
            {
                return this.View.LayoutLines.FoldingStrategy;
            }
            set
            {
                this.View.LayoutLines.FoldingStrategy = value;
                if (value == null)
                    this.View.LayoutLines.FoldingCollection.Clear();
            }
        }

        /// <summary>
        /// マーカーパターンセットを表す
        /// </summary>
        public MarkerPatternSet MarkerPatternSet
        {
            get
            {
                return this.View.MarkerPatternSet;
            }
        }

        /// <summary>
        /// ドキュメントを表す
        /// </summary>
        public Document Document
        {
            get;
            private set;
        }

        /// <summary>
        /// レイアウト行を表す
        /// </summary>
        public LineToIndexTable LayoutLineCollection
        {
            get { return this.View.LayoutLines; }
        }

        /// <summary>
        /// 選択中の文字列を表す
        /// </summary>
        public string SelectedText
        {
            get
            {
                return this._Controller.SelectedText;
            }
            set
            {
                int oldLength = this.Document.Length;
                this._Controller.SelectedText = value;
            }
        }

        /// <summary>
        /// 選択範囲を表す
        /// </summary>
        /// <remarks>
        /// Lengthが0の場合はキャレット位置を表します。
        /// 矩形選択モードの場合、選択範囲の文字数ではなく、開始位置から終了位置までの長さとなります
        /// </remarks>
        public TextRange Selection
        {
            get { return (TextRange)GetValue(SelectionProperty); }
            set { SetValue(SelectionProperty, value); }
        }

        /// <summary>
        /// Selectionの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty SelectionProperty =
            DependencyProperty.Register("Selection", typeof(TextRange), typeof(FooTextBox), new PropertyMetadata(0));

        /// <summary>
        /// 拡大率を表す
        /// </summary>
        public double MagnificationPower
        {
            get { return (double)GetValue(MagnificationPowerPropertyKey); }
            set { SetValue(MagnificationPowerPropertyKey, value); }
        }

        /// <summary>
        /// 拡大率を表す依存プロパティ
        /// </summary>
        public static readonly DependencyProperty MagnificationPowerPropertyKey =
            DependencyProperty.Register("MagnificationPower", typeof(double), typeof(FooTextBox), new PropertyMetadata(1.0, OnPropertyChanged));

        /// <summary>
        /// キャレット位置を表す
        /// </summary>
        public TextPoint CaretPostion
        {
            get { return (TextPoint)GetValue(CaretPostionPropertyKey); }
            set { SetValue(CaretPostionPropertyKey, value); }
        }

        static readonly DependencyProperty CaretPostionPropertyKey =
            DependencyProperty.Register("CaretPostion", typeof(TextPoint), typeof(FooTextBox), new PropertyMetadata(new TextPoint(), OnPropertyChanged));

        /// <summary>
        /// レタリング方向を表す
        /// </summary>
        public new FlowDirection FlowDirection
        {
            get { return (FlowDirection)GetValue(FlowDirectionProperty); }
            set { SetValue(FlowDirectionProperty, value); }
        }

        /// <summary>
        /// レタリング方向を表す。これは依存プロパティです
        /// </summary>
        public new static readonly DependencyProperty FlowDirectionProperty =
            DependencyProperty.Register("FlowDirection", typeof(FlowDirection), typeof(FooTextBox), new PropertyMetadata(FlowDirection.LeftToRight,OnPropertyChanged));

        /// <summary>
        /// フォントファミリーを表す
        /// </summary>
        public new FontFamily FontFamily
        {
            get { return (FontFamily)GetValue(FontFamilyProperty); }
            set { SetValue(FontFamilyProperty, value); }
        }

        /// <summary>
        /// FontFamilyの依存プロパティを表す
        /// </summary>
        public new static readonly DependencyProperty FontFamilyProperty =
            DependencyProperty.Register("FontFamily", typeof(FontFamily), typeof(FooTextBox), new PropertyMetadata(new FontFamily("Cambria"), OnPropertyChanged));

        /// <summary>
        /// フォントサイズを表す
        /// </summary>
        public new double FontSize
        {
            get { return (double)GetValue(FontSizeProperty); }
            set { SetValue(FontSizeProperty, value); }
        }

        /// <summary>
        /// FontSizeの依存プロパティを表す
        /// </summary>
        public new static readonly DependencyProperty FontSizeProperty =
            DependencyProperty.Register("FontSize", typeof(double), typeof(FooTextBox), new PropertyMetadata(12.0,OnPropertyChanged));

        /// <summary>
        /// フォントスタイルを表す
        /// </summary>
        public new FontStyle FontStyle
        {
            get { return (FontStyle)GetValue(FontStyleProperty); }
            set { SetValue(FontStyleProperty, value); }
        }

        /// <summary>
        /// FontStyleの依存プロパティを表す
        /// </summary>
        public new static readonly DependencyProperty FontStyleProperty =
            DependencyProperty.Register("FontStyle", typeof(FontStyle), typeof(FooTextBox), new PropertyMetadata(FontStyle.Normal,OnPropertyChanged));

        /// <summary>
        /// フォントの幅を表す
        /// </summary>
        public new FontWeight FontWeight
        {
            get { return (FontWeight)GetValue(FontWeightProperty); }
            set { SetValue(FontWeightProperty, value); }
        }

        /// <summary>
        /// FontWeigthの依存プロパティを表す
        /// </summary>
        public new static readonly DependencyProperty FontWeightProperty =
            DependencyProperty.Register("FontWeigth", typeof(FontWeight), typeof(FooTextBox), new PropertyMetadata(FontWeights.Normal,OnPropertyChanged));

        /// <summary>
        /// デフォルトの文字色を表す。これは依存プロパティです
        /// </summary>
        public new Windows.UI.Color Foreground
        {
            get { return (Windows.UI.Color)GetValue(ForegroundProperty); }
            set { SetValue(ForegroundProperty, value); }
        }

        /// <summary>
        /// Foregroundの依存プロパティを表す
        /// </summary>
        public new static readonly DependencyProperty ForegroundProperty =
            DependencyProperty.Register("Foreground", typeof(Windows.UI.Color), typeof(FooTextBox), new PropertyMetadata(Colors.Black, OnPropertyChanged));

        /// <summary>
        /// 背景色を表す。これは依存プロパティです
        /// </summary>
        public new Windows.UI.Color Background
        {
            get { return (Windows.UI.Color)GetValue(BackgroundProperty); }
            set { SetValue(BackgroundProperty, value); }
        }

        /// <summary>
        /// Backgroundの依存プロパティを表す
        /// </summary>
        public new static readonly DependencyProperty BackgroundProperty =
            DependencyProperty.Register("Background", typeof(Windows.UI.Color), typeof(FooTextBox), new PropertyMetadata(Colors.White, OnPropertyChanged));

        /// <summary>
        /// コントロールコードの文字色を表す。これは依存プロパティです
        /// </summary>
        public Windows.UI.Color ControlChar
        {
            get { return (Windows.UI.Color)GetValue(ControlCharProperty); }
            set { SetValue(ControlCharProperty, value); }
        }

        /// <summary>
        /// ControlCharの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty ControlCharProperty =
            DependencyProperty.Register("ControlChar", typeof(Windows.UI.Color), typeof(FooTextBox), new PropertyMetadata(Colors.Gray, OnPropertyChanged));

        /// <summary>
        /// 選択時の背景色を表す。これは依存プロパティです
        /// </summary>
        public Windows.UI.Color Hilight
        {
            get { return (Windows.UI.Color)GetValue(HilightProperty); }
            set { SetValue(HilightProperty, value); }
        }

        /// <summary>
        /// Hilightの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty HilightProperty =
            DependencyProperty.Register("Hilight", typeof(Windows.UI.Color), typeof(FooTextBox), new PropertyMetadata(Colors.DeepSkyBlue, OnPropertyChanged));

        /// <summary>
        /// キーワード１の文字色を表す。これは依存プロパティです
        /// </summary>
        public Windows.UI.Color Keyword1
        {
            get { return (Windows.UI.Color)GetValue(Keyword1Property); }
            set { SetValue(Keyword1Property, value); }
        }

        /// <summary>
        /// Keyword1の依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty Keyword1Property =
            DependencyProperty.Register("Keyword1", typeof(Windows.UI.Color), typeof(FooTextBox), new PropertyMetadata(Colors.Blue, OnPropertyChanged));

        /// <summary>
        /// キーワード2の文字色を表す。これは依存プロパティです
        /// </summary>
        public Windows.UI.Color Keyword2
        {
            get { return (Windows.UI.Color)GetValue(Keyword2Property); }
            set { SetValue(Keyword2Property, value); }
        }

        /// <summary>
        /// Keyword2の依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty Keyword2Property =
            DependencyProperty.Register("Keyword2", typeof(Windows.UI.Color), typeof(FooTextBox), new PropertyMetadata(Colors.DarkCyan, OnPropertyChanged));

        /// <summary>
        /// コメントの文字色を表す。これは依存プロパティです
        /// </summary>
        public Windows.UI.Color Comment
        {
            get { return (Windows.UI.Color)GetValue(CommentProperty); }
            set { SetValue(CommentProperty, value); }
        }

        /// <summary>
        /// Commentの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty CommentProperty =
            DependencyProperty.Register("Comment", typeof(Windows.UI.Color), typeof(FooTextBox), new PropertyMetadata(Colors.Green, OnPropertyChanged));

        /// <summary>
        /// 文字リテラルの文字色を表す。これは依存プロパティです
        /// </summary>
        public Windows.UI.Color Literal
        {
            get { return (Windows.UI.Color)GetValue(LiteralProperty); }
            set { SetValue(LiteralProperty, value); }
        }

        /// <summary>
        /// Literalの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty LiteralProperty =
            DependencyProperty.Register("Literal", typeof(Windows.UI.Color), typeof(FooTextBox), new PropertyMetadata(Colors.Brown, OnPropertyChanged));

        /// <summary>
        /// URLの文字色を表す。これは依存プロパティです
        /// </summary>
        public Windows.UI.Color URL
        {
            get { return (Windows.UI.Color)GetValue(URLProperty); }
            set { SetValue(URLProperty, value); }
        }

        /// <summary>
        /// URLの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty URLProperty =
            DependencyProperty.Register("URL", typeof(Windows.UI.Color), typeof(FooTextBox), new PropertyMetadata(Colors.Blue, OnPropertyChanged));


        /// <summary>
        /// ラインマーカーの色を表す
        /// </summary>
        public Windows.UI.Color LineMarker
        {
            get { return (Windows.UI.Color)GetValue(LineMarkerProperty); }
            set { SetValue(LineMarkerProperty, value); }
        }

        /// <summary>
        /// LineMarkerの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty LineMarkerProperty =
            DependencyProperty.Register("LineMarker", typeof(Windows.UI.Color), typeof(FooTextBox), new PropertyMetadata(Colors.Gray, OnPropertyChanged));

        /// <summary>
        /// 挿入モード時のキャレットの色を表す
        /// </summary>
        public Windows.UI.Color InsertCaret
        {
            get { return (Windows.UI.Color)GetValue(InsertCaretProperty); }
            set { SetValue(InsertCaretProperty, value); }
        }

        /// <summary>
        /// InsertCaretの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty InsertCaretProperty =
            DependencyProperty.Register("InsertCaret", typeof(Windows.UI.Color), typeof(FooTextBox), new PropertyMetadata(Colors.Black, OnPropertyChanged));

        /// <summary>
        /// 上書きモード時のキャレット職を表す
        /// </summary>
        public Windows.UI.Color OverwriteCaret
        {
            get { return (Windows.UI.Color)GetValue(OverwriteCaretProperty); }
            set { SetValue(OverwriteCaretProperty, value); }
        }

        /// <summary>
        /// OverwriteCaretの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty OverwriteCaretProperty =
            DependencyProperty.Register("OverwriteCaret", typeof(Windows.UI.Color), typeof(FooTextBox), new PropertyMetadata(Colors.Black, OnPropertyChanged));

        /// <summary>
        /// 挿入モードなら真を返し、そうでないなら、偽を返す。これは依存プロパティです
        /// </summary>
        public bool InsertMode
        {
            get { return (bool)GetValue(InsertModeProperty); }
            set { SetValue(InsertModeProperty, value); }
        }

        /// <summary>
        /// InsertModeの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty InsertModeProperty =
            DependencyProperty.Register("InsertMode",
            typeof(bool),
            typeof(FooTextBox),
            new PropertyMetadata(true, OnPropertyChanged));

        /// <summary>
        /// タブの文字数を表す。これは依存プロパティです
        /// </summary>
        public int TabChars
        {
            get { return (int)GetValue(TabCharsProperty); }
            set { SetValue(TabCharsProperty, value); }
        }

        /// <summary>
        /// TabCharsの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty TabCharsProperty =
            DependencyProperty.Register("TabChars",
            typeof(int),
            typeof(FooTextBox),
            new PropertyMetadata(4, OnPropertyChanged));

        /// <summary>
        /// 矩形選択モードなら真を返し、そうでないなら偽を返す。これは依存プロパティです
        /// </summary>
        public bool RectSelectMode
        {
            get { return (bool)GetValue(RectSelectModeProperty); }
            set { SetValue(RectSelectModeProperty, value); }
        }

        /// <summary>
        /// RectSelectModeの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty RectSelectModeProperty =
            DependencyProperty.Register("RectSelectMode", typeof(bool), typeof(FooTextBox), new PropertyMetadata(false, OnPropertyChanged));

        /// <summary>
        /// 折り返しの方法を指定する
        /// </summary>
        /// <remarks>
        /// 変更した場合、レイアウトの再構築を行う必要があります
        /// </remarks>
        public LineBreakMethod LineBreakMethod
        {
            get { return (LineBreakMethod)GetValue(LineBreakProperty); }
            set { SetValue(LineBreakProperty, value); }
        }

        /// <summary>
        /// LineBreakMethodの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty LineBreakProperty =
            DependencyProperty.Register("LineBreakMethod", typeof(LineBreakMethod), typeof(FooTextBox), new PropertyMetadata(LineBreakMethod.None, OnPropertyChanged));


        /// <summary>
        /// 折り返しの幅を指定する。LineBreakMethod.CharUnit以外の時は無視されます
        /// </summary>
        /// <remarks>
        /// 変更した場合、レイアウトの再構築を行う必要があります
        /// </remarks>
        public int LineBreakCharCount
        {
            get { return (int)GetValue(LineBreakCharCountProperty); }
            set { SetValue(LineBreakCharCountProperty, value); }
        }

        /// <summary>
        /// LineBreakCharCountの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty LineBreakCharCountProperty =
            DependencyProperty.Register("LineBreakCharCount", typeof(int), typeof(FooTextBox), new PropertyMetadata(80));        

        /// <summary>
        /// キャレットを描くなら真。そうでないなら偽を返す。これは依存プロパティです
        /// </summary>
        public bool DrawCaret
        {
            get { return (bool)GetValue(DrawCaretProperty); }
            set { SetValue(DrawCaretProperty, value); }
        }

        /// <summary>
        /// DrawCaretの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty DrawCaretProperty =
            DependencyProperty.Register("DrawCaret", typeof(bool), typeof(FooTextBox), new PropertyMetadata(true, OnPropertyChanged));


        /// <summary>
        /// キャレットラインを描くなら真。そうでないなら偽を返す。これは依存プロパティです
        /// </summary>
        public bool DrawCaretLine
        {
            get { return (bool)GetValue(DrawCaretLineProperty); }
            set { SetValue(DrawCaretLineProperty, value); }
        }

        /// <summary>
        /// DrawCaretLineの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty DrawCaretLineProperty =
            DependencyProperty.Register("DrawCaretLine", typeof(bool), typeof(FooTextBox), new PropertyMetadata(false, OnPropertyChanged));

        /// <summary>
        /// 行番号を描くなら真。そうでなければ偽。これは依存プロパティです
        /// </summary>
        public bool DrawLineNumber
        {
            get { return (bool)GetValue(DrawLineNumberProperty); }
            set { SetValue(DrawLineNumberProperty, value); }
        }

        /// <summary>
        /// ルーラーを描くなら真。そうでなければ偽。これは依存プロパティです
        /// </summary>
        public bool DrawRuler
        {
            get { return (bool)GetValue(DrawRulerProperty); }
            set { SetValue(DrawRulerProperty, value); }
        }

        /// <summary>
        /// DrawRulerの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty DrawRulerProperty =
            DependencyProperty.Register("DrawRuler", typeof(bool), typeof(FooTextBox), new PropertyMetadata(false, OnPropertyChanged));


        /// <summary>
        /// DrawLineNumberの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty DrawLineNumberProperty =
            DependencyProperty.Register("DrawLineNumber", typeof(bool), typeof(FooTextBox), new PropertyMetadata(false, OnPropertyChanged));

        /// <summary>
        /// URLに下線を引くなら真。そうでないなら偽を表す。これは依存プロパティです
        /// </summary>
        public bool MarkURL
        {
            get { return (bool)GetValue(MarkURLProperty); }
            set { SetValue(MarkURLProperty, value); }
        }

        /// <summary>
        /// MarkURLの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty MarkURLProperty =
            DependencyProperty.Register("MarkURL", typeof(bool), typeof(FooTextBox), new PropertyMetadata(false, OnPropertyChanged));

        /// <summary>
        /// 全角スペースを表示するなら真。そうでないなら偽
        /// </summary>
        public bool ShowFullSpace
        {
            get { return (bool)GetValue(ShowFullSpaceProperty); }
            set { SetValue(ShowFullSpaceProperty, value); }
        }

        /// <summary>
        /// ShowFullSpaceの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty ShowFullSpaceProperty =
            DependencyProperty.Register("ShowFullSpace", typeof(bool), typeof(FooTextBox), new PropertyMetadata(false, OnPropertyChanged));

        /// <summary>
        /// 半角スペースを表示するなら真。そうでないなら偽
        /// </summary>
        public bool ShowHalfSpace
        {
            get { return (bool)GetValue(ShowHalfSpaceProperty); }
            set { SetValue(ShowHalfSpaceProperty, value); }
        }

        /// <summary>
        /// ShowHalfSpaceの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty ShowHalfSpaceProperty =
            DependencyProperty.Register("ShowHalfSpace", typeof(bool), typeof(FooTextBox), new PropertyMetadata(false, OnPropertyChanged));

        /// <summary>
        /// タブを表示するなら真。そうでないなら偽
        /// </summary>
        public bool ShowTab
        {
            get { return (bool)GetValue(ShowTabProperty); }
            set { SetValue(ShowTabProperty, value); }
        }

        /// <summary>
        /// ShowTabの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty ShowTabProperty =
            DependencyProperty.Register("ShowTab", typeof(bool), typeof(FooTextBox), new PropertyMetadata(false, OnPropertyChanged));

        /// <summary>
        /// 改行マークを表示するなら真。そうでないなら偽
        /// </summary>
        public bool ShowLineBreak
        {
            get { return (bool)GetValue(ShowLineBreakProperty); }
            set { SetValue(ShowLineBreakProperty, value); }
        }

        /// <summary>
        /// ShowLineBreakの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty ShowLineBreakProperty =
            DependencyProperty.Register("ShowLineBreak", typeof(bool), typeof(FooTextBox), new PropertyMetadata(false,OnPropertyChanged));

        
        #endregion

    }
    /// <summary>
    /// コンテキストメニューのイベントデーターを表す
    /// </summary>
    public class FooContextMenuEventArgs
    {
        /// <summary>
        /// 処理済みなら真。そうでないなら偽
        /// </summary>
        public bool Handled = false;
        /// <summary>
        /// コンテキストメニューを表示すべき座標を表す
        /// </summary>
        public Windows.Foundation.Point Postion;
        /// <summary>
        /// コンストラクター
        /// </summary>
        /// <param name="pos"></param>
        public FooContextMenuEventArgs(Windows.Foundation.Point pos)
        {
            this.Postion = pos;
        }
    }
}
