/*
 * Copyright (C) 2007 u6k.yu1@gmail.com, All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 *    3. Neither the name of Clarkware Consulting, Inc. nor the names of its
 *       contributors may be used to endorse or promote products derived
 *       from this software without prior written permission. For written
 *       permission, please contact clarkware@clarkware.com.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL
 * CLARKWARE CONSULTING OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN  ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package jp.gr.java_conf.u6k.filelock;

import java.awt.Container;
import java.awt.FontMetrics;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.SpringLayout;
import javax.swing.UIManager;
import javax.swing.UnsupportedLookAndFeelException;
import javax.swing.WindowConstants;

/**
 * <p>
 * Swingアプリケーションとして起動するメイン・クラスです。
 * </p>
 * 
 * @version $Id$
 */
@SuppressWarnings("serial")
class SwingMain extends JFrame {

    /**
     * <p>
     * Swingコンポーネント同士の間隔。
     * </p>
     */
    private static final int COMPONENT_INTERVAL = 10;

    /**
     * <p>
     * ウィンドウの初期幅。
     * </p>
     */
    private static final int WIDTH              = 400;

    /**
     * <p>
     * ウィンドウの初期高さ。
     * </p>
     */
    private static final int HEIGHT             = 300;

    /**
     * <p>
     * ファイルのロック状況を表示するテーブル。
     * </p>
     */
    private JTable           lockTable;

    /**
     * <p>
     * ウィンドウを初期化します。
     * </p>
     * 
     * @param args
     *            アプリケーション引数。
     * @throws IOException
     *             ファイルのロックに失敗した場合。
     */
    public SwingMain(String[] args) throws IOException {
        final FileLockUtil fileLockUtil = new FileLockUtil(args);

        this.addWindowListener(new WindowAdapter() {

            @Override
            public void windowClosed(WindowEvent e) {
                fileLockUtil.close();
                System.exit(0);
            }

        });

        this.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
        this.setSize(SwingMain.WIDTH, SwingMain.HEIGHT);
        this.setTitle(ResourceUtil.get("app.title") + " " + ResourceUtil.get("app.version"));
        this.setIconImage(Toolkit.getDefaultToolkit().createImage(this.getClass().getResource("/jp/gr/java_conf/u6k/filelock/resources/lock.png")));
        this.setLocationRelativeTo(null);

        Container c = this.getContentPane();
        SpringLayout l = new SpringLayout();
        c.setLayout(l);

        JButton closeButton = new JButton(ResourceUtil.get("message.close"));
        closeButton.setMnemonic(KeyEvent.VK_C);
        closeButton.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent e) {
                SwingMain.this.dispose();
            }

        });
        l.putConstraint(SpringLayout.SOUTH, closeButton, -SwingMain.COMPONENT_INTERVAL, SpringLayout.SOUTH, c);
        l.putConstraint(SpringLayout.EAST, closeButton, -SwingMain.COMPONENT_INTERVAL, SpringLayout.EAST, c);
        c.add(closeButton);

        Map<String, Boolean> lockStateMap = new TreeMap<String, Boolean>();
        for (String path : fileLockUtil.lockFiles()) {
            lockStateMap.put(path, true);
        }
        for (String path : fileLockUtil.lockFailFiles()) {
            lockStateMap.put(path, false);
        }
        List<String[]> lockListItems = new ArrayList<String[]>();
        for (Map.Entry<String, Boolean> entry : lockStateMap.entrySet()) {
            if (entry.getValue()) {
                lockListItems.add(new String[]{ResourceUtil.get("message.lock"), entry.getKey()});
            } else {
                lockListItems.add(new String[]{ResourceUtil.get("message.fail"), entry.getKey()});
            }
        }
        this.lockTable = new JTable(lockListItems.toArray(new String[0][]), new String[]{"State", "Path"});
        this.lockTable.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
        JScrollPane lockTableScroll = new JScrollPane(this.lockTable);
        l.putConstraint(SpringLayout.SOUTH, lockTableScroll, -SwingMain.COMPONENT_INTERVAL, SpringLayout.NORTH, closeButton);
        l.putConstraint(SpringLayout.NORTH, lockTableScroll, SwingMain.COMPONENT_INTERVAL, SpringLayout.NORTH, c);
        l.putConstraint(SpringLayout.EAST, lockTableScroll, -SwingMain.COMPONENT_INTERVAL, SpringLayout.EAST, c);
        l.putConstraint(SpringLayout.WEST, lockTableScroll, SwingMain.COMPONENT_INTERVAL, SpringLayout.WEST, c);
        c.add(lockTableScroll);
    }

    private void columnResize() {
        FontMetrics fm = this.getGraphics().getFontMetrics();

        for (int columnIndex = 0; columnIndex < this.lockTable.getColumnCount(); columnIndex++) {
            int maxSize = 0;
            for (int rowIndex = 0; rowIndex < this.lockTable.getRowCount(); rowIndex++) {
                int s = fm.stringWidth((String) this.lockTable.getValueAt(rowIndex, columnIndex));
                if (s > maxSize) {
                    maxSize = s;
                }
            }

            this.lockTable.getColumnModel().getColumn(columnIndex).setPreferredWidth(maxSize);
        }
    }

    /**
     * <p>
     * アプリケーションのエントリーポイントです。
     * </p>
     * 
     * @param args
     *            アプリケーション引数。
     * @throws IOException
     *             ファイルのロックに失敗した場合。
     * @throws ClassNotFoundException
     *             レイアウト・マネージャの設定に失敗した場合。
     * @throws InstantiationException
     *             レイアウト・マネージャの設定に失敗した場合。
     * @throws IllegalAccessException
     *             レイアウト・マネージャの設定に失敗した場合。
     * @throws UnsupportedLookAndFeelException
     *             レイアウト・マネージャの設定に失敗した場合。
     */
    public static void main(String[] args) throws IOException, ClassNotFoundException, InstantiationException, IllegalAccessException, UnsupportedLookAndFeelException {
        Thread.setDefaultUncaughtExceptionHandler(new Thread.UncaughtExceptionHandler() {

            public void uncaughtException(Thread t, Throwable e) {
                JOptionPane.showMessageDialog(null, e.toString(), "Error", JOptionPane.ERROR_MESSAGE);
            }

        });

        UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());

        SwingMain one = new SwingMain(args);
        one.setVisible(true);
        one.columnResize();
    }

}
