/*
 * Copyright (C) 2007 u6k.yu1@gmail.com, All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 *    3. Neither the name of Clarkware Consulting, Inc. nor the names of its
 *       contributors may be used to endorse or promote products derived
 *       from this software without prior written permission. For written
 *       permission, please contact clarkware@clarkware.com.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL
 * CLARKWARE CONSULTING OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN  ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package jp.gr.java_conf.u6k.filelock;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

/**
 * <p>
 * 指定したファイルをロックします。ディレクトリを指定した場合、子ファイルを再帰的に検索し、ロックします。ロックに失敗しても例外はスローしません。
 * </p>
 * 
 * @version $Id$
 */
public final class FileLockUtil implements Closeable {

    /**
     * <p>
     * ロックしたファイルのパスと{@link FileLock}インスタンスのマップ。
     * </p>
     */
    private Map<String, FileLock>    lockMap          = new TreeMap<String, FileLock>();

    /**
     * <p>
     * ファイルとチャネルのマップ。
     * </p>
     */
    private Map<String, FileChannel> channelMap       = new TreeMap<String, FileChannel>();

    /**
     * <p>
     * ロックに成功したパスのリスト。
     * </p>
     */
    private List<String>             lockFileList     = new ArrayList<String>();

    /**
     * <p>
     * ロックに失敗したパスのリスト。
     * </p>
     */
    private List<String>             lockFailFileList = new ArrayList<String>();

    /**
     * <p>
     * 指定したファイルをロックし、{@link FileLockUtil}インスタンスを初期化します。ディレクトリを指定した場合、子ファイルを再帰的に検索し、ロックします。ロックに失敗しても例外はスローしません。ロックに成功したパスは{@link #lockFiles()}メソッドで、失敗したパスは{@link #lockFailFiles()}メソッドで取得できます。
     * </p>
     * 
     * @param paths
     *            ロックするファイルの配列。
     * @throws NullPointerException
     *             paths引数がnullの場合。paths配列中にnullが混入していた場合。
     */
    public FileLockUtil(String[] paths) {
        /*
         * 引数を確認します。
         */
        if (paths == null) {
            throw new NullPointerException("paths");
        }
        for (int i = 0; i < paths.length; i++) {
            if (paths[i] == null) {
                throw new NullPointerException("paths[" + i + "] == null");
            }
        }

        /*
         * ファイルをロックします。
         */
        for (String path : paths) {
            this.lock(path);
        }
    }

    private void lock(String path) {
        // 正規化したファイルを取得します。
        File file;
        try {
            file = new File(path).getCanonicalFile();
        } catch (IOException e) {
            this.lockFailFileList.add(path);
            return;
        }

        if (file.isFile()) {
            // ファイルであればロックします。
            try {
                FileChannel fch = new RandomAccessFile(file, "rw").getChannel();
                this.channelMap.put(file.getAbsolutePath(), fch);

                FileLock l = fch.tryLock();
                this.lockMap.put(file.getAbsolutePath(), l);

                this.lockFileList.add(file.getAbsolutePath());
            } catch (IOException e) {
                this.lockFailFileList.add(file.getAbsolutePath());
                return;
            }
        } else if (file.isDirectory()) {
            // ディレクトリであれば、再帰的にファイルをロックします。
            for (File child : file.listFiles()) {
                this.lock(child.getAbsolutePath());
            }
        }
    }

    /**
     * <p>
     * ロックに成功したファイルのパスの配列を返します。
     * </p>
     * 
     * @return ロックに成功したファイルのパスの配列。
     */
    public String[] lockFiles() {
        return this.lockFileList.toArray(new String[0]);
    }

    /**
     * <p>
     * ロックに失敗したファイルのパスの配列を返します。
     * </p>
     * 
     * @return ロックに失敗したファイルのパスの配列。
     */
    public String[] lockFailFiles() {
        return this.lockFailFileList.toArray(new String[0]);
    }

    /**
     * <p>
     * インスタンスが保持している全てのリソースを開放します。つまり、ロックしている全てのファイルは開放され、{@link #lockFiles()}、{@link #lockFailFiles()}メソッドは空配列を返すようになります。
     * </p>
     */
    public void close() {
        for (FileChannel fch : this.channelMap.values()) {
            try {
                fch.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }

        this.channelMap.clear();
        this.lockMap.clear();
        this.lockFileList.clear();
        this.lockFailFileList.clear();
    }

}
