package com.yuji.ef.utility;

import java.util.ArrayList;
import java.util.List;

import android.database.sqlite.SQLiteDatabase;

import com.evernote.edam.notestore.NoteList;
import com.evernote.edam.type.Note;
import com.evernote.edam.type.Notebook;
import com.yuji.ef.common.CommonUtil;
import com.yuji.ef.dao.DatabaseHelper;
import com.yuji.ef.dao.DirNode;
import com.yuji.ef.dao.FileNode;
import com.yuji.ef.dao.Node;
import com.yuji.ef.dao.NodeCacheDao;
import com.yuji.ef.dao.NodeDao;
import com.yuji.ef.dao.RootNode;
import com.yuji.ef.exception.EfError;
import com.yuji.ef.exception.RollbackException;

public class FolderUtil {
	private EvernoteUtil util = EvernoteUtil.getInstance();
	private NodeDao nodeDao = (NodeDao) NodeCacheDao.getInstance();

	public static FolderUtil getInstance() {
		if (instance == null) {
			instance = new FolderUtil();
		}
		return instance;
	}

	private static FolderUtil instance = null;

	private FolderUtil() {

	}

	private EfError getErrorId(EvernoteUtil.Error err){
		return (err == EvernoteUtil.Error.CONNECT)? EfError.NETWORK : EfError.SYSTEM;
	}
	
	public void updateNotebook(SQLiteDatabase db, List<Long> removeList) throws RollbackException {
		List<Notebook> noteBookList = util.getNoteBookList();
		if (noteBookList == null){
			throw new RollbackException(getErrorId(util.getErrorCode()));
		}
		
		long id;
		Node top = nodeDao.searchRoot(db);

		for (Notebook notebook : noteBookList) {
			String guid = notebook.getGuid();
			String name = notebook.getName();

			Node oldBook = nodeDao.searchByGuid(db, guid);
			if (oldBook == null) {
				DirNode dirNode = new DirNode(name, null);
				guid = notebook.getGuid();
				dirNode.setGuid(guid);

				dirNode.setSelected(true);
				id = addDirNodeNT(top, dirNode);
			} else {
				String oldName = oldBook.getName();
				if (!name.equals(oldName)) {
					// 名称が異なる
					nodeDao.updateNameNT(oldBook, name);
				}
			}
		}

		// 削除
		List<Node> list = nodeDao.searchBook(db);
		if (list == null){
			throw new RollbackException(getErrorId(util.getErrorCode()));
		}
		for (Node node : list) {
			boolean isFound = false;
			for (Notebook book : noteBookList) {
				String guid = book.getGuid();
				if (node.getGuid().equals(guid)) {
					isFound = true;
					break;
				}
			}
			if (!isFound) {
				if (CommonUtil.isNull(node.getChildrenString())) {
					// 子が無い場合は削除
					nodeDao.deleteNT(node.getId());
				} else {
					// フォルダに変更
					nodeDao.updateGuidNT(node, null);
					// 削除候補
					removeList.add(node.getId());
				}
			}
		}
	}

	public void deleteNotebook(SQLiteDatabase db, List<Long> removeList) {
		for (long id : removeList){
			Node node = nodeDao.searchById(db, id);
			if (node != null){
				if (CommonUtil.isNull(node.getChildrenString())){
					nodeDao.deleteNT(node.getId());
				}
			}
		}
	}
	
	public void updateNote(SQLiteDatabase db) throws RollbackException {
		List<Node> list = nodeDao.searchSelectedBook(db);
		if (list == null){
			throw new RollbackException(EfError.SYSTEM);
		}
		for (Node node : list) {
			String guid = node.getGuid();
			List<NoteList> noteListList = util.getNoteListByNotebook(guid);
			if (noteListList == null){
				throw new RollbackException(getErrorId(util.getErrorCode()));
			}
			updateNote(db, noteListList);
		}
	}

	public void updateNote(SQLiteDatabase db, List<NoteList> noteListList) throws RollbackException {
		for (NoteList noteList : noteListList) {
			List<Note> notes = noteList.getNotes();
			for (Note note : notes) {
				String name = note.getTitle();
				String guid = note.getGuid();
				String nbGuid = note.getNotebookGuid();

				Node oldNote = nodeDao.searchByGuid(db, guid);
				if (oldNote == null) {
					// 新規ノート
					DirNode parent = (DirNode) nodeDao.searchByGuid(db, nbGuid);
					if (parent == null) {
						throw new RollbackException(EfError.NOT_FOUND);
					}

					FileNode fileNode = new FileNode(name, null);
					fileNode.setGuid(guid);

					long id = addFileNodeNT(parent, fileNode);

					oldNote = fileNode;
					oldNote.setId(id);
				} else {
					DirNode oldParent = (DirNode) nodeDao.searchById(db,
							oldNote.getParent());
					if (oldParent == null) {
						throw new RollbackException(EfError.NOT_FOUND);
					}

					String oldName = oldNote.getName();
					if (!name.equals(oldName)) {
						// 名称が異なる
						nodeDao.updateNameNT(oldNote, name);
					}

					String pGuid = oldParent.getGuid();
					if (!CommonUtil.isNull(pGuid) && !pGuid.equals(nbGuid)) {
						// 親が異なる
						DirNode parent = (DirNode) nodeDao.searchByGuid(db,
								nbGuid);
						if (parent == null) {
							throw new RollbackException(EfError.NOT_FOUND);
						}

						long id;
						id = nodeDao.removeChildrenIdNT(oldParent,
								oldNote.getId());
						if (id < 0) {
							throw new RollbackException(EfError.NOT_FOUND);
						}
						id = nodeDao.addChildrenIdNT(parent, oldNote.getId());
						if (id < 0) {
							throw new RollbackException(EfError.NOT_FOUND);
						}
						nodeDao.updateParentNT(oldNote, parent.getId());
					}
				}
				// Evernoteに存在するノートである
				nodeDao.updateFlagNT(oldNote, 1);
			}
		}
	}

	private void deleteNode(SQLiteDatabase db) throws RollbackException {
		List<Node> list = nodeDao.searchNoteByFlag(db, 0);
		if (list == null){
			throw new RollbackException(getErrorId(util.getErrorCode()));
		}
		for (Node node : list) {
			nodeDao.deleteNT(node.getId());
			long pid = node.getParent();
			Node p = nodeDao.searchById(db, pid);
			if (p != null) {
//				nodeDao.removeChildrenIdNT(p, pid);
				nodeDao.removeChildrenIdNT(p, node.getId());
			}
		}
	}

	public void update(boolean isRoot, boolean isBook, boolean isNote) throws RollbackException {
		EvernoteUtil util = EvernoteUtil.getInstance();
		if (!util.isLoggedIn()){
			throw new RollbackException(EfError.NO_LOGIN);
		}
		
		DatabaseHelper helper = DatabaseHelper.getInstance();
		SQLiteDatabase db = helper.getWritableDatabase();

		db.beginTransaction();
		try {
			if (isRoot) {
				if (nodeDao.isEmpty()) {
					Node top = new RootNode("/", null);
					top.setStatus(Node.Status.OPEN);
					long id = nodeDao.addNT(top);
					top.setId(id);
				}
			}

			ArrayList<Long> removeList = new ArrayList<Long>();
			if (isBook) {
				updateNotebook(db, removeList);
			}
			if (isNote) {
				nodeDao.updateNoteFlag(0);
				updateNote(db);
				deleteNode(db);			
			}
			if (isBook){
				deleteNotebook(db, removeList);				
			}
			
			db.setTransactionSuccessful();
			nodeDao.clear();
		}
		catch (RuntimeException e){
			throw e;
		} finally {
			db.endTransaction();
		}
	}

	public long addFileNode(Node parent, Node node) {
		DatabaseHelper helper = DatabaseHelper.getInstance();
		SQLiteDatabase db = helper.getWritableDatabase();
		return addFileNode(db, parent, node);
	}

	public long addFileNode(SQLiteDatabase db, Node parent, Node node) {
		long id = -1;

		db.beginTransaction();
		try {
			id = addFileNodeNT(parent, node);
			db.setTransactionSuccessful();
		} finally {
			db.endTransaction();
		}
		return id;
	}

	private long addFileNodeNT(Node parent, Node node) {
		long id;
		node.setParent(parent.getId());
		id = nodeDao.addNT(node);
		nodeDao.addChildrenIdNT(parent, id);
		return id;
	}

	public long addDirNode(Node parent, Node node) {
		DatabaseHelper helper = DatabaseHelper.getInstance();
		SQLiteDatabase db = helper.getWritableDatabase();
		return addDirNode(db, parent, node);
	}

	public long addDirNode(SQLiteDatabase db, Node parent, Node node) {
		long id = -1;

		db.beginTransaction();
		try {
			id = addDirNodeNT(parent, node);
			db.setTransactionSuccessful();
		} finally {
			db.endTransaction();
		}
		return id;
	}

	private long addDirNodeNT(Node parent, Node node) {
		long id;
		node.setParent(parent.getId());
		id = nodeDao.addNT(node);
		node.setId(id);
		nodeDao.addChildrenIdNT(parent, node.getId());
		return id;
	}

	public long deleteNode(Node node) {
		DatabaseHelper helper = DatabaseHelper.getInstance();
		SQLiteDatabase db = helper.getWritableDatabase();
		return deleteNode(db, node);
	}

	public long deleteNode(SQLiteDatabase db, Node node) {
		long id = -1;

		db.beginTransaction();
		try {
			id = deleteNodeNT(node);
			db.setTransactionSuccessful();
		} finally {
			db.endTransaction();
		}
		return id;
	}

	private long deleteNodeNT(Node node) {
		long id;

		if (node instanceof DirNode) {
			DirNode dirNode = (DirNode) node;
			List<Long> list = dirNode.getChildren();
			for (Long l : list) {
				id = l.longValue();
				Node n = nodeDao.searchById(id);
				if (n != null) {
					deleteNodeNT(n);
				}
			}
		}

		id = node.getParent();
		if (id >= 0) {
			Node p = nodeDao.searchById(id);
			if (p != null) {
				nodeDao.removeChildrenIdNT(p, node.getId());
			}
		}
		nodeDao.deleteNT(node.getId());

		return 0;
	}
	
	public void createTestNote(){
		int pos = 1;
		List<Node> list = nodeDao.searchBook();
		Node node = list.get(pos);
		String guid = node.getGuid();
		
		EvernoteUtil util = EvernoteUtil.getInstance();
		util.createNotes(guid, "テスト", 100);
	}	
}
