/*
 * Copyright (c) 2009, Takeyuki Nagao
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the
 * following conditions are met:
 * 
 *  * Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *  * Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other
 *    materials provided with the distribution.
 *    
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 */

package test.dvi;
import java.awt.Image;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.logging.Logger;
import java.util.zip.ZipOutputStream;

import org.junit.Assert;
import org.junit.Test;

import dvi.DviException;
import dvi.DviResolution;
import dvi.DviSize;
import dvi.api.DviContext;
import dvi.ctx.DefaultDviContext;
import dvi.gs.GhostscriptUtils;
import dvi.image.split.DefaultSplitImageWriter;
import dvi.image.split.FileImagePiece;
import dvi.image.split.ImageFileConfig;
import dvi.image.split.SplitPiece;
import dvi.image.split.ZipSplitImageReader;
import dvi.image.split.ZipSplitImageWriter;
import dvi.util.DviUtils;
import dvi.util.ZipBuilder;
public class GhostscriptSplitterTest {
  private static final Logger LOGGER = Logger.getLogger(GhostscriptSplitterTest.class.getName());
  
  @Test
  public void testCreateCommandShell() throws DviException, IOException {
    DviContext ctx = new DefaultDviContext();
    final DviResolution res = new DviResolution(300, 3);
    final DviSize unit = new DviSize(256, 256);
    final File tmpDir = File.createTempFile("dvicore-unittest", null);
    tmpDir.delete();
    tmpDir.mkdir();
    final DefaultSplitImageWriter imageWriter = new DefaultSplitImageWriter
      (new File(tmpDir, "gs"), ImageFileConfig.PNG, res);
    InputStream is = getClass().getResourceAsStream("sample011.ps");
    org.junit.Assert.assertNotNull("Failed to read sample011.ps", is);
    GhostscriptUtils.renderAndSplit(ctx, is, unit, imageWriter, res, "pnmraw", 1, 0, null);
    final Collection<SplitPiece> pieces = imageWriter.getSplitImage().getPieces();
    for (SplitPiece piece : pieces) {
      if (piece instanceof FileImagePiece) {
        FileImagePiece sf = (FileImagePiece) piece;
        File f = sf.getFile();
        if (f != null) {
          f.delete();
        }
      }
    }
    tmpDir.delete();
  }
  
  @Test
  public void testZipDeflated() throws Exception {
    testZipWithMethod(ZipOutputStream.DEFLATED);
  }
  
  @Test
  public void testZipStored() throws Exception {
    testZipWithMethod(ZipOutputStream.STORED);
  }
  
  public void testZipWithMethod(int method) throws Exception {
    DviContext ctx = new DefaultDviContext();
    final DviResolution res = new DviResolution(300, 3);
    final DviSize unit = new DviSize(256, 256);
    final String basename = "dvicore-unittest2";
    final File tmpFile = File.createTempFile(basename, ".zip");
    try {
      final FileOutputStream fos = new FileOutputStream(tmpFile);
      final ZipBuilder zb = new ZipBuilder(fos);
      try {
        final ZipSplitImageWriter imageWriter = new ZipSplitImageWriter
          (basename, ImageFileConfig.PNG, res, zb);
        InputStream is = getClass().getResourceAsStream("sample011.ps");
        org.junit.Assert.assertNotNull("Failed to read sample011.ps", is);
        GhostscriptUtils.renderAndSplit(ctx, is, unit, imageWriter, res, "pnmraw", 1, 0, null);
        ZipSplitImageReader imageReader = new ZipSplitImageReader(tmpFile);
        for (SplitPiece piece : imageReader.getSplitImage()) {
          LOGGER.info(piece.toString());
          Image image = piece.getImage();
          Assert.assertNotNull("Output image is null", image);
        }
      } finally {
        DviUtils.silentClose(zb);
      }
    } finally {
      tmpFile.delete();
    }
  }
}
