/*
 * Copyright (c) 2009, Takeyuki Nagao
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the
 * following conditions are met:
 * 
 *  * Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *  * Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other
 *    materials provided with the distribution.
 *    
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 */

package jp.sourceforge.dvibrowser.dvicore;

// immutable.

public final class DviResolution
implements java.io.Serializable
{
  private static final long serialVersionUID = -8353020586758660972L;
  public static final int MAX_SHRINK_FACTOR = 1024;

  private final int dpi;
  private final int shrinkFactor;
  private final double actualDpi;

  public DviResolution(int dpi, int shrinkFactor) {
    this.dpi = dpi;
    this.shrinkFactor = shrinkFactor;
    checkVars();
    actualDpi = (double) dpi / shrinkFactor;
  }

  private void checkVars() {
    if (shrinkFactor <= 0 || MAX_SHRINK_FACTOR < shrinkFactor)
      throw new IllegalArgumentException
        ("Illegal value of shrinkFactor: " + shrinkFactor);
  }

  public double actualDpi() {
    return actualDpi;
  }
  public int dpi() { return dpi; }
  public int shrinkFactor() { return shrinkFactor; }

  public DviResolution approximate(double approximateDpi) {
    int sf = (int) Math.floor(dpi / approximateDpi);
    sf = Math.min(sf, MAX_SHRINK_FACTOR);
    sf = Math.max(sf, 1);
    return new DviResolution(dpi, sf);
  }

  public String toString() {
    return String.format(getClass().getName() + "[dpi=%d,shrinkFactor=%d,actualDpi=%.1f]", dpi, shrinkFactor, actualDpi());
  }
  public boolean equals(Object obj) {
    if (obj instanceof DviResolution) {
      DviResolution a = (DviResolution) obj;
      return a.dpi == dpi && a.shrinkFactor == shrinkFactor;
    }
    return false;
  }

  public int hashCode() {
    return dpi + 33*shrinkFactor;
  }
}
