/*
 * Copyright (c) 2012, Takeyuki Nagao
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the
 * following conditions are met:
 * 
 *  * Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *  * Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other
 *    materials provided with the distribution.
 *    
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 */

package jp.sourceforge.dvibrowser.dvi2epub.cmd;

import java.io.PrintWriter;
import java.util.List;

import jp.sourceforge.dvibrowser.dvi2epub.reflect.Dispatcher;
import jp.sourceforge.dvibrowser.dvi2epub.reflect.MemberWalker;
import jp.sourceforge.dvibrowser.dvi2epub.reflect.MemberWalkerException;

public class AnnotatedCommandLineParser implements CommandLineParser {
	private final AnnotatedCommand command;
	private final OptionMapper mapper;
	private List<Option> options;

	public AnnotatedCommandLineParser(AnnotatedCommand cmd) throws CommandException {
		this.command = cmd;
		this.mapper = command.createOptionMapper();
		scanOptions();
	}

	protected void scanOptions() throws CommandException {
		OptionParserAdapter adapter = new OptionParserAdapter(mapper);
		MemberWalker walker = new MemberWalker(adapter);
		try {
			walker.walk(command);
		} catch (MemberWalkerException e) {
			throw new CommandException(e);
		}
		options = adapter.getOptions();
	}

	@Override
	public void parse(String[] args) throws CommandException {
		ParserState state = new ParserState(args);
		while (!state.wantStop()) {
			String arg1 = state.shift();
//			System.out.println("arg1=" + arg1);
			if (arg1 == null) {
				state.wantStop(true);
			} else if (arg1.startsWith("--")) {
				// Long parameter
				parseLongOption(state, arg1);
			} else if ("-".equals(arg1)) {
				state.wantStop(true);
			} else if (arg1.startsWith("-")) {
				// Short parameter
				parseShortOption(state, arg1);
			} else {
				// No parameter
				state.unshift(arg1);
				state.wantStop(true);
			}
		}
		
		if (state.hasError()) {
			throw new CommandException(state.getThrowable());
		}
		
		command.setArgs(state.getList().toArray(new String [0]));
	}

	private void parseShortOption(final ParserState state, final String arg1) throws CommandException {
		char c = arg1.charAt(1);
		String arguments = null;
		if (arg1.length() > 2) {
			arguments = arg1.substring(2);
		}
		doParseOption(state, "" + c, arguments);
	}

	private void parseLongOption(final ParserState state, final String arg1) throws CommandException {
		String name = arg1.substring(2);
		String arguments = null;
		int pos = name.indexOf('=');
		if (-1 != pos) {
			arguments = name.substring(pos + 1);
			name = name.substring(0, pos);
		}
		
		doParseOption(state, name, arguments);
	}
	
	private void doParseOption(final ParserState state, final String name, final String parameter) throws CommandException
	{
		state.setOptionName(name);
		state.setOptionParameter(parameter);
		
		for (Option option : getOptions()) {
			try {
				if (option != null && option.matches(name)) {
					Dispatcher<Object[]> d = new Dispatcher<Object[]>(getMapper(),
							"map", option.getAnnotation(), state);
					if (d.dispatch()) {
						Object[] args = d.getResult();
						option.getMethod().invoke(getCommand(), args);
						return;
					}
				}
			} catch (Exception ex) {
				state.stopWithError(ex);
				return;
			}

		}
	}

	public AnnotatedCommand getCommand() {
		return command;
	}

	@Override
	public void printHelp(PrintWriter pw) throws CommandException {
		pw.println("Usage: " + command.getApplicationName() + " " + command.getCommandLineSyntax());
		if (getOptions().size() > 0) {
			pw.println("The most commonly used options are:");
		}
		
		OptionPrinter printer = command.createOptionPrinter(pw);
		printer.begin();
		for (Option option : getOptions()) {
			printer.print(option);
		}
		printer.end();
	}

	public List<Option> getOptions() {
		return options;
	}

	public OptionMapper getMapper() {
		return mapper;
	}
}
