/*
 * Copyright (c) 2009, Takeyuki Nagao
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the
 * following conditions are met:
 * 
 *  * Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *  * Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other
 *    materials provided with the distribution.
 *    
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 */

package jp.sourceforge.dvibrowser.dvicore.util.progress;

import java.util.logging.Logger;

import jp.sourceforge.dvibrowser.dvicore.DviException;
import jp.sourceforge.dvibrowser.dvicore.DviObject;


public class ManagedProgressItem
extends DviObject
implements ProgressItem
{
  private static final Logger LOGGER = Logger.getLogger(ManagedProgressItem.class.getName());
  
  public static final int STATE_INIT = 0;
  public static final int STATE_OPEN = 1;
  public static final int STATE_CLOSED = 2;
  private final AbstractProgressModel recorder;
  private final ProgressItem item;
  private int start, end, current;
  private int state = 0;
  public ManagedProgressItem(AbstractProgressModel recorder, ProgressItem item)
  {
    super(recorder);
    this.recorder = recorder;
    this.item = item;
  }
  
  public boolean isOpen() {
    return state == STATE_OPEN;
  }

  public boolean isClosed() {
    return state == STATE_CLOSED;
  }
  
  public AbstractProgressModel getProgressRecorder()
  {
    return recorder;
  }

  public ProgressItem getOriginalItem()
  {
    return item;
  }

  public void close() throws DviException
  {
    if (state == STATE_INIT)
      throw new IllegalStateException("Item is never opened.");
    if (state == STATE_CLOSED) return;
    state = STATE_CLOSED;
    this.current = this.end;
    item.close();
    LOGGER.fine("close: " + item);
    recorder.fireProgressCloseEvent(this);
  }

  public void open(int start, int end) throws DviException
  {
    if (state != STATE_INIT)
      throw new IllegalStateException("Item is not open or closed.");
    state = STATE_OPEN;
    this.start = start;
    this.current = start;
    this.end = end;
    item.open(start, end);
    LOGGER.fine("opened: " + item);
    recorder.fireProgressOpenEvent(this);
  }

  public void update(int current) throws DviException
  {
    if (state != STATE_OPEN)
      throw new IllegalStateException("Item is not open or closed.");
    this.current = current;
    item.update(current);
    LOGGER.fine("update: " + item);
    recorder.fireProgressUpdateEvent(this);
  }
  
  public String toString()
  {
    return getClass().getName() + "[" + item.toString() + "]";
  }
  
  protected void finalize()
  {
    try {
      close();
    } catch (DviException e) {
      e.printStackTrace();
    }
  }

  public int getStart()
  {
    return start;
  }

  public int getEnd()
  {
    return end;
  }

  public int getCurrent()
  {
    return current;
  }

  public int getState()
  {
    return state;
  }
}