/*
 * Copyright (c) 2009, Takeyuki Nagao
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the
 * following conditions are met:
 * 
 *  * Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *  * Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other
 *    materials provided with the distribution.
 *    
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 */

package jp.sourceforge.dvibrowser.dvicore.render;

import java.util.concurrent.atomic.AtomicLong;

import jp.sourceforge.dvibrowser.dvicore.api.GammaCorrector;


public class DefaultGammaCorrector
implements GammaCorrector
{
  private static final AtomicLong serializer = new AtomicLong();
  private final double factor, exponent;
  private final long serial;

  public DefaultGammaCorrector()
  {
    this(1.0, 1.0);
  }
  
  public DefaultGammaCorrector(double factor, double exponent)
  {
    this.factor = factor;
    this.exponent = exponent;
    this.serial = serializer.incrementAndGet();
  }

  public double factor() { return factor; }
  public double exponent() { return exponent; }

  public int correctGamma(int c, int maxval)
  {
    return (int) (
        1024 *
          Math.min(
            1.0,
            Math.pow(factor * c / maxval, exponent)
          )
        + 0.5
      );
  }
  
  @Override
  public int hashCode() {
    return (int) (33 * (factor + 33 * exponent));
  }
  
  @Override
  public boolean equals(Object o) {
    if (!(o instanceof DefaultGammaCorrector)) return false;
    DefaultGammaCorrector gc = (DefaultGammaCorrector) o;
    return (factor == gc.factor && exponent == gc.exponent);
  }
  
  public String getCacheKey() {
    return getClass().getName() + "--" + serial;
  }
  
  @Override
  public String toString() {
    return getClass().getName() + "[factor=" + factor+ ",exponent=" + exponent + "]";
  }
}
