//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Sep-30, 2012.
//
#include "dcvceditmodecursor.h"

#include "utils/dccomponentutil.h"
#include "dcvceditmodeterminal.h"
#include "dccell.h"
#include "dcaxon.h"
#include "dcreceptor.h"
#include "dcvcpage.h"

#include <QtOpenGL>

static const double PI = 3.14159265358979323846264338327950288419717;

//TEMP
#include "dccuberenderer.h"
static DCCubeRenderer *s_cursorShape = NULL;

DCVCEditModeCursor::DCVCEditModeCursor()
    : d_ownerCell(NULL), d_ownerAxon(NULL), d_ownerReceptor(NULL),
      d_cursorType(DCV_CURSOR_NONE), d_dropTarget(NULL), d_dragOffsetX(0), d_dragOffsetY(0), d_isDragging(false)
{
    d_terminal = new DCVCEditModeTerminal;
}

DCVCEditModeCursor::~DCVCEditModeCursor()
{
    if (d_terminal)
    {
        delete d_terminal;
        d_terminal = NULL;
    }
}


void DCVCEditModeCursor::setOwnerFromAxon(DCAxon *owner)
{
    if (d_ownerAxon)
    {
        d_ownerAxon->unregisterEditCursor();
    }

    d_ownerAxon = owner;
    d_ownerCell = d_ownerAxon->getOwnerCell();
    d_cursorType = DCV_CURSOR_FROM_AXON;
    if (d_ownerCell)
    {
        changePageBelonging(d_ownerCell->getPageBelonging());
        setVisible(DCVComponent::DCV_VISIBLE_FULL, DCVComponent::DCV_VISIBLE_FULL);
        owner->registerEditCursor(this);
        float rad = owner->getViewAngle() / 180 * PI;
        float len = owner->getViewLength() + 0.5;
        setPageX(d_ownerCell->getViewPageX() + cos(rad) * len);
        setPageY(d_ownerCell->getViewPageY() + sin(rad) * len);
    }
    else
    {
        changePageBelonging(NULL);
        setVisible(DCVComponent::DCV_VISIBLE_NONE, DCVComponent::DCV_VISIBLE_NONE);
    }
}

void DCVCEditModeCursor::setOwnerFromReceptor(DCReceptor *owner)
{
    if (d_ownerReceptor)
    {
        d_ownerReceptor->unregisterEditCursor();
    }

    d_ownerReceptor = owner;
    d_ownerCell = owner->getOwnerCell();
    if (d_ownerReceptor && d_ownerCell)
    {
        d_cursorType = DCV_CURSOR_FROM_RECEPTAOR;
        changePageBelonging(d_ownerCell->getPageBelonging());
        setVisible(DCVComponent::DCV_VISIBLE_FULL, DCVComponent::DCV_VISIBLE_FULL);
        owner->registerEditCursor(this);
        float rad = d_ownerCell->getAxon()->getViewAngle() / 180 * PI;
        float len = d_ownerCell->getAxon()->getViewLength() + 0.2;
        setPageX(d_ownerCell->getViewPageX() - cos(rad) * len);
        setPageY(d_ownerCell->getViewPageY() - sin(rad) * len);
    }
    else
    {
        changePageBelonging(NULL);
        setVisible(DCVComponent::DCV_VISIBLE_NONE, DCVComponent::DCV_VISIBLE_NONE);
    }
}

void DCVCEditModeCursor::unsetCursor()
{
    if (d_ownerAxon)
    {
        d_ownerAxon->unregisterEditCursor();
        d_ownerAxon = NULL;
    }
    if (d_ownerReceptor)
    {
        d_ownerReceptor->unregisterEditCursor();
        d_ownerReceptor = NULL;
    }
    d_ownerCell = NULL;
    changePageBelonging(NULL);
    setVisible(DCVComponent::DCV_VISIBLE_NONE, DCVComponent::DCV_VISIBLE_NONE);
}

void DCVCEditModeCursor::changePageBelonging(DCVCPage *page)
{
    if (getPageBelonging() != page)
    {
        if (page)
        {
            page->registerEditCursor(this);
        }
        else
        {
            getPageBelonging()->unregisterEditCursor();
        }
        emit pageBelongingChanged(page);
    }
}

void DCVCEditModeCursor::setDropTarget(DCVComponent *dropTarget)
{
    bool changed = false;
    if (d_dropTarget != dropTarget)
    {
        d_dropTarget = dropTarget;
        changed = true;
    }

    if (changed)
        emit dropTargetChanged(dropTarget);
}

bool DCVCEditModeCursor::isResizingArea(float x, float y, float z) const
{
    (void)x; (void)y; (void)z;
    return false;
}

void DCVCEditModeCursor::prepareChildrenForDraw(bool isAnimationInterval)
{
    (void)isAnimationInterval;

    float matrix[16];

    switch(d_cursorType)
    {
    case DCV_CURSOR_FROM_AXON:
        glGetFloatv(GL_MODELVIEW_MATRIX, matrix);
        d_terminal->setViewMatrixForReceptorPoint(NULL, matrix);
        break;

    case DCV_CURSOR_FROM_RECEPTAOR:
        glGetFloatv(GL_MODELVIEW_MATRIX, matrix);
        d_terminal->setViewMatrixForAxonPoint(matrix);
        break;

    default:
        break;
    }
}

void DCVCEditModeCursor::drawChildren(bool isAnimationInterval)
{
    d_terminal->draw(isAnimationInterval);
}

void DCVCEditModeCursor::drawChildrenForSelection(QList<DCVComponent*> *itemList)
{
    d_terminal->drawForSelection(itemList);
}

void DCVCEditModeCursor::translate()
{
    glTranslatef(getPageX() , 0.15f, getPageY());
}

void DCVCEditModeCursor::renderOwnShape(bool isAnimationInterval, bool renderAsWireframe)
{
    (void)isAnimationInterval; (void)renderAsWireframe;

    if (!s_cursorShape)
    {
        s_cursorShape = new DCCubeRenderer(0.3f,0.3f, 0.01f, 1);
    }
    s_cursorShape->draw();
}

bool DCVCEditModeCursor::startDrag(float x, float y, float z, bool isResizingDrag)
{
    (void)x; (void)y; (void)z; (void)isResizingDrag;

    d_dragOffsetX = getPageX() - x;
    d_dragOffsetY = getPageY() - z;

    d_isDragging = true;

    return true;
}

bool DCVCEditModeCursor::dragging(float x, float y, float z, bool isResizingDrag)
{
    (void)x; (void)y; (void)z; (void)isResizingDrag;

    setPageX(d_dragOffsetX + x);
    setPageY(d_dragOffsetY + z);
    d_isDragging = true;
    return true;
}

bool DCVCEditModeCursor::endDrag(float x, float y, float z, bool isResizingDrag)
{
    (void)x; (void)y; (void)z; (void)isResizingDrag;

    d_isDragging = false;
    return true;
}

void DCVCEditModeCursor::updateShape()
{
    delete s_cursorShape;
    s_cursorShape = NULL;
}
