//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on Oct-20, 2012.
//

#include "dcuieditor.h"

#include <QVBoxLayout>
#include <QLabel>
#include <QSplitter>
#include <QTextStream>
#include <QWebFrame>
#include <QMessageBox>
#include <QFileInfo>

#include "dcuitexteditor.h"

static DCUIEditor *s_instance = NULL;

DCUIJSInterface::DCUIJSInterface()
{

}

DCUIJSInterface::~DCUIJSInterface()
{

}

void DCUIJSInterface::setValue(const QString name, float value)
{
    (void)name; (void)value;
}

float DCUIJSInterface::getValue(const QString name) const
{
    return 0;
}


DCUIEditor::DCUIEditor(QWidget *parent) :
    QDialog(parent), d_editingFile(""),d_isEditing(false), d_isInInitialization(true), d_isChangeMade(false)
{
    setWindowTitle(tr("interface editor"));
    QSplitter *splitter = new QSplitter;

    d_tabWidget = new QTabWidget;
    d_webView = new QWebView;
    d_textEditor = new DCUITextEditor;
    d_jsInterface = new DCUIJSInterface;

    d_tabWidget->addTab(d_webView, tr("Preview"));
    d_tabWidget->addTab(d_textEditor, tr("Source"));

    splitter->addWidget(d_tabWidget);

    QVBoxLayout *layout = new QVBoxLayout;

    layout->addWidget(splitter);

    QHBoxLayout *buttonLayout = new QHBoxLayout;
    d_saveButton = new QPushButton(tr("Save"));
    d_closeButton = new QPushButton(tr("Close"));
    buttonLayout->addStretch();
    buttonLayout->addWidget(d_saveButton);
    buttonLayout->addWidget(d_closeButton);
    layout->addLayout(buttonLayout);

    setLayout(layout);

    QWebPage *page = d_webView->page();
    if (page)
    {
        QWebFrame *frame = page->mainFrame();
        if (frame)
        {
            connect(frame, SIGNAL(javaScriptWindowObjectCleared()), this, SLOT(javaScriptWindowObjectCleared()));
        }
    }
    connect(d_tabWidget, SIGNAL(currentChanged(int)), this, SLOT(tabChanged(int)));
    connect(d_textEditor, SIGNAL(textChanged()), this, SLOT(textChanged()));
    connect(d_saveButton, SIGNAL(clicked()), this, SLOT(saveChange()));
    connect(d_closeButton, SIGNAL(clicked()), this, SLOT(close()));
}

bool DCUIEditor::getIsModified() const
{
    if (d_isEditing && d_isChangeMade)
    {
        if (d_originalSource != d_textEditor->toPlainText())
        {
            return true;
        }
    }
    return false;
}

bool DCUIEditor::maybeSave()
{
    if (d_isEditing)
    {
        if (getIsModified())
        {
            int button = QMessageBox::question(this, tr("File modified"), tr("File modified. Do you want to save the change?"), QMessageBox::Save | QMessageBox::No | QMessageBox::Cancel, QMessageBox::Save );
            if (button == QMessageBox::Save)
            {
                saveChange();
                return true;
            }
            else if (button == QMessageBox::No)
            {
                return true;
            }
            else if (button == QMessageBox::Cancel)
            {
                return false;
            }
        }
    }
    return true;
}

DCUIEditor* DCUIEditor::getEditor()
{
    if (!s_instance)
    {
        s_instance = new DCUIEditor;
    }

    return s_instance;
}


void DCUIEditor::destroyEditor()
{
    if (s_instance)
    {
        delete s_instance;
        s_instance = NULL;
    }
}

void DCUIEditor::closeEvent(QCloseEvent *event)
{
    if (maybeSave())
    {
        event->accept();
    }
    else
    {
        event->ignore();
    }
}

bool DCUIEditor::startEditing(const QString &sysFilePath)
{
    d_editingFile = sysFilePath;
    d_isInInitialization = true;
    d_isChangeMade = false;

    QFile file(d_editingFile);
    if (file.exists())
    {
        file.open(QFile::ReadOnly);
        QTextStream in(&file);
        d_originalSource = in.readAll();
        d_textEditor->setPlainText(d_originalSource);
        d_webView->setHtml(d_originalSource);
        file.close();

        show();
        activateWindow();
        d_saveButton->setEnabled(false);
        updateWindowTitle(false);
        d_isEditing = true;
    }
    else
    {
        d_isEditing = false;
    }
    d_isInInitialization = false;
    return d_isEditing;
}

bool DCUIEditor::saveChange()
{
    bool saved = false;
    QFile file(d_editingFile);
    if (file.open(QFile::WriteOnly | QFile::Truncate))
    {
        QTextStream out(&file);
        out << d_textEditor->toPlainText();
        file.close();
        d_originalSource = d_textEditor->toPlainText();
        updateWindowTitle(false);
        d_isChangeMade = false;
        saved = true;
    }

    return saved;
}

void DCUIEditor::javaScriptWindowObjectCleared()
{
    QWebPage *page = d_webView->page();
    if (page)
    {
        QWebFrame *frame = page->mainFrame();
        if (frame)
        {
            frame->addToJavaScriptWindowObject("engine", d_jsInterface);
        }
    }
}

void DCUIEditor::tabChanged(int index)
{
    if (index == 0)
    {
        d_webView->setHtml(d_textEditor->toPlainText());
    }
    else
    {
        d_textEditor->setFocus();
    }
}

void DCUIEditor::textChanged()
{
    if (!d_isInInitialization && !d_isChangeMade)
    {
         d_isChangeMade = true;
         updateWindowTitle(true);
        d_saveButton->setEnabled(true);
    }
}

void DCUIEditor::updateWindowTitle(bool fileModified)
{
    QFileInfo fileInfo(d_editingFile);
    QString title = fileInfo.fileName();
    title.append(tr(" - Interface editor"));
    if (fileModified)
        title.append(" *");
    setWindowTitle(title);
}
