/*
 * blanco Framework
 * Copyright (C) 2004-2006 WATANABE Yoshinori
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 */
package blanco.commons.sql.format;

import java.util.ArrayList;

import blanco.commons.sql.format.valueobject.BlancoSqlToken;

/**
 * BlancoSqlFormatter: SQL`c[. SQL߂ꂽ[ɏ]`܂B
 * 
 * tH[}bg{邽߂ɂ́A͂SQLSQLƂđÓł邱ƂOƂȂ܂B
 * 
 * ̃NXSQL`̃[ɂẮALURLQƂB
 * http://homepage2.nifty.com/igat/igapyon/diary/2005/ig050613.html
 * 
 * ̃NX SQL͂镔łB<br>
 * 2005.08.12 Tosiki Iga: ̃[eBeB\bh public static܂B<br>
 * 2005.08.12 Tosiki Iga: 65535(Ƃ-1)̓zCgXy[XƂĈ悤ύX܂B
 * 
 * @author WATANABE Yoshinori (a-san) : original version at 2005.07.04.
 * @author IGA Tosiki : marge into blanc Framework at 2005.07.04
 */
public class BlancoSqlParser {

    /**
     * ͑O̕
     */
    private String fBefore;

    /**
     * ͒̕B
     */
    private char fChar;

    /**
     * ͒̈ʒu
     */
    private int fPos;

    /**
     * QȂLB
     * 
     * ȂA|| ͕񌋍ɂ܂B
     */
    private static final String[] twoCharacterSymbol = { "<>", "<=", ">=", "||" };

    /**
     * p[T̃CX^X쐬܂B
     */
    public BlancoSqlParser() {
    }

    /**
     * ^ꂽAzCgXy[Xǂ𔻒肵܂B
     * 
     * @param argChar
     * @return
     */
    public static boolean isSpace(final char argChar) {
        // 2005.07.26 Tosiki Iga \r ͈͂Ɋ܂߂Kv܂B
        // 2005.08.12 Tosiki Iga 65535(Ƃ-1)̓zCgXy[XƂĈ悤ύX܂B
        return argChar == ' ' || argChar == '\t' || argChar == '\n'
                || argChar == '\r' || argChar == 65535;
    }

    /**
     * ƂĔFđÓǂ𔻒肵܂B
     * 
     * SpȂǂƂĔFêƔf܂<br>
     * ̃\bhBlancoSqlEditorPluginQƂ܂B
     * 
     * @param argChar
     * @return
     */
    public static boolean isLetter(final char argChar) {
        // SQLɂ A_[XRA͉p̒Ԃł.
        // blanco ɂ # ͉p̒Ԃł.
        // ɓ{܂߂ȂĂ͂ȂȂB
        // return ('A' <= c && c <= 'Z') || ('a' <= c && c <= 'z')
        // || (c == '_' || c == '#');
        if (isSpace(argChar)) {
            return false;
        }
        if (isDigit(argChar)) {
            return false;
        }
        if (isSymbol(argChar)) {
            return false;
        }
        return true;
    }

    /**
     * ǂ𔻒肵܂B
     * 
     * @param argChar
     * @return
     */
    public static boolean isDigit(final char argChar) {
        return '0' <= argChar && argChar <= '9';
    }

    /**
     * Lǂ𔻒肵܂B
     * 
     * @param argChar
     * @return
     */
    public static boolean isSymbol(final char argChar) {
        switch (argChar) {
        case '"': // double quote
        case '?': // question mark
        case '%': // percent
        case '&': // ampersand
        case '\'': // quote
        case '(': // left paren
        case ')': // right paren
        case '|': // vertical bar
        case '*': // asterisk
        case '+': // plus sign
        case ',': // comma
        case '-': // minus sign
        case '.': // period
        case '/': // solidus
        case ':': // colon
        case ';': // semicolon
        case '<': // less than operator
        case '=': // equals operator
        case '>': // greater than operator

            // blancoł # ͕̈ꕔł case '#':
            // A_[XRA͋LƂ͈܂ case '_': //underscore
            // ȍ~͕̈ۗ̕
            // case '!':
            // case '$':
            // case '[':
            // case '\\':
            // case ']':
            // case '^':
            // case '{':
            // case '}':
            // case '~':
            return true;
        default:
            return false;
        }
    }

    /**
     * g[Nɐi߂܂B
     * 
     * posi߂BsɌʂԂBtypeɂ̎ނݒ肷B
     * 
     * sSQL̏ꍇAO܂B ł́A@`FbN͍sĂȂ_ɒڂĂB
     * 
     * @return g[NԂ.
     */
    BlancoSqlToken nextToken() {
        if (fPos >= fBefore.length()) {
            fPos++;
            return new BlancoSqlToken(BlancoSqlTokenConstants.END, "");
        }

        fChar = fBefore.charAt(fPos);

        if (isSpace(fChar)) {
            String workString = "";
            for (;;) {
                workString += fChar;
                fChar = fBefore.charAt(fPos);
                if (!isSpace(fChar)) {
                    return new BlancoSqlToken(BlancoSqlTokenConstants.SPACE,
                            workString);
                }
                fPos++;
                if (fPos >= fBefore.length()) {
                    return new BlancoSqlToken(BlancoSqlTokenConstants.SPACE,
                            workString);
                }
            }
        } else if (fChar == ';') {
            fPos++;
            // 2005.07.26 Tosiki Iga Z~R͏Ił͂Ȃ悤ɂB
            return new BlancoSqlToken(BlancoSqlTokenConstants.SYMBOL, ";");
        } else if (isDigit(fChar)) {
            String s = "";
            while (isDigit(fChar) || fChar == '.') {
                // if (ch == '.') type = Token.REAL;
                s += fChar;
                fPos++;

                if (fPos >= fBefore.length()) {
                    // 𒴂Ăꍇɂ͏f܂B
                    break;
                }

                fChar = fBefore.charAt(fPos);
            }
            return new BlancoSqlToken(BlancoSqlTokenConstants.VALUE, s);
        } else if (isLetter(fChar)) {
            String s = "";
            // 񒆂̃hbgɂẮAƈ̂ƂčlB
            while (isLetter(fChar) || isDigit(fChar) || fChar == '.') {
                s += fChar;
                fPos++;
                if (fPos >= fBefore.length()) {
                    break;
                }

                fChar = fBefore.charAt(fPos);
            }
            for (int i = 0; i < BlancoSqlConstants.SQL_RESERVED_WORDS.length; i++) {
                if (s
                        .compareToIgnoreCase(BlancoSqlConstants.SQL_RESERVED_WORDS[i]) == 0) {
                    return new BlancoSqlToken(BlancoSqlTokenConstants.KEYWORD,
                            s);
                }
            }
            return new BlancoSqlToken(BlancoSqlTokenConstants.NAME, s);
        }
        // single line comment
        else if (fChar == '-') {
            fPos++;
            char ch2 = fBefore.charAt(fPos);
            // -- ȂƂ
            if (ch2 != '-') {
                return new BlancoSqlToken(BlancoSqlTokenConstants.SYMBOL, "-");
            }
            fPos++;
            String s = "--";
            for (;;) {
                fChar = fBefore.charAt(fPos);
                s += fChar;
                fPos++;
                if (fChar == '\n' || fPos >= fBefore.length()) {
                    return new BlancoSqlToken(BlancoSqlTokenConstants.COMMENT,
                            s);
                }
            }
        }
        // }`CRg
        else if (fChar == '/') {
            fPos++;
            char ch2 = fBefore.charAt(fPos);
            // /* ȂƂ
            if (ch2 != '*') {
                return new BlancoSqlToken(BlancoSqlTokenConstants.SYMBOL, "/");
            }

            String s = "/*";
            fPos++;
            int ch0 = -1;
            for (;;) {
                ch0 = fChar;
                fChar = fBefore.charAt(fPos);
                s += fChar;
                fPos++;
                if (ch0 == '*' && fChar == '/') {
                    return new BlancoSqlToken(BlancoSqlTokenConstants.COMMENT,
                            s);
                }
            }
        } else if (fChar == '\'') {
            fPos++;
            String s = "'";
            for (;;) {
                fChar = fBefore.charAt(fPos);
                s += fChar;
                fPos++;
                if (fChar == '\'') {
                    return new BlancoSqlToken(BlancoSqlTokenConstants.VALUE, s);
                }
            }
        } else if (fChar == '\"') {
            fPos++;
            String s = "\"";
            for (;;) {
                fChar = fBefore.charAt(fPos);
                s += fChar;
                fPos++;
                if (fChar == '\"') {
                    return new BlancoSqlToken(BlancoSqlTokenConstants.NAME, s);
                }
            }
        }

        else if (isSymbol(fChar)) {
            // L
            String s = "" + fChar;
            fPos++;
            if (fPos >= fBefore.length()) {
                return new BlancoSqlToken(BlancoSqlTokenConstants.SYMBOL, s);
            }
            // Q̋Lǂׂ
            char ch2 = fBefore.charAt(fPos);
            for (int i = 0; i < twoCharacterSymbol.length; i++) {
                if (twoCharacterSymbol[i].charAt(0) == fChar
                        && twoCharacterSymbol[i].charAt(1) == ch2) {
                    fPos++;
                    s += ch2;
                    break;
                }
            }
            return new BlancoSqlToken(BlancoSqlTokenConstants.SYMBOL, s);
        } else {
            fPos++;
            return new BlancoSqlToken(BlancoSqlTokenConstants.UNKNOWN, ""
                    + fChar);
        }
    }

    /**
     * SQLg[N̔zɕϊ܂B
     * 
     * @param argSql
     *            ϊOSQL
     * @return Token̔z
     */
    public ArrayList parse(final String argSql) {
        fPos = 0;
        fBefore = argSql;

        final ArrayList list = new ArrayList();
        for (;;) {
            final BlancoSqlToken token = nextToken();
            if (token.getType() == BlancoSqlTokenConstants.END) {
                break;
            }

            list.add(token);
        }
        return list;
    }
}
