/*
 * Copyright (C) 2012 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.ksk.contacts.util;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import com.ksk.contacts.R;
import com.ksk.contacts.tel.util.PriorityNumberDB;
import com.ksk.contacts.tel.util.PriorityNumberDB.PriorityNumber;
import com.ksk.contacts.util.CommonData.GroupListData;
import com.ksk.contacts.util.CommonData.MemberInfo;

import android.content.Context;
import android.database.Cursor;
import android.database.CursorJoiner;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Matrix;
import android.provider.ContactsContract;
import android.provider.ContactsContract.CommonDataKinds;
import android.provider.ContactsContract.CommonDataKinds.Email;
import android.provider.ContactsContract.CommonDataKinds.Event;
import android.provider.ContactsContract.CommonDataKinds.GroupMembership;
import android.provider.ContactsContract.CommonDataKinds.Note;
import android.provider.ContactsContract.CommonDataKinds.Organization;
import android.provider.ContactsContract.CommonDataKinds.Phone;
import android.provider.ContactsContract.CommonDataKinds.Photo;
import android.provider.ContactsContract.CommonDataKinds.SipAddress;
import android.provider.ContactsContract.CommonDataKinds.StructuredName;
import android.provider.ContactsContract.CommonDataKinds.StructuredPostal;
import android.provider.ContactsContract.Data;
import android.provider.ContactsContract.Groups;
import android.provider.ContactsContract.RawContacts;

public class ContactMgr {

	//共通データクラス
	private CommonData mCommonData = new CommonData();

	//Context情報
	private Context mContext;

	public class GroupName {
		public int    id;       //グループID
		public String title;    //グループ名
	}

	//電話番号/Email情報
	public class NumberInfo {
		public String  label;    //ラベル情報(自宅、携帯、会社 等)
		public String  number;   //番号情報
	}

	public class ContactInfo {
		public int              contact_id;     //電話番号
		public int              raw_contact_id; //電話番号
		public Bitmap           photo;          //電話帳登録画像
		public String           dispname;       //名前(表示名)
		public char             yomi = 0;       //フリガナ
		public int              favorite;       //お気に入り
		public List<Integer>    group;          //グループリスト
		public List<NumberInfo> phone;          //電話番号リスト
		public List<NumberInfo> email;          //Emailリスト

		public String           note;
		public String           postal;
		public String           event;

		public String           account_name;   //アカウント名(google,docomo等)
		public int              account_type;   //アカウント種別

		//コンストラクタ
		public ContactInfo(){
			group = new ArrayList<Integer>();
			phone = new ArrayList<NumberInfo>();
			email = new ArrayList<NumberInfo>();
		}
	}

	public ContactMgr(Context context){
		mContext = context;
	}

	public ContactInfo getContactInfo(int contact_id) {

		//任意の電話帳登録メンバー情報の取得
		Cursor cursor = mContext.getContentResolver().query(ContactsContract.Data.CONTENT_URI, null, Data.CONTACT_ID + "=?", new String[]{String.valueOf(contact_id)}, null);

		//cursolよりContact情報を取得
		List<ContactInfo> list = getContactListFromCursol(cursor);

		if(list.size() != 0){
			return list.get(0);

		}else{
			return null;
		}
	}

	public List<ContactInfo> getContactList() {

		//電話帳登録メンバーリストの取得(全員分)
		Cursor cursor = mContext.getContentResolver().query(ContactsContract.Data.CONTENT_URI, null, null, null, Data.RAW_CONTACT_ID + " ASC");

		//cursolよりContact情報を取得
		List<ContactInfo> list = getContactListFromCursol(cursor);

		//フリガナ順にソート
		Collections.sort(list, new DataComparator());

		return list;
	}

	public List<ContactInfo> getFavoriteList() {

		//電話帳登録メンバーリストの取得(全員分)
		Cursor cursor = mContext.getContentResolver().query(ContactsContract.Data.CONTENT_URI
															,null
															,"(" + Data.MIMETYPE + "=? OR " + Data.MIMETYPE + "=?) AND " + StructuredName.STARRED + "='1'"
															,new String[]{ StructuredName.CONTENT_ITEM_TYPE, Photo.CONTENT_ITEM_TYPE }
															,null);

		//cursolよりContact情報を取得
		List<ContactInfo> list = getContactListFromCursol(cursor);

		//フリガナ順にソート
		Collections.sort(list, new DataComparator());

		return list;
	}

	public List<GroupName> getGroupNameList() {

		//グループ名リストの取得
		List<GroupName> list = new ArrayList<GroupName>();

		//ContentProvider経由でグループ名一覧の取得
		//削除済みフラグがONになっていないグループのみを対象とする。
		Cursor cursor = mContext.getContentResolver().query(Groups.CONTENT_URI, new String[] { Groups._ID, Groups.TITLE, Groups.DELETED }, Groups.DELETED + "='0'", null, null);

		//取得失敗、または未登録以外は以下処理を実施する。
		if(cursor != null){
			if(cursor.moveToFirst() != false){

				//カーソルが終端に達するまで処理を継続
				do {
					//グループ情報(ID/タイトル)を設定
					GroupName info = new GroupName();
					info.id    = cursor.getInt(cursor.getColumnIndex(Groups._ID));

					info.title = cursor.getString(cursor.getColumnIndex(Groups.TITLE));

					//グループ名からタグ"System Group: "の除外
					String tag_system_name = mContext.getResources().getString(R.string.Group_System_Name);
					info.title = info.title.replaceAll(tag_system_name + " ", "");    //半角空白も含めて削除

					//グループ名リストに追加する。
					list.add(info);

				} while (cursor.moveToNext());
			}
			cursor.close();
		}

		return list;
	}

	public List<MemberInfo> getGroupMember(Cursor member, int groupId) {

		List<MemberInfo> list = new ArrayList<MemberInfo>();

		//優先番号情報を取得
		PriorityNumberDB PN_DB = new PriorityNumberDB(mContext);

		Cursor group = mContext.getContentResolver().query(Data.CONTENT_URI,
													       new String[]{ GroupMembership.DISPLAY_NAME },
													       Data.MIMETYPE + "=? AND " + GroupMembership.GROUP_ROW_ID + "=?",
													       new String[]{
													           GroupMembership.CONTENT_ITEM_TYPE,
													           String.valueOf(groupId) },
													       GroupMembership.DISPLAY_NAME);
		if(!group.moveToFirst()){
			group.close();
			return null;
		}

		CursorJoiner joiner = new CursorJoiner(member, new String[]{ Data.DISPLAY_NAME }, group, new String[]{ GroupMembership.DISPLAY_NAME });

		for(CursorJoiner.Result result : joiner){
		    switch(result){
			    case BOTH :
			    	MemberInfo data = mCommonData.new MemberInfo();
			    	data.contact_id = member.getInt(member.getColumnIndex(Data.CONTACT_ID));
			    	data.name       = member.getString(member.getColumnIndex(Data.DISPLAY_NAME));
		    		data.photo      = getBitmapCompress(member.getBlob(member.getColumnIndex(ContactsContract.CommonDataKinds.Photo.PHOTO)));

			    	PriorityNumber pn_info = PN_DB.getPriorityNumber(data.contact_id);
			    	if((pn_info != null) && (pn_info.phone != null) && (!pn_info.phone.equals(""))){
			    		data.tel   = pn_info.phone;
	                }
	                if((pn_info != null) && (pn_info.mail != null) && (!pn_info.mail.equals(""))){
	                	data.email = pn_info.mail;
	                }
	                if((pn_info != null) && (pn_info.custom != null) && (!pn_info.custom.equals(""))){
	                	data.custom.send = pn_info.custom;
		                if(pn_info.custom.compareTo("@") >= 0){
		                	data.custom.kind = mCommonData.SEND_KIND_MAIL;
		                }else{
		                	data.custom.kind = mCommonData.SEND_KIND_TEL;
		                }
	                }
	                data.info = getContactInfo(data.contact_id);

			    	list.add(data);
		    }
		}
		group.close();

		return list;
	}

	public List<MemberInfo> getNoGroupMember(Cursor member) {

		List<MemberInfo> list = new ArrayList<MemberInfo>();

		//優先番号情報を取得
		PriorityNumberDB PN_DB = new PriorityNumberDB(mContext);

		Cursor group = mContext.getContentResolver().query(Data.CONTENT_URI,
													       new String[]{ GroupMembership.DISPLAY_NAME },
													       Data.MIMETYPE + "=?",
													       new String[]{
													           GroupMembership.CONTENT_ITEM_TYPE },
													       GroupMembership.DISPLAY_NAME);

		CursorJoiner joiner = new CursorJoiner(member, new String[]{ Data.DISPLAY_NAME }, group, new String[]{ GroupMembership.DISPLAY_NAME });

		for(CursorJoiner.Result result : joiner){
		    switch(result){
			    case LEFT :
			    	MemberInfo info = mCommonData.new MemberInfo();
			    	info.contact_id = member.getInt(member.getColumnIndex(Data.CONTACT_ID));
			    	info.name       = member.getString(member.getColumnIndex(Data.DISPLAY_NAME));
			    	info.photo      = getBitmapCompress(member.getBlob(member.getColumnIndex(ContactsContract.CommonDataKinds.Photo.PHOTO)));

			    	PriorityNumber pn_info = PN_DB.getPriorityNumber(info.contact_id);
			    	if((pn_info != null) && (pn_info.phone != null) && (!pn_info.phone.equals(""))){
			    		info.tel   = pn_info.phone;
	                }
	                if((pn_info != null) && (pn_info.mail != null) && (!pn_info.mail.equals(""))){
	                	info.email = pn_info.mail;
	                }
	                if((pn_info != null) && (pn_info.custom != null) && (!pn_info.custom.equals(""))){
	                	info.custom.send = pn_info.custom;
		                if(pn_info.custom.compareTo("@") >= 0){
		                	info.custom.kind = mCommonData.SEND_KIND_MAIL;
		                }else{
		                	info.custom.kind = mCommonData.SEND_KIND_TEL;
		                }
	                }

			    	list.add(info);
		    }
		}
		group.close();

		return list;
	}
	public List<GroupListData> getGroupMemberList() {

		List<MemberInfo> list = null;

		List<GroupListData> group_list = new ArrayList<GroupListData>();

		//電話帳登録メンバーリストの取得(全員分)
		Cursor cursor = mContext.getContentResolver().query(Data.CONTENT_URI
															,new String[]{ Data.CONTACT_ID, Data.DISPLAY_NAME, Photo.PHOTO }
															,ContactsContract.Data.MIMETYPE + "=?"
															,new String[] { StructuredName.CONTENT_ITEM_TYPE }
															,StructuredName.DISPLAY_NAME);
		if(!cursor.moveToFirst()){
			cursor.close();
			return null;
		}

		//グループ未設定メンバーの登録
		GroupListData group = mCommonData.new GroupListData();
		group.group_id    = -1;
		group.group_title = "未登録";
		list = getNoGroupMember(cursor);
		if(list != null){
			group.member_list.addAll(list);
		}
		group_list.add(group);

		//Group一覧と所属メンバ一覧を取得
		List<GroupName>   GroupList = getGroupNameList();
		if(GroupList != null){
			for(int i=0; i<GroupList.size(); i++){

				list = getGroupMember(cursor, GroupList.get(i).id);

				if(list != null){
					group = mCommonData.new GroupListData();
					group.group_id    = GroupList.get(i).id;
					group.group_title = GroupList.get(i).title;
					group.member_list.addAll(list);
					group_list.add(group);
				}
			}
		}
		cursor.close();

		return group_list;
	}

	public int getContactIDfromNumber(String number) {

		int contact_id = -1;

		//ContentProvider経由で電話番号からContactIDを取得する
		Cursor cursor = mContext.getContentResolver().query(Phone.CONTENT_URI, null, Phone.NUMBER + "=?", new String[]{number}, null);
		if(cursor != null){
			if(cursor.moveToFirst()){
				contact_id = cursor.getInt(cursor.getColumnIndex(Data.CONTACT_ID));
			}
		}
		return contact_id;
	}

	public String getGroupNamefromID(int group_id) {

		String group_name = null;

		List<GroupName> GroupList = getGroupNameList();

		int k = 0;
		for(k=0; k<GroupList.size(); k++){
			if(group_id == GroupList.get(k).id){
				group_name = GroupList.get(k).title;
				break;
			}
		}
		if(k == GroupList.size()){
			group_name = "未分類";
		}


		return group_name;
	}

	private List<ContactInfo> getContactListFromCursol(Cursor cursor) {

		int contact_id     = -1;
		int contact_id_old = -1;

		List<ContactInfo> list = new ArrayList<ContactInfo>();

		//取得失敗、または未登録以外は以下処理を実施する。
		if(cursor != null){
			if(cursor.moveToFirst() != false){

				//カーソルが終端に達するまで処理を継続
				ContactInfo contact = null;
				do{
					//ContactIDを取得する。
					contact_id = cursor.getInt(cursor.getColumnIndex(Data.CONTACT_ID));
					if(contact_id != contact_id_old){
						contact = new ContactInfo();
						list.add(contact);

						contact_id_old = contact_id;
					}
					contact.contact_id = contact_id;
					contact.raw_contact_id = cursor.getInt(cursor.getColumnIndex(Data.RAW_CONTACT_ID));

					//アカウント情報を取得する
					contact.account_name = cursor.getString(cursor.getColumnIndex(RawContacts.ACCOUNT_NAME));
					contact.account_type = cursor.getInt(cursor.getColumnIndex(RawContacts.ACCOUNT_TYPE));

					//MIMETYPEを取得する。
					String mimetype    = cursor.getString(cursor.getColumnIndex(Data.MIMETYPE));

					//MIMETYPEがStructuredNameの場合
					//  →名前、お気に入り等のユーザー情報が格納されている
					if(mimetype.equals(StructuredName.CONTENT_ITEM_TYPE)){
						contact.dispname = cursor.getString(cursor.getColumnIndex(StructuredName.DISPLAY_NAME));

						//フリカナが空(null)でないかをチェックしたうえで格納する。
						String head_str = cursor.getString(cursor.getColumnIndex(StructuredName.PHONETIC_FAMILY_NAME));
						if(head_str == null){
							head_str = cursor.getString(cursor.getColumnIndex(StructuredName.PHONETIC_MIDDLE_NAME));
							if(head_str == null){
								head_str  = cursor.getString(cursor.getColumnIndex(StructuredName.PHONETIC_GIVEN_NAME));
							}
						}
						if(head_str != null){
							contact.yomi = head_str.toUpperCase().charAt(0);
						}

						contact.favorite = cursor.getInt(cursor.getColumnIndex(StructuredName.STARRED));

					//MIMETYPEがEmailの場合
					//  →Email関連情報が格納されている
					}else if(mimetype.equals(Email.CONTENT_ITEM_TYPE)){
						NumberInfo data = new NumberInfo();
						data.number = cursor.getString(cursor.getColumnIndex(Data.DATA1));

						int type    = cursor.getInt(cursor.getColumnIndex(Email.TYPE));
						String[] label = mContext.getResources().getStringArray(R.array.ContactLabel);
						if(type != Email.TYPE_CUSTOM){
							data.label   = label[type];
						}else{
							data.label   = cursor.getString(cursor.getColumnIndex(CommonDataKinds.Email.LABEL));
						}

						contact.email.add(data);

					//MIMETYPEがPhoneの場合
					//  →Phone関連情報が格納されている
					}else if(mimetype.equals(Phone.CONTENT_ITEM_TYPE)){
						NumberInfo data = new NumberInfo();
						data.number = cursor.getString(cursor.getColumnIndex(Data.DATA1));

						int type    = cursor.getInt(cursor.getColumnIndex(Phone.TYPE));
						String[] label = mContext.getResources().getStringArray(R.array.ContactLabel);
						if(type != Phone.TYPE_CUSTOM){
							data.label   = label[type];
						}else{
							data.label   = cursor.getString(cursor.getColumnIndex(Phone.LABEL));
						}

						contact.phone.add(data);

					//MIMETYPEがGroupMembershipの場合
					//  →グループ登録状況などの情報が格納されている
					}else if(mimetype.equals(GroupMembership.CONTENT_ITEM_TYPE)){
						contact.group.add(cursor.getInt(cursor.getColumnIndex(GroupMembership.GROUP_ROW_ID)));

					//MIMETYPEがphotoの場合
					//  →電話帳登録画像情報が格納されている
					}else if(mimetype.equals(Photo.CONTENT_ITEM_TYPE)){
						contact.photo = getBitmapCompress(cursor.getBlob(cursor.getColumnIndex(ContactsContract.CommonDataKinds.Photo.PHOTO)));

					//MIMETYPEがnoteの場合
					//  →ノート情報が格納されている
					}else if(mimetype.equals(Note.CONTENT_ITEM_TYPE)){
						contact.note = cursor.getString(cursor.getColumnIndex(Note.NOTE));

					//MIMETYPEがStructuredPostalの場合
					//  →住所情報が格納されている
					}else if(mimetype.equals(StructuredPostal.CONTENT_ITEM_TYPE)){
						String temp = "";
						contact.postal = "";
						temp = cursor.getString(cursor.getColumnIndex(StructuredPostal.COUNTRY));
						if(temp != null){
							contact.postal += temp;
						}
						temp = cursor.getString(cursor.getColumnIndex(StructuredPostal.REGION));
						if(temp != null){
							contact.postal += temp;
						}
						temp = cursor.getString(cursor.getColumnIndex(StructuredPostal.CITY));
						if(temp != null){
							contact.postal += temp;
						}
						temp = cursor.getString(cursor.getColumnIndex(StructuredPostal.STREET));
						if(temp != null){
							contact.postal += temp;
						}
						temp = cursor.getString(cursor.getColumnIndex(StructuredPostal.POSTCODE));
						if(temp != null){
							contact.postal += temp;
						}

					//MIMETYPEがEventの場合
					//  →誕生日情報が格納されている
					}else if(mimetype.equals(Event.CONTENT_ITEM_TYPE)){
						contact.event = cursor.getString(cursor.getColumnIndex(Event.START_DATE));


					}else if(mimetype.equals(SipAddress.CONTENT_ITEM_TYPE)){
						/* Not Used */

					}else if(mimetype.equals(Organization.CONTENT_ITEM_TYPE)){
						/* Not Used */
					}


				}while(cursor.moveToNext());
			}
			cursor.close();
		}

		//TODO
		//データ整形
		//  →行削除を行うため、終端より検索する
		for(int i=(list.size() - 1); i>=0; i--){
			ContactInfo contact = list.get(i);

			//削除済みデータの削除
			if( ((contact.dispname == null) || (contact.dispname.equals(""))) &&
				(contact.email.size() == 0) &&
				(contact.phone.size() == 0) ){
				list.remove(i);
			}

			//名前未入力時の補正
			if( (contact.dispname == null) || (contact.dispname.equals("")) ) {

				//名前が未入力の状態の場合は、電話番号を代入する。
				//電話番号もない場合は、Emailを入力する。
				if(contact.phone.size() != 0){
					contact.dispname = contact.phone.get(0).number;

				}else if(contact.email.size() != 0){
					contact.dispname = contact.email.get(0).number;

				}else{
					contact.dispname = "Unknown";
				}
			}
			if(contact.yomi == 0) {
				contact.yomi = contact.dispname.toUpperCase().charAt(0);
			}
		}

		return list;
	}

	public void deleteContact(Integer id){
		//TODO：ContactsContract.Dataの削除だけでよさそうだが。。。
		mContext.getContentResolver().delete(ContactsContract.Data.CONTENT_URI, ContactsContract.Data.CONTACT_ID+"="+id, null);
		mContext.getContentResolver().delete(ContactsContract.RawContacts.CONTENT_URI, ContactsContract.RawContacts.CONTACT_ID+"="+id, null);
	}

	public Bitmap getBitmapCompress(byte[] data){
		Bitmap bmp = null;
		Bitmap bmp2 = null;

		if(data == null){
			return null;
		}

		//Bitmapの取得
		bmp = BitmapFactory.decodeByteArray(data, 0, data.length, null);

		//60×60に圧縮
		int w = bmp.getWidth();
		int h = bmp.getHeight();
		float scale = Math.min((float)60/w, (float)60/h);
		Matrix matrix = new Matrix();
		matrix.postScale(scale, scale);
		bmp2 = Bitmap.createBitmap(bmp, 0, 0, w, h, matrix, true);

		return bmp2;
	}

	public String getNameIndex(int c){

		String Index = null;

		/* UTF-16 */

		//ぁ(u3041)～お(u304A)
		//ァ～ォ
		if( ((c >= 0x3041) && (c <= 0x304A)) ||
			((c >= 0xFF71) && (c <= 0xFF75)) ||
			((c >= 0xFF67) && (c <= 0xFF6B)) ){
			Index = "あ";

		//か(u304B)～ご(u3054)
		//カ～コ
		}else if( ((c >= 0x304B) && (c <= 0x3054)) ||
				  ((c >= 0xFF76) && (c <= 0xFF7A)) ){
			Index = "か";

		//さ(u3055)～ぞ(u305E)
		//サ～ソ
		}else if( ((c >= 0x3055) && (c <= 0x305E)) ||
				  ((c >= 0xFF7B) && (c <= 0xFF7F)) ){
			Index = "さ";

		//た(u305F)～ど(u3069)
		//タ～ト
		}else if( ((c >= 0x305F) && (c <= 0x3069)) ||
				  ((c >= 0xFF80) && (c <= 0xFF84)) ){
			Index = "た";

		//な(u306A)～の(u306E)
		//ナ～ノ
		}else if( ((c >= 0x306A) && (c <= 0x306E)) ||
				  ((c >= 0xFF85) && (c <= 0xFF89)) ){
			Index = "な";

		//は(u306F)～ぽ(u307D)
		//ハ～ホ
		}else if( ((c >= 0x306F) && (c <= 0x307D)) ||
				  ((c >= 0xFF8A) && (c <= 0xFF8E)) ){
			Index = "は";

		//ま(u307E)～も(u3082)
		//マ～モ
		}else if( ((c >= 0x307E) && (c <= 0x3082)) ||
				  ((c >= 0xFF8F) && (c <= 0xFF93)) ){
			Index = "ま";

		//ゃ(u3083)～よ(u3088)
		//ャ～ヨ
		}else if( ((c >= 0x3083) && (c <= 0x3088)) ||
				  ((c >= 0xFF94) && (c <= 0xFF96)) ||
				  ((c >= 0xFF6C) && (c <= 0xFF6E)) ){
			Index = "や";

		//ら(u3089)～ろ(u308D)
		//ラ～ロ
		}else if( ((c >= 0x3089) && (c <= 0x308D)) ||
				  ((c >= 0xFF97) && (c <= 0xFF9B)) ){
			Index = "ら";

		//ゎ(u308E)～ん(u3093)
		//ワ～ン
		}else if( ((c >= 0x308E) && (c <= 0x3093)) ||
				  ((c >= 0xFF9C) && (c <= 0xFF9D)) ||
				   (c >= 0xFF66) ){
			Index = "わ";

		//A～z
		}else if( ((c >= 0x0041) && (c <= 0x005A)) ||
				  ((c >= 0x0061) && (c <= 0x007A)) ){
			Index = "" + (char)c;

		//０～９
		}else if((c >= 0x0030) && (c <= 0x0039)){
			Index = "" + (char)c;

		}else{
			Index = "他";

		}

		return Index;
	}
}