/*
 * Aipo is a groupware program developed by Aimluck,Inc.
 * http://aipostyle.com/
 *
 * Copyright(C) 2011 avanza Co.,Ltd. All rights reserved.
 * http://www.avnz.co.jp/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.webmail;

import java.util.List;

import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALStringField;
import com.aimluck.eip.cayenne.om.portlet.EipMMailAccount;
import com.aimluck.eip.common.ALAbstractFormData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALEipGroupSend;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.mail.util.ALMailUtils;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.util.ALEipUtils;
import com.aimluck.eip.webmail.util.WebMailUtils;

/**
 * 携帯Webメールフォームデータの入力内容をセッションにて管理するためのクラスです。 <br />
 * 
 */
public class CellWebMailFormData extends ALAbstractFormData {
  private static final JetspeedLogger logger =
    JetspeedLogFactoryService.getLogger(CellWebMailFormData.class.getName());

  /** 宛名 */
  private ALStringField to = null;

  /** CC */
  private ALStringField cc = null;

  /** BCC */
  private ALStringField bcc = null;

  /** 件名 */
  private ALStringField subject = null;

  /** 本文 */
  private ALStringField body = null;

  private int userId = -1;

  private int accountId = -1;

  /** 開封確認 */
  private ALStringField hasDispositionNotification;

  /** 宛先（グループ送信）hiddenパラメータ */
  private ALStringField toGroupsend;

  /** 宛先（グループ送信）表示名 */
  private ALStringField toGroupsendName;

  /** CC（グループ送信）hiddenパラメータ */
  private ALStringField ccGroupsend;

  /** CC（グループ送信）表示名 */
  private ALStringField ccGroupsendName;

  /** BCC（グループ送信）hiddenパラメータ */
  private ALStringField bccGroupsend;

  /** BCC（グループ送信）表示名 */
  private ALStringField bccGroupsendName;

  /** ログインユーザーのデフォルトアカウント情報 */
  private EipMMailAccount defaultAccount;

  @Override
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    super.init(action, rundata, context);
    userId = ALEipUtils.getUserId(rundata);
    try {
      accountId =
        Integer.parseInt(ALEipUtils.getTemp(
          rundata,
          context,
          WebMailUtils.ACCOUNT_ID));
    } catch (Exception e) {
      accountId = 0;
    }

    // ログインユーザーのデフォルトアカウント取得
    // ポートレット設定よりデフォルトアカウントIDを取得
    String defaultAccountId =
      ALEipUtils
        .getPortlet(rundata, context)
        .getPortletConfig()
        .getInitParameter("p3a-accounts");
    if (null != defaultAccountId && defaultAccountId.length() > 0) {
      // ポートレット設定より取得できた場合
      defaultAccount =
        ALMailUtils.getMailAccount(userId, Integer.parseInt(defaultAccountId));
    } else {
      // ポートレット設定より取得できなかった場合、ユーザーの全アカウントの中で最初に検索されたアカウントとする
      defaultAccountId =
        WebMailUtils
          .getMailAccountNameList(userId)
          .get(0)
          .getAccountId()
          .toString();
      try {
        defaultAccount =
          WebMailUtils.getMailAccount(Integer.parseInt(defaultAccountId));
      } catch (Exception e) {
        logger.error("(携帯・スマートフォン)Webメールポートレットデフォルトアカウントの取得に失敗しました。 "
          + ALEipUtils.getUserFullName(userId), e);
        throw new ALPageNotFoundException();
      }

    }
  }

  /**
   * フィールドの初期化をします。
   * 
   * @see com.aimluck.eip.common.ALAbstractFormData#initField()
   */
  @Override
  public void initField() {
    // フィールドの初期化
    // To
    to = new ALStringField();
    // CC
    cc = new ALStringField();
    // BCC
    bcc = new ALStringField();
    // Subject
    subject = new ALStringField();
    // Body
    body = new ALStringField();
    body.setFieldName("本文");
    body.setTrim(false);
    // 開封要求フラグ
    hasDispositionNotification = new ALStringField();
    // 宛先（グループ送信）hiddenパラメータ
    toGroupsend = new ALStringField();
    // 宛先（グループ送信）表示名
    toGroupsendName = new ALStringField();
    // CC（グループ送信）hiddenパラメータ
    ccGroupsend = new ALStringField();
    // CC（グループ送信）表示名
    ccGroupsendName = new ALStringField();
    // BCC（グループ送信）hiddenパラメータ
    bccGroupsend = new ALStringField();
    // BCC（グループ送信）表示名
    bccGroupsendName = new ALStringField();
  }

  @Override
  /**
   * アドレス帳で選択した内容とセッションから取得したメール作成フォーム内容を メールフォームにセットします。
   * 
   * @see com.aimluck.eip.common.ALAbstractFormData#setFormData(RunData,
   *      context, List<String>)
   * @param action
   * @param rundata
   * @throws ALPageNotFoundException
   * @throws ALDBErrorException
   * @return 成功:true 失敗:false
   */
  protected boolean setFormData(RunData rundata, Context context,
      List<String> msgList) throws ALPageNotFoundException, ALDBErrorException {
    boolean res = super.setFormData(rundata, context, msgList);
    if (accountId <= 0 || userId <= 0) {
      return res;
    }
    try {

      // 宛先タイプの設定値("to","cc","bcc")
      String mail_sent_type =
        ALEipUtils.getTemp(rundata, context, WebMailConsts.MAIL_SENT_TYPE);
      // グループ送信フラグ(グループ送信の場合は"true",それ以外はnull)
      String group_send_flg =
        rundata.getParameters().getString("group_send_flg");

      // アドレス帳でチェックされたオブジェクトのvalue値
      // グループ送信フラグが"true"の場合はvalue値はグループID
      // それ以外は個人メールアドレス
      ALStringField checked_value;
      checked_value = new ALStringField();

      // グループ送信表示名
      ALStringField group_name;
      group_name = new ALStringField();

      // 各フィールドにセッション格納値をセット
      to.setValue(ALEipUtils.getTemp(rundata, context, WebMailConsts.STR_TO));
      toGroupsendName.setValue(ALEipUtils.getTemp(
        rundata,
        context,
        WebMailConsts.STR_TO_GROUPSEND_NAME));
      toGroupsend.setValue(ALEipUtils.getTemp(
        rundata,
        context,
        WebMailConsts.STR_TO_GROUPSEND));
      cc.setValue(ALEipUtils.getTemp(rundata, context, WebMailConsts.STR_CC));
      ccGroupsendName.setValue(ALEipUtils.getTemp(
        rundata,
        context,
        WebMailConsts.STR_CC_GROUPSEND_NAME));
      ccGroupsend.setValue(ALEipUtils.getTemp(
        rundata,
        context,
        WebMailConsts.STR_CC_GROUPSEND));
      bcc.setValue(ALEipUtils.getTemp(rundata, context, WebMailConsts.STR_BCC));
      bccGroupsendName.setValue(ALEipUtils.getTemp(
        rundata,
        context,
        WebMailConsts.STR_BCC_GROUPSEND_NAME));
      bccGroupsend.setValue(ALEipUtils.getTemp(
        rundata,
        context,
        WebMailConsts.STR_BCC_GROUPSEND));
      subject.setValue(ALEipUtils.getTemp(
        rundata,
        context,
        WebMailConsts.STR_SUBJECT));
      body.setValue(ALEipUtils
        .getTemp(rundata, context, WebMailConsts.STR_BODY));
      hasDispositionNotification.setValue(ALEipUtils.getTemp(
        rundata,
        context,
        WebMailConsts.STR_HAS_DISPOSITION_NOTIFICATION));

      Object[] objs = rundata.getParameters().getKeys();
      int length = objs.length;
      boolean first_flg = true;
      if ("true".equals(group_send_flg)) {
        // アドレス帳でチェックされたオブジェクトのvalue値を取得(グループ送信の場合)
        for (int i = 0; i < length; i++) {
          if (objs[i].toString().startsWith("check")) {
            String str = rundata.getParameters().getString(objs[i].toString());
            // グループ送信の場合は表示名を取得しセット(複数の場合はカンマ区切り)
            ALEipGroupSend eipGroupSend =
              ALEipUtils.getGroupSend(Integer.valueOf(str));
            String str_group_name = eipGroupSend.getGroupSendName().getValue();
            // チェックされたオブジェクトのvalue値(グループID)をセット
            // 複数の場合はカンマ区切り
            // チェックされたオブジェクトのvalue値(グループID)をセット
            // 複数の場合はカンマ区切り
            if (first_flg) {
              checked_value.setValue("g " + str + " " + str_group_name);
              group_name.setValue(str_group_name);
              first_flg = false;
            } else {
              checked_value.setValue(checked_value.getValue()
                + "\n"
                + "g "
                + str
                + " "
                + str_group_name);
              group_name.setValue(group_name.getValue() + "," + str_group_name);
            }
          }
        }
        // 宛先タイプの設定値から設定先フィルードを決定し、
        // チェックされたオブジェクトのvalue値と表示名をセットする。
        if (WebMailConsts.STR_TO.endsWith(mail_sent_type)) {
          toGroupsend.setValue(checked_value.getValue());
          toGroupsendName.setValue(group_name.getValue());
        } else if (WebMailConsts.STR_CC.endsWith(mail_sent_type)) {
          ccGroupsend.setValue(checked_value.getValue());
          ccGroupsendName.setValue(group_name.getValue());
        } else {
          bccGroupsend.setValue(checked_value.getValue());
          bccGroupsendName.setValue(group_name.getValue());
        }
      } else {
        // アドレス帳でチェックされたオブジェクトのvalue値を取得(グループ送信以外の場合)
        boolean no_check = true;
        for (int i = 0; i < length; i++) {
          if (objs[i].toString().startsWith("check")) {
            no_check = false;
            String str = rundata.getParameters().getString(objs[i].toString());
            // チェックされたオブジェクトのvalue値(メールアドレス)をセット
            // 複数の場合はカンマ区切り
            if (first_flg) {
              checked_value.setValue(str);
              first_flg = false;
            } else {
              checked_value.setValue(checked_value.getValue() + "," + str);
            }
          }
        }
        if (no_check
          && (rundata.getParameters().containsKey("set_personal_flg") || rundata
            .getParameters()
            .containsKey("set_corp_flg"))) {
          msgList.add("『 アドレスを選択してください。 』");
          return false;
        }
        // 宛先タイプの設定値から設定先フィルードを決定し、
        // チェックされたオブジェクトのvalue値をセットする。
        if (!(null == checked_value.getValue() || "".equals(checked_value
          .getValue()))) {
          if (WebMailConsts.STR_TO.endsWith(mail_sent_type)) {
            if (null == to.getValue() || "".equals(to.getValue())) {
              to.setValue(checked_value.getValue());
            } else {
              to.setValue(to.getValue() + "," + checked_value.getValue());
            }
          } else if (WebMailConsts.STR_CC.endsWith(mail_sent_type)) {
            if (null == cc.getValue() || "".equals(cc.getValue())) {
              cc.setValue(checked_value.getValue());
            } else {
              cc.setValue(cc.getValue() + "," + checked_value.getValue());
            }
          } else {
            if (null == bcc.getValue() || "".equals(bcc.getValue())) {
              bcc.setValue(checked_value.getValue());
            } else {
              bcc.setValue(bcc.getValue() + "," + checked_value.getValue());
            }
          }
        }
      }
    } catch (Exception ex) {
      String msg = "メール情報の取得に失敗しました。";
      logger.error("(携帯・スマートフォン)Webメールポートレット メール情報の取得に失敗しました。"
        + ALEipUtils.getUserFullName(userId), ex);
      msgList.add(msg);
      return false;
    }
    return res;
  }

  /**
   * 携帯画面フィールド用 メールアドレス変換処理
   * 
   * @param addrs_filed
   *            変換前メールアドレス
   * @return 変換後メールアドレス
   */
  public String getAddrForCell(ALStringField addrs_filed) {
    return WebMailFormData.getAddrForCell(addrs_filed);
  }

  /**
   * 処理中のアカウントを返す。<BR>
   * 
   * @return EipMMailAccount アカウント情報
   * @throws Exception
   */
  public EipMMailAccount getDefaultAccount() throws Exception {
    EipMMailAccount selectedAccount =
      ALMailUtils.getMailAccount(userId, accountId);
    if (null == selectedAccount) {
      return defaultAccount;
    }
    return selectedAccount;
  }

  /**
   * 宛先メールアドレスを返す。
   * 
   * @return メールアドレス
   */
  public ALStringField getTo() {
    return to;
  }

  /**
   * 宛先（グループ送信）表示名を返す。
   * 
   * @return 宛先（グループ送信）表示名
   */
  public ALStringField getToGroupsendName() {
    return toGroupsendName;
  }

  /**
   * 「宛先（グループ送信）」hiddenパラメータ（グループID）を返す。
   * 
   * @return 「宛先（グループ送信）」hiddenパラメータ（グループID）
   */
  public ALStringField getToGroupsend() {
    return toGroupsend;
  }

  /**
   * CCメールアドレスを返す。
   * 
   * @return メールアドレス
   */
  public ALStringField getCc() {
    return cc;
  }

  /**
   * CC（グループ送信）表示名を返す。
   * 
   * @return CC（グループ送信）表示名
   */
  public ALStringField getCcGroupsendName() {
    return ccGroupsendName;
  }

  /**
   * CC（グループ送信）hiddenパラメータを返す。
   * 
   * @return 「CC（グループ送信）」hiddenパラメータ（グループID）
   */
  public ALStringField getCcGroupsend() {
    return ccGroupsend;
  }

  /**
   * BCCメールアドレスを返す。
   * 
   * @return メールアドレス
   */
  public ALStringField getBcc() {
    return bcc;
  }

  /**
   * BCC（グループ送信）表示名を返す。
   * 
   * @return BCC（グループ送信）表示名
   */
  public ALStringField getBccGroupsendName() {
    return bccGroupsendName;
  }

  /**
   * BCC（グループ送信）hiddenパラメータを返す。
   * 
   * @return 「BCC（グループ送信）」hiddenパラメータ（グループID）
   */
  public ALStringField getBccGroupsend() {
    return bccGroupsend;
  }

  /**
   * 件名を返す。
   * 
   * @return 件名
   */
  public ALStringField getSubject() {
    return subject;
  }

  /**
   * メール本文を返す。
   * 
   * @return メール本文
   */
  public ALStringField getBody() {
    return body;
  }

  /**
   * 開封確認フラグを返す。<BR>
   * 
   * @return 開封確認チェックON:"ON",開封確認チェックOFF:""
   */
  public ALStringField getHasDispositionNotification() {
    return hasDispositionNotification;
  }

  @Override
  protected void setValidator() {
    // Validateの定義必要なし
  }

  @Override
  public boolean validate(List<String> msgList) {
    // このメソッドは利用されません。
    return (msgList.size() == 0);
  }

  @Override
  protected boolean deleteFormData(RunData rundata, Context context,
      List<String> msgList) {
    // このメソッドは利用されません。
    return true;
  }

  @Override
  protected boolean insertFormData(RunData rundata, Context context,
      List<String> msgList) {
    // このメソッドは利用されません。
    return true;
  }

  @Override
  protected boolean loadFormData(RunData rundata, Context context,
      List<String> msgList) throws ALDBErrorException {
    // このメソッドは利用されません。
    return true;
  }

  @Override
  protected boolean updateFormData(RunData rundata, Context context,
      List<String> msgList) throws ALPageNotFoundException, ALDBErrorException {
    // このメソッドは利用されません。
    return false;
  }
}
