/*
 * Aipo is a groupware program developed by Aimluck,Inc.
 * Copyright (C) 2004-2011 Aimluck,Inc.
 * http://www.aipo.com
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.addressbook;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALDateField;
import com.aimluck.commons.field.ALStringField;
import com.aimluck.eip.addressbook.util.AddressBookUtils;
import com.aimluck.eip.addressbookuser.util.AddressBookUserUtils;
import com.aimluck.eip.cayenne.om.portlet.EipMAddressGroup;
import com.aimluck.eip.cayenne.om.portlet.EipMAddressbook;
import com.aimluck.eip.cayenne.om.portlet.EipTAddressbookGroupMap;
import com.aimluck.eip.common.ALAbstractFormData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALEipConstants;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.orm.query.SelectQuery;
import com.aimluck.eip.services.accessctl.ALAccessControlConstants;
import com.aimluck.eip.services.eventlog.ALEventlogConstants;
import com.aimluck.eip.services.eventlog.ALEventlogFactoryService;
import com.aimluck.eip.util.ALEipUtils;

/**
 * アドレス帳グループの入力用フォームデータです。
 * 
 */
public class AddressBookGroupFormData extends ALAbstractFormData {
  private static final JetspeedLogger logger =
    JetspeedLogFactoryService.getLogger(AddressBookGroupFormData.class
      .getName());

  // グループオブジェクトのリスト
  // private List<EipMAddressGroup> groupList;

  // アドレスオブジェクトのリスト(全体表示用)
  // private List<AddressBookFilterData> allAddressList;

  // アドレスオブジェクトのリスト(グループ別表示用)
  // private List<Object> addressList;

  // このグループへ登録されているアドレスのリスト
  private List<AddressBookResultData> addresses;

  private ALStringField group_name;

  private ALStringField public_flag;

  private ALDateField create_date;

  private ALDateField update_date;

  private Integer loginuserid;

  private Integer gid;

  @Override
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    super.init(action, rundata, context);

    loginuserid = Integer.valueOf(ALEipUtils.getUserId(rundata));
  }

  /**
   * 画面フィールド初期化
   * 
   * @see com.aimluck.eip.common.ALData#initField()
   */
  @Override
  public void initField() {
    // groupList = new ArrayList<EipMAddressGroup>();
    // allAddressList = new ArrayList<AddressBookFilterData>();
    // addressList = new ArrayList<Object>();
    addresses = new ArrayList<AddressBookResultData>();

    group_name = new ALStringField();
    // change by motegi start 個人アドレス帳
    // group_name.setFieldName("グループ名");
    group_name.setFieldName("会社名");
    // change end
    group_name.setTrim(true);

    public_flag = new ALStringField();
    public_flag.setFieldName("公開区分");

    create_date = new ALDateField();
    create_date.setFieldName("登録日");

    update_date = new ALDateField();
    update_date.setFieldName("最終更新日");

  }

  /**
   * 
   */
  @Override
  protected void setValidator() {
    group_name.setNotNull(true);
    group_name.limitMaxLength(50);
    public_flag.setNotNull(true);
    public_flag.limitMaxLength(1);
  }

  /**
   * 
   * @param msgList
   * @return
   */
  @Override
  protected boolean validate(List<String> msgList) {
    try {
      SelectQuery<EipMAddressGroup> query =
        Database.query(EipMAddressGroup.class);
      if (ALEipConstants.MODE_INSERT.equals(getMode())) {
        Expression exp1 =
          ExpressionFactory.matchExp(
            EipMAddressGroup.GROUP_NAME_PROPERTY,
            group_name.getValue());
        query.setQualifier(exp1);
        Expression exp2 =
          ExpressionFactory.matchExp(
            EipMAddressGroup.OWNER_ID_PROPERTY,
            loginuserid);
        query.andQualifier(exp2);
      } else {

        Expression exp1 =
          ExpressionFactory.matchExp(
            EipMAddressGroup.GROUP_NAME_PROPERTY,
            group_name.getValue());
        query.setQualifier(exp1);
        Expression exp2 =
          ExpressionFactory.matchExp(
            EipMAddressGroup.OWNER_ID_PROPERTY,
            loginuserid);
        query.andQualifier(exp2);
        Expression exp3 =
          ExpressionFactory.noMatchDbExp(
            EipMAddressGroup.GROUP_ID_PK_COLUMN,
            gid);
        query.andQualifier(exp3);
      }
      if (query.fetchList().size() != 0) {
        msgList.add("社外グループ名『 <span class='em'>"
          + group_name
          + "</span> 』は既に登録されています。");
      }

      group_name.validate(msgList);
      public_flag.validate(msgList);
    } catch (Exception ex) {
      // change start
      // logger.error("Exception", ex);
      String mode = "登録";
      if (ALEipConstants.MODE_UPDATE.equals(getMode())) {
        mode = "更新";
      }
      String msg =
        "会社情報の"
          + mode
          + "に失敗しました。["
          + ALEipUtils.getBaseUser(loginuserid).getUserName()
          + "]";
      logger.error(msg, ex);
      msgList.add(msg);
      // change end
      return false;

    }
    return msgList.size() == 0;
  }

  /**
   * フォームへデータをセットします。
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return
   * @throws ALPageNotFoundException
   * @throws ALDBErrorException
   */
  @Override
  protected boolean setFormData(RunData rundata, Context context,
      List<String> msgList) throws ALPageNotFoundException, ALDBErrorException {
    // ユーザ登録グループの公開区分は常に非公開(F)とする
    setPublicFlag(new ALStringField("F"));
    boolean res = super.setFormData(rundata, context, msgList);
    // addressList = new ArrayList<Object>();
    if (res) {
      try {
        if (ALEipConstants.MODE_UPDATE.equals(getMode())) {
          gid =
            Integer.valueOf(ALEipUtils.getTemp(
              rundata,
              context,
              ALEipConstants.ENTITY_ID));
        }

        String str[] = rundata.getParameters().getStrings("address_to");
        if (hasEmpty(str)) {
          return res;
        }

        SelectQuery<EipMAddressbook> query =
          Database.query(EipMAddressbook.class);
        Expression exp =
          ExpressionFactory.inDbExp(EipMAddressbook.ADDRESS_ID_PK_COLUMN, str);
        query.setQualifier(exp);

        List<EipMAddressbook> list = query.fetchList();
        int size = list.size();
        for (int i = 0; i < size; i++) {
          EipMAddressbook address = list.get(i);
          AddressBookResultData rd = new AddressBookResultData();
          rd.initField();
          rd.setAddressId(address.getAddressId().intValue());
          rd.setName(address.getLastName() + " " + address.getFirstName());
          addresses.add(rd);
        }
      } catch (Exception ex) {
        logger.error("Exception", ex);
        throw new ALDBErrorException();
      }
    }
    return res;
  }

  private boolean hasEmpty(String[] list) {
    if (list == null || list.length == 0) {
      return true;
    }

    String str = null;
    int len = list.length;
    for (int i = 0; i < len; i++) {
      str = list[i];
      if (str == null || "".equals(str)) {
        return true;
      }
    }
    return false;
  }

  /**
   * 個人アドレス情報編集画面表示処理
   * 
   * @param rundata
   *            JetSpeedランデータ
   * @param context
   *            JetSpeedコンテキスト
   * @param msgList
   *            エラーメッセージリスト
   * @return 処理結果(正常：true、異常:false)
   * 
   * @throws ALDBErrorException
   * @see com.aimluck.eip.common.ALAbstractFormData#loadFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean loadFormData(RunData rundata, Context context,
      List<String> msgList) throws ALDBErrorException {
    try {
      // オブジェクトモデルを取得
      EipMAddressGroup group =
        AddressBookUtils.getEipMAddressGroup(rundata, context);
      if (group == null) {
        // change start
        // return false;
        throw new ALDBErrorException();
        // change end
      }
      // グループ名
      group_name.setValue(group.getGroupName());
      // 公開区分
      public_flag.setValue(group.getPublicFlag());
    } catch (Exception ex) {
      // change start
      // return false;
      String msg =
        "会社情報の編集画面の表示に失敗しました。["
          + ALEipUtils.getBaseUser(loginuserid).getUserName()
          + "]";
      logger.error(msg, ex);
      throw new ALDBErrorException();
      // change end
    }
    return true;
  }

  /**
   * 指定社外グループに属するアドレスを取得する（選択済み個人アドレスリスト用データの取得を行なう）。
   * 
   * @param rundata
   * @param context
   * @return
   */
  public boolean loadAddresses(RunData rundata, Context context) {
    try {
      String gid =
        ALEipUtils.getTemp(rundata, context, ALEipConstants.ENTITY_ID);
      if (gid != null) {
        // この社外グループの所有者が現ユーザと異なる場合はエラー
        SelectQuery<EipMAddressGroup> query =
          Database.query(EipMAddressGroup.class);
        Expression exp1 =
          ExpressionFactory.matchDbExp(
            EipMAddressGroup.GROUP_ID_PK_COLUMN,
            Integer.valueOf(gid));
        query.setQualifier(exp1);
        Expression exp2 =
          ExpressionFactory.matchExp(
            EipMAddressGroup.OWNER_ID_PROPERTY,
            Integer.valueOf(ALEipUtils.getUserId(rundata)));
        query.andQualifier(exp2);

        List<EipMAddressGroup> list = query.fetchList();
        if (list.size() == 0) {
          return false;
        }
      } else {
        return false;
      }

      // 指定社外グループに属するアドレスのリストを取得
      SelectQuery<EipTAddressbookGroupMap> mapquery =
        Database.query(EipTAddressbookGroupMap.class);
      Expression mapexp =
        ExpressionFactory.matchDbExp(
          EipTAddressbookGroupMap.EIP_TADDRESS_GROUP_PROPERTY
            + "."
            + EipMAddressGroup.GROUP_ID_PK_COLUMN,
          Integer.valueOf(gid));
      mapquery.setQualifier(mapexp);
      // add start
      mapquery.orderAscending(EipTAddressbookGroupMap.EIP_MADDRESSBOOK_PROPERTY
        + "."
        + EipMAddressbook.LAST_NAME_KANA_PROPERTY);
      // add end

      List<EipTAddressbookGroupMap> aList = mapquery.fetchList();

      int size = aList.size();
      for (int i = 0; i < size; i++) {
        EipTAddressbookGroupMap record = aList.get(i);
        EipMAddressbook addressbook =
          Database.get(EipMAddressbook.class, record.getAddressId());

        AddressBookResultData rd = new AddressBookResultData();
        rd.initField();
        rd.setAddressId(record.getAddressId().intValue());

        // change start
        // rd
        // .setName(addressbook.getLastName() + " " +
        // addressbook.getFirstName());
        rd.setName(AddressBookUtils.makeFullName(
          addressbook.getLastName(),
          addressbook.getFirstName()));
        // change end
        addresses.add(rd);
      }

    } catch (Exception ex) {
      // change start
      // logger.error("Exception", ex);
      String msg =
        "会社情報の編集画面の表示に失敗しました。["
          + ALEipUtils.getBaseUser(loginuserid).getUserName()
          + "]";
      logger.error(msg, ex);
      // change end
      return false;
    }
    return true;
  }

  /**
   * 個人アドレス情報登録処理
   * 
   * @param rundata
   *            JetSpeedランデータ
   * @param context
   *            JetSpeedコンテキスト
   * @param msgList
   *            エラーメッセージリスト
   * @return 処理結果(正常：true、異常:false)
   * 
   * @see com.aimluck.eip.common.ALAbstractFormData#insertFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean insertFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {
      EipMAddressGroup address = Database.create(EipMAddressGroup.class);
      rundata.getParameters().setProperties(address);
      address.setGroupName(group_name.getValue());
      address.setPublicFlag(public_flag.getValue());

      int uid = ALEipUtils.getUserId(rundata);
      address.setOwnerId(Integer.valueOf(uid));

      Date now = new Date();
      address.setCreateDate(now);
      address.setUpdateDate(now);

      // Address-Groupマッピングテーブルへのデータ追加
      for (int i = 0; i < addresses.size(); i++) {

        int addressid =
          Integer.valueOf((int) (addresses.get(i).getAddressId().getValue()));

        // add by motegi
        SelectQuery<EipTAddressbookGroupMap> query2 =
          Database.query(EipTAddressbookGroupMap.class);
        Expression exp3 =
          ExpressionFactory.matchExp(
            EipTAddressbookGroupMap.ADDRESS_ID_PROPERTY,
            addressid);
        query2.setQualifier(exp3);

        List<EipTAddressbookGroupMap> maps = query2.fetchList();
        Database.deleteAll(maps);
        // add end

        EipTAddressbookGroupMap map =
          Database.create(EipTAddressbookGroupMap.class);

        map.setEipMAddressbook(Database.get(EipMAddressbook.class, Integer
          .valueOf(addressid)));
        map.setEipTAddressGroup(address);
      }

      Database.commit();

      // イベントログに保存
      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
        address.getGroupId(),
        ALEventlogConstants.PORTLET_TYPE_ADDRESSBOOK_GROUP,
        group_name.getValue());

      return true;
    } catch (Exception ex) {
      Database.rollback();
      String msg =
        "会社情報の登録に失敗しました。["
          + ALEipUtils.getBaseUser(loginuserid).getUserName()
          + "]";
      logger.error(msg, ex);
      msgList.add(msg);
      return false;
    }
  }

  /**
   * 個人アドレス情報更新処理
   * 
   * @param rundata
   *            JetSpeedランデータ
   * @param context
   *            JetSpeedコンテキスト
   * @param msgList
   *            エラーメッセージリスト
   * @return 処理結果(正常：true、異常:false)
   * 
   * @see com.aimluck.eip.common.ALAbstractFormData#updateFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean updateFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {
      // オブジェクトモデルを取得
      EipMAddressGroup group =
        AddressBookUtils.getEipMAddressGroup(rundata, context);
      if (group == null) {
        // change start
        // return false;
        throw new Exception("指定されたアドレス帳社外グループがデータベースにありません。");
        // change end
      }
      // add start
      if (group.getOwnerId().intValue() != loginuserid.intValue()) {
        throw new Exception("自分の登録した以外のアドレス帳社外グループの更新はできません。対象オーナーID="
          + group.getOwnerId().intValue());
      }
      // add end

      group.setGroupName(group_name.getValue());
      group.setPublicFlag(public_flag.getValue());
      group.setUpdateDate(new Date());

      // Address-Groupマッピングテーブルへのデータ追加

      String gid =
        ALEipUtils.getTemp(rundata, context, ALEipConstants.ENTITY_ID);

      // Address-Group Mapテーブル情報を一旦削除
      SelectQuery<EipTAddressbookGroupMap> query1 =
        Database.query(EipTAddressbookGroupMap.class);
      Expression exp1 =
        ExpressionFactory.matchDbExp(
          EipTAddressbookGroupMap.EIP_TADDRESS_GROUP_PROPERTY
            + "."
            + EipMAddressGroup.GROUP_ID_PK_COLUMN,
          gid);
      query1.setQualifier(exp1);

      List<EipTAddressbookGroupMap> maps = query1.fetchList();
      Database.deleteAll(maps);

      // add start
      List<Integer> addressIds = new ArrayList<Integer>();
      for (EipTAddressbookGroupMap map : maps) {
        addressIds.add(map.getAddressId());
      }

      // 更新対象の社外グループに所属していた個人アドレスは、一旦”未所属”グループ
      // に付け替える
      for (Integer n : addressIds) {

        boolean isSelected = false;
        for (AddressBookResultData r : addresses) {
          if (r.getAddressId().getValue() == n.longValue()) {
            isSelected = true;
            break;
          }
        }
        if (isSelected) {
          // 今次選択済みメンバーは”未所属”グループに付け替えない。
          continue;
        }

        EipTAddressbookGroupMap map =
          Database.create(EipTAddressbookGroupMap.class);
        map.setEipMAddressbook(Database.get(EipMAddressbook.class, n));
        map.setEipTAddressGroup(AddressBookUserUtils
          .getDefaultEipMAddressGroup());
      }
      // add end

      // Address-Group Mapテーブルへ指定されたアドレスを追加
      for (int i = 0; i < addresses.size(); i++) {

        // add start
        // 元々が未所属の場合はそのレコードを削除
        EipTAddressbookGroupMap orgMap =
          Database.get(
            EipTAddressbookGroupMap.class,
            EipTAddressbookGroupMap.ADDRESS_ID_COLUMN,
            addresses.get(i).getAddressId().getValue());
        if (orgMap != null) {
          Database.delete(orgMap);
        }

        // add end

        EipTAddressbookGroupMap map =
          Database.create(EipTAddressbookGroupMap.class);
        int addressid =
          Integer.valueOf((int) (addresses.get(i).getAddressId().getValue()));
        map.setEipMAddressbook(Database.get(EipMAddressbook.class, Integer
          .valueOf(addressid)));
        map.setEipTAddressGroup(group);
      }

      Database.commit();

      // イベントログに保存
      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
        group.getGroupId(),
        ALEventlogConstants.PORTLET_TYPE_ADDRESSBOOK_GROUP,
        group_name.getValue());

    } catch (Exception ex) {
      Database.rollback();
      String msg =
        "会社情報の更新に失敗しました。["
          + ALEipUtils.getBaseUser(loginuserid).getUserName()
          + "]";
      logger.error(msg, ex);
      msgList.add(msg);
      return false;
    }
    return true;
  }

  /**
   * 個人アドレス情報削除処理
   * 
   * @param rundata
   *            JetSpeedランデータ
   * @param context
   *            JetSpeedコンテキスト
   * @param msgList
   *            エラーメッセージリスト
   * @return 処理結果(正常：true、異常:false)
   * 
   * @see com.aimluck.eip.common.ALAbstractFormData#deleteFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean deleteFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {
      String groupid =
        ALEipUtils.getTemp(rundata, context, ALEipConstants.ENTITY_ID);
      if (groupid == null || Integer.valueOf(groupid) == null) {
        logger.debug("[AddressBook] Cannot find Ext Group ID .");
        // change start
        // return false;
        throw new Exception("指定されたアドレス帳社外グループがデータベースにありません。");
        // change end
      }

      // Eip_t_addressbook_group_Mapテーブルから削除
      SelectQuery<EipTAddressbookGroupMap> query1 =
        Database.query(EipTAddressbookGroupMap.class);
      Expression exp1 =
        ExpressionFactory.matchDbExp(
          EipTAddressbookGroupMap.EIP_TADDRESS_GROUP_PROPERTY
            + "."
            + EipMAddressGroup.GROUP_ID_PK_COLUMN,
          Integer.valueOf(groupid));
      query1.setQualifier(exp1);

      List<EipTAddressbookGroupMap> maps = query1.fetchList();

      // add by motegi start 個人アドレス帳対応
      List<Integer> addressIds = new ArrayList<Integer>();
      for (EipTAddressbookGroupMap map : maps) {
        addressIds.add(map.getAddressId());
      }

      // 削除対象の社外グループに所属していた個人アドレスは、”未所属”グループ
      // に付け替える
      for (Integer n : addressIds) {
        EipTAddressbookGroupMap map =
          Database.create(EipTAddressbookGroupMap.class);
        map.setEipMAddressbook(Database.get(EipMAddressbook.class, n));
        map.setEipTAddressGroup(AddressBookUserUtils
          .getDefaultEipMAddressGroup());
      }
      //

      Database.deleteAll(maps);

      // Eip_m_address_groupテーブルから削除
      SelectQuery<EipMAddressGroup> query2 =
        Database.query(EipMAddressGroup.class);
      Expression exp2 =
        ExpressionFactory.matchDbExp(
          EipMAddressGroup.GROUP_ID_PK_COLUMN,
          Integer.valueOf(groupid));
      query2.setQualifier(exp2);
      // add start
      Expression exp3 =
        ExpressionFactory.matchExp(
          EipMAddressGroup.OWNER_ID_PROPERTY,
          loginuserid);
      query2.andQualifier(exp3);
      // add end

      List<EipMAddressGroup> groups = query2.fetchList();
      // add start
      if (groups == null || groups.size() == 0) {
        throw new Exception("削除対象の会社情報レコードが存在しません。GROUP_ID=" + groupid);
      }
      // add end
      EipMAddressGroup delete_group = groups.get(0);
      int entityId = delete_group.getGroupId();
      String groupName = delete_group.getGroupName();
      Database.deleteAll(groups);

      Database.commit();

      // イベントログに保存
      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
        entityId,
        ALEventlogConstants.PORTLET_TYPE_ADDRESSBOOK_GROUP,
        groupName);

      // 検索画面用フィルタにて設定されているグループフィルタをセッションから削除する。
      String filtername =
        AddressBookFilterdSelectData.class.getName()
          + ALEipConstants.LIST_FILTER;
      ALEipUtils.removeTemp(rundata, context, filtername);
    } catch (Exception ex) {
      String msg =
        "会社情報の削除処理に失敗しました。["
          + ALEipUtils.getBaseUser(loginuserid).getUserName()
          + "]";
      logger.error(msg, ex);
      msgList.add(msg);
      return false;
    }
    return true;
  }

  public ALStringField getGroupName() {
    return group_name;
  }

  public ALStringField getPublicFlag() {
    return public_flag;
  }

  public void setPublicFlag(ALStringField field) {
    public_flag = field;
  }

  /**
   * @return
   */
  public ALDateField getCreateDate() {
    return create_date;
  }

  /**
   * @return
   */
  public ALDateField getUpdateDate() {
    return update_date;
  }

  /**
   * @param field
   */
  public void setCreateDate(ALDateField field) {
    create_date = field;
  }

  /**
   * @param field
   */
  public void setUpdateDate(ALDateField field) {
    update_date = field;
  }

  /**
   * 追加（編集）画面の所属者選択リスト用のデータを取得する。
   * 
   * @param rundata
   * @param context
   */
  public void loadFilter(RunData rundata, Context context) {
    // del by motegi
    // try {
    // // 自分がオーナのグループを取得
    // SelectQuery<EipMAddressGroup> query1 =
    // Database.query(EipMAddressGroup.class);
    // Expression exp1 =
    // ExpressionFactory.matchExp(EipMAddressGroup.OWNER_ID_PROPERTY, Integer
    // .valueOf(ALEipUtils.getUserId(rundata)));
    // query1.setQualifier(exp1);
    //
    // List<EipMAddressGroup> glist = query1.fetchList();
    //
    // // Mapへ値をセット
    // for (int i = 0; i < glist.size(); i++) {
    // EipMAddressGroup group = glist.get(i);
    // groupList.add(group);
    // }

    // // アドレス一覧を取得(全体用)
    // SelectQuery<EipMAddressbook> query2 =
    // Database.query(EipMAddressbook.class);
    // Expression exp21 =
    // ExpressionFactory.matchExp(EipMAddressbook.PUBLIC_FLAG_PROPERTY, "T");
    // Expression exp22 =
    // ExpressionFactory.matchExp(EipMAddressbook.OWNER_ID_PROPERTY, Integer
    // .valueOf(ALEipUtils.getUserId(rundata)));
    // query2.setQualifier(exp21.orExp(exp22));
    //
    // List<EipMAddressbook> list = query2.fetchList();
    //
    // int addressNum = list.size();
    // AddressBookFilterData address;
    // for (int i = 0; i < addressNum; i++) {
    // EipMAddressbook rec = list.get(i);
    // address = new AddressBookFilterData();
    // address.initField();
    // address.setAddressId(rec.getAddressId().intValue());
    // address.setFullName(rec.getLastName(), rec.getFirstName());
    // allAddressList.add(address);
    // }
    //
    // // アドレス一覧をグループID付で取得(特定グループ用)
    // String sql =
    // "SELECT eip_t_addressbook_group_map.GROUP_ID,
    // eip_m_addressbook.ADDRESS_ID, eip_m_addressbook.LAST_NAME,
    // eip_m_addressbook.FIRST_NAME "
    // + "FROM eip_m_addressbook "
    // + "LEFT JOIN eip_t_addressbook_group_map ON eip_m_addressbook.address_id
    // = eip_t_addressbook_group_map.address_id "
    // + "LEFT JOIN eip_m_addressbook_group ON
    // eip_t_addressbook_group_map.group_id = eip_m_addressbook_group.group_id "
    // + "WHERE eip_m_addressbook.PUBLIC_FLAG='T' "
    // + " OR eip_m_addressbook.OWNER_ID="
    // + ALEipUtils.getUserId(rundata)
    // + " ORDER BY eip_t_addressbook_group_map.GROUP_ID ASC";
    // List<DataRow> list2 =
    // Database.sql(EipTAddressbookGroupMap.class, sql).fetchListAsDataRow();
    //
    // DataRow dataRow = null;
    // addressNum = list2.size();
    // for (int i = 0; i < addressNum; i++) {
    // dataRow = list2.get(i);
    // address = new AddressBookFilterData();
    // address.initField();
    // address.setAddressId((Integer) Database.getFromDataRow(
    // dataRow,
    // EipMAddressbook.ADDRESS_ID_PK_COLUMN));
    // address.setFullName((String) Database.getFromDataRow(
    // dataRow,
    // EipMAddressbook.LAST_NAME_COLUMN), (String) Database.getFromDataRow(
    // dataRow,
    // EipMAddressbook.FIRST_NAME_COLUMN));
    // addressList.add(address);
    // }
    // } catch (Exception ex) {
    // logger.error("Exception", ex);
    // }
  }

  // public List<EipMAddressGroup> getGroupList() {
  // return groupList;
  // }

  // public List<Object> getAddressList() {
  // return addressList;
  // }

  // del by motegi
  // public List<AddressBookFilterData> getAllAddressList() {
  // return allAddressList;
  // }

  // del by motegi
  // public List<AddressBookFilterData> getGroupAddressList(int gid) {
  // List<AddressBookFilterData> list = new ArrayList<AddressBookFilterData>();
  // for (int i = 0; i < addressList.size(); i++) {
  // AddressBookFilterData fData = (AddressBookFilterData) addressList.get(i);
  // if (fData.getGroupId().getValue() == gid) {
  // list.add(fData);
  // }
  // }
  // return list;
  // }

  /**
   * @return
   */
  public List<AddressBookResultData> getAddresses() {
    return addresses;
  }

  /**
   * アクセス権限チェック用メソッド。 アクセス権限の機能名を返します。
   * 
   * @return
   */
  @Override
  public String getAclPortletFeature() {
    return ALAccessControlConstants.POERTLET_FEATURE_ADDRESSBOOK_COMPANY_GROUP;
  }
}
