/*
 * Aipo is a groupware program developed by Aimluck,Inc.
 * Copyright (C) 2004-2011 Aimluck,Inc.
 * http://www.aipo.com
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.addressbook;

import java.util.ArrayList;
import java.util.List;
import java.util.jar.Attributes;

import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.utils.ALDateUtil;
import com.aimluck.eip.addressbook.util.AddressBookUtils;
import com.aimluck.eip.addressbookuser.util.AddressBookUserUtils;
import com.aimluck.eip.cayenne.om.portlet.EipMAddressGroup;
import com.aimluck.eip.cayenne.om.portlet.EipMAddressbook;
import com.aimluck.eip.cayenne.om.portlet.EipTAddressbookGroupMap;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALEipConstants;
import com.aimluck.eip.common.ALEipUser;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.orm.query.ResultList;
import com.aimluck.eip.orm.query.SelectQuery;
import com.aimluck.eip.services.accessctl.ALAccessControlConstants;
import com.aimluck.eip.util.ALCommonUtils;
import com.aimluck.eip.util.ALEipUtils;

/**
 * アドレス帳ワード検索用データクラスです。(社外アドレス検索用)
 * 
 */
public class AddressBookFilterdSelectData extends
    AbstractAddressBookFilterdSelectData<EipMAddressbook, EipMAddressbook> {
  /** logger */
  private static final JetspeedLogger logger =
    JetspeedLogFactoryService.getLogger(AddressBookFilterdSelectData.class
      .getName());

  /** フィルタに利用するグループリスト */
  private List<AddressBookGroupResultData> groupList;

  // add start
  /** ログインユーザーID */
  private int loginuserid;

  // add end

  /**
   * 初期処理
   * 
   * @param action
   *            アクションクラス
   * @param rundata
   *            JetSpeedランデータ
   * @param context
   *            JetSpeedコンテキスト
   * @throws ALPageNotFoundException
   * @throws ALDBErrorException
   */
  @Override
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {

    // add start
    // セッションの絞込み条件クリア処理（リクエストパラメータのクリアフラグがある場合）
    if (rundata.getParameters().containsKey("clearSession")) {
      clearAddressBookCorpSession(rundata, context);
    }
    // add end

    String sort = ALEipUtils.getTemp(rundata, context, LIST_SORT_STR);
    if (sort == null || sort.equals("")) {
      // change start by motegi 個人アドレス帳対応
      // ALEipUtils.setTemp(rundata, context, LIST_SORT_STR, "name_kana");
      ALEipUtils.setTemp(rundata, context, LIST_SORT_STR, "company_name");
      // change end
    }

    // add start
    loginuserid = ALEipUtils.getUserId(rundata);
    // add end

    super.init(action, rundata, context);
  }

  /**
   * 個人アドレスの一覧取得処理
   * 
   * @param rundata
   *            JetSpeedランデータ
   * @param context
   *            JetSpeedコンテキスト
   * @return 個人アドレスリスト
   */
  @Override
  protected ResultList<EipMAddressbook> selectList(RunData rundata,
      Context context) {

    try {
      SelectQuery<EipMAddressbook> query = getSelectQuery(rundata, context);
      // buildSelectQueryForListView(query);
      buildSelectQueryForListViewSort(query, rundata, context);

      List<EipMAddressbook> list = query.getResultList();
      int totalSize = list.size();
      return new ResultList<EipMAddressbook>(
        buildPaginatedList(list),
        current_page,
        getRowsNum(),
        totalSize);
      // return buildPaginatedList(query.getResultList());
    } catch (Exception ex) {
      // change start
      // logger.error("Exception", ex);
      String msg =
        "個人アドレスの一覧取得に失敗しました。[" + ALEipUtils.getUserLoginName(loginuserid) + "]";
      logger.error(msg, ex);
      // change end
      return null;
    }
  }

  /**
   * 社外アドレスタブ選択時のアドレス帳の詳細情報を表示します。
   * 
   * @param rundata
   * @param context
   * @return
   */
  @Override
  protected EipMAddressbook selectDetail(RunData rundata, Context context) {
    try {
      return AddressBookUtils.getEipMAddressbook(rundata, context);
    } catch (Exception ex) {
      // change start
      // logger.error("Exception", ex);
      String msg =
        "個人アドレスの詳細取得に失敗しました。["
          + ALEipUtils.getBaseUser(loginuserid).getUserName()
          + "]";
      logger.error(msg, ex);
      // change end
      return null;
    }
  }

  /**
   * ResultData に値を格納して返します。（一覧データ） <BR>
   * 
   * @param obj
   *            DBから取得し個人アドレス情報
   * @return 画面表示用データ
   * @see com.aimluck.eip.common.ALAbstractSelectData#getListData(java.lang.Object)
   */
  @Override
  protected Object getResultData(EipMAddressbook record) {
    try {

      AddressBookResultData rd = new AddressBookResultData();
      rd.initField();
      rd.setAddressId(record.getAddressId().intValue());
      // change start
      // rd.setName(new StringBuffer()
      // .append(record.getLastName())
      // .append(" ")
      // .append(record.getFirstName())
      // .toString());
      // rd.setNameKana(new StringBuffer()
      // .append(record.getLastNameKana())
      // .append(' ')
      // .append(record.getFirstNameKana())
      // .toString());

      rd.setName(AddressBookUtils.makeFullName(record.getLastName(), record
        .getFirstName()));
      rd.setNameKana(AddressBookUtils.makeFullName(
        record.getLastNameKana(),
        record.getFirstNameKana()));
      // change end

      // change by motegi start 個人アドレス帳
      // EipMAddressbookCompany company = record.getEipMAddressbookCompany();
      //
      // if
      // (!AddressBookUtils.EMPTY_COMPANY_NAME.equals(company.getCompanyName()))
      // {
      // // 「未分類」の会社情報ではない場合
      // rd.setCompanyName(ALCommonUtils.compressString(
      // company.getCompanyName(),
      // getStrLength()));
      // rd.setCompanyId(company.getCompanyId().toString());
      // rd.setPostName(ALCommonUtils.compressString(
      // company.getPostName(),
      // getStrLength()));
      // }
      EipMAddressGroup company =
        AddressBookUtils.getEipMAddressGroup(record.getAddressId());
      if (company != null) {
        rd.setCompanyId(company.getGroupId().toString());
        rd.setCompanyName(ALCommonUtils.compressString(
          company.getGroupName(),
          getStrLength()));
      }
      // change end

      rd.setPositionName(ALCommonUtils.compressString(
        record.getPositionName(),
        getStrLength()));
      rd.setEmail(record.getEmail());
      rd.setTelephone(record.getTelephone());
      rd.setCellularPhone(record.getCellularPhone());
      rd.setCellularMail(record.getCellularMail());
      rd.setPublicFlag(record.getPublicFlag());

      // add by motegi start 個人アドレス帳対応
      rd.setFaxNumber(record.getFaxNumber());
      // add end
      // add start 2012.2.21 受入障害対応No.288
      // 携帯用のメールアドレス
      // カンマ,ダブルコーテーションをエスケープ処理したアドレスにしてセット
      rd.setMailAddressCellPhoneHiddenValue(AddressBookUtils
        .getMailAddressCellPhoneHiddenValue(rd.getName().getValue()
          + rd.getPositionName().getValue(), rd.getEmail().getValue()));
      // add end
      return rd;
    } catch (Exception ex) {
      // change start
      // logger.error("Exception", ex);
      String msg =
        "個人アドレスの一覧取得に失敗しました。[" + ALEipUtils.getUserLoginName(loginuserid) + "]";
      logger.error(msg, ex);
      // change end
      return null;
    }
  }

  /**
   * 詳細情報の返却データ取得。
   * 
   * @param record
   * @return
   */
  @Override
  protected Object getResultDataDetail(EipMAddressbook record) {
    try {

      AddressBookResultData rd = new AddressBookResultData();
      rd.initField();

      // 登録ユーザ名の設定
      ALEipUser createdUser =
        ALEipUtils.getALEipUser(record.getCreateUserId().intValue());
      String createdUserName = createdUser.getAliasName().getValue();
      rd.setCreatedUser(createdUserName);

      // 更新ユーザ名の設定
      String updatedUserName;
      if (record.getCreateUserId().equals(record.getUpdateUserId())) {
        updatedUserName = createdUserName;
      } else {
        ALEipUser updatedUser =
          ALEipUtils.getALEipUser(record.getUpdateUserId().intValue());
        updatedUserName = updatedUser.getAliasName().getValue();
      }
      rd.setUpdatedUser(updatedUserName);

      // アドレスID の設定
      int addressId = record.getAddressId().intValue();
      rd.setAddressId(addressId);
      // change start
      // rd.setName(new StringBuffer()
      // .append(record.getLastName())
      // .append(' ')
      // .append(record.getFirstName())
      // .toString());
      // rd.setNameKana(new StringBuffer()
      // .append(record.getLastNameKana())
      // .append(' ')
      // .append(record.getFirstNameKana())
      // .toString());
      rd.setName(AddressBookUtils.makeFullName(record.getLastName(), record
        .getFirstName()));
      rd.setNameKana(AddressBookUtils.makeFullName(
        record.getLastNameKana(),
        record.getFirstNameKana()));
      // change end
      rd.setEmail(record.getEmail());
      rd.setTelephone(record.getTelephone());
      rd.setCellularPhone(record.getCellularPhone());
      rd.setCellularMail(record.getCellularMail());
      rd.setPositionName(record.getPositionName());
      rd.setPublicFlag(record.getPublicFlag());

      // change by motegi start 個人アドレス帳
      // EipMAddressbookCompany company = record.getEipMAddressbookCompany();
      // if
      // (!AddressBookUtils.EMPTY_COMPANY_NAME.equals(company.getCompanyName()))
      // {
      // // 「未分類」の会社情報ではない場合、会社情報を設定する
      // rd.setCompanyName(company.getCompanyName());
      // rd.setCompanyNameKana(company.getCompanyNameKana());
      // rd.setPostName(company.getPostName());
      // rd.setZipcode(company.getZipcode());
      // rd.setCompanyAddress(company.getAddress());
      // rd.setCompanyTelephone(company.getTelephone());
      // rd.setCompanyFaxNumber(company.getFaxNumber());
      // rd.setCompanyUrl(company.getUrl());
      // }
      EipMAddressGroup company =
        AddressBookUtils.getEipMAddressGroup(record.getAddressId());
      if (company != null) {
        rd.setCompanyId(company.getGroupId().toString());
        rd.setCompanyName(ALCommonUtils.compressString(
          company.getGroupName(),
          getStrLength()));
      }
      // change end

      rd.setCreateDate(ALDateUtil.format(record.getCreateDate(), "yyyy年M月d日"));
      rd.setUpdateDate(ALDateUtil.format(record.getUpdateDate(), "yyyy年M月d日"));

      // add by motegi start 個人アドレス帳対応
      rd.setFaxNumber(record.getFaxNumber());
      rd.setNote(record.getNote());
      // add end

      return rd;
    } catch (Exception ex) {
      // change start
      // logger.error("Exception", ex);
      String msg =
        "個人アドレスの詳細取得に失敗しました。[" + ALEipUtils.getUserLoginName(loginuserid) + "]";
      logger.error(msg, ex);
      // change end
      return null;
    }
  }

  /**
   * ソート用カラムマップ取得
   * 
   * @return ソート用カラムマップ
   */
  @Override
  protected Attributes getColumnMap() {
    Attributes map = new Attributes();
    map.putValue("group", EipMAddressbook.EIP_TADDRESSBOOK_GROUP_MAP_PROPERTY
      + "."
      + EipTAddressbookGroupMap.EIP_TADDRESS_GROUP_PROPERTY
      + "."
      + EipMAddressGroup.GROUP_ID_PK_COLUMN);
    // change start
    map.putValue("name_kana", EipMAddressbook.LAST_NAME_KANA_PROPERTY);
    // map.putValue("name_kana", EipMAddressbook.LAST_NAME_PROPERTY);
    // change end
    // change by motegi start 個人アドレス帳対応
    // map.putValue(
    // "company_name_kana",
    // EipMAddressbook.EIP_MADDRESSBOOK_COMPANY_PROPERTY
    // + "."
    // + EipMAddressbookCompany.COMPANY_NAME_KANA_PROPERTY);
    map.putValue(
      "company_name",
      EipMAddressbook.EIP_TADDRESSBOOK_GROUP_MAP_PROPERTY
        + "."
        + EipTAddressbookGroupMap.EIP_TADDRESS_GROUP_PROPERTY
        + "."
        + EipMAddressGroup.GROUP_NAME_PROPERTY);
    // change end
    // add start
    // map.putValue("first_name_kana",
    // EipMAddressbook.FIRST_NAME_KANA_PROPERTY);

    // add end
    return map;
  }

  /**
   * 検索条件を設定した SelectQuery を返します。 <BR>
   * 
   * @param rundata
   *            JetSpeedランデータ
   * @param context
   *            JetSpeedコンテキスト
   * @return SQLオブジェクト
   */
  protected SelectQuery<EipMAddressbook> getSelectQuery(RunData rundata,
      Context context) {
    SelectQuery<EipMAddressbook> query = Database.query(EipMAddressbook.class);

    Expression exp21 =
      ExpressionFactory.matchExp(EipMAddressbook.PUBLIC_FLAG_PROPERTY, "T");
    Expression exp22 =
      ExpressionFactory.matchExp(EipMAddressbook.OWNER_ID_PROPERTY, ALEipUtils
        .getUserId(rundata));
    Expression exp23 =
      ExpressionFactory.matchExp(EipMAddressbook.PUBLIC_FLAG_PROPERTY, "F");
    query.setQualifier(exp21.orExp(exp22.andExp(exp23)));

    return getSelectQueryForIndex(query, rundata, context);
  }

  /**
   * インデックス検索のためのカラムを返します。
   * 
   * @return インデックス検索のためのカラム名
   */
  @Override
  protected String getColumnForIndex() {
    return EipMAddressbook.LAST_NAME_KANA_PROPERTY;
  }

  /**
   * アドレス帳社外グループ選択プルダウン用データ取得
   * 
   * @return アドレス帳社外グループリスト
   */
  public List<AddressBookGroupResultData> getGroupList() {
    return groupList;
  }

  /**
   * アドレス帳社外グループ選択プルダウン用データ取得
   * 
   * @param rundata
   *            JetSpeedランデータ
   * @param context
   *            JetSpeedコンテキスト
   */
  public void loadGroups(RunData rundata, Context context) {
    groupList = new ArrayList<AddressBookGroupResultData>();
    try {
      SelectQuery<EipMAddressGroup> query =
        Database.query(EipMAddressGroup.class);
      Expression exp =
        ExpressionFactory.matchExp(EipMAddressGroup.OWNER_ID_PROPERTY, Integer
          .valueOf(ALEipUtils.getUserId(rundata)));
      query.setQualifier(exp);
      query.orderAscending(EipMAddressGroup.GROUP_NAME_PROPERTY);

      List<EipMAddressGroup> aList = query.fetchList();

      int size = aList.size();
      for (int i = 0; i < size; i++) {
        EipMAddressGroup record = aList.get(i);
        AddressBookGroupResultData rd = new AddressBookGroupResultData();
        rd.initField();
        rd.setGroupId(record.getGroupId().intValue());
        rd.setGroupName(record.getGroupName());
        groupList.add(rd);
      }

      // add by motegi start
      // "未所属"を先頭に追加
      EipMAddressGroup defaultGroup =
        AddressBookUserUtils.getDefaultEipMAddressGroup();
      AddressBookGroupResultData r = new AddressBookGroupResultData();
      r.initField();
      r.setGroupId(defaultGroup.getGroupId().intValue());
      r.setGroupName(AddressBookUserUtils.EMPTY_COMPANY_NAME);
      groupList.add(0, r);
      // add end

    } catch (Exception ex) {
      // change start
      // logger.error("Exception", ex);
      String msg =
        "会社グループプルダウン用のデータ取得に失敗しました。["
          + ALEipUtils.getUserLoginName(loginuserid)
          + "]";
      logger.error(msg, ex);
      // change end
    }
  }

  /**
   * アクセス権限チェック用メソッド。 アクセス権限の機能名を返します。
   * 
   * @return アクセス権限の機能名
   */
  @Override
  public String getAclPortletFeature() {
    return ALAccessControlConstants.POERTLET_FEATURE_ADDRESSBOOK_ADDRESS_OUTSIDE;
  }

  /**
   * ソート用の <code>SelectQuery</code> を構築します。
   * 
   * @param query
   *            SQLオブジェクト
   * @param rundata
   *            JetSpeedランデータ
   * @param context
   *            JetSpeedコンテキスト
   * @return ソート用の構築済みSQLオブジェクト
   */
  @Override
  protected SelectQuery<EipMAddressbook> buildSelectQueryForListViewSort(
      SelectQuery<EipMAddressbook> query, RunData rundata, Context context) {
    String sort = ALEipUtils.getTemp(rundata, context, LIST_SORT_STR);
    String sort_type = ALEipUtils.getTemp(rundata, context, LIST_SORT_TYPE_STR);
    String crt_key = null;

    Attributes map = getColumnMap();
    if (sort == null) {
      return query;
    }
    crt_key = map.getValue(sort);
    if (crt_key == null) {
      return query;
    }

    if ("name_kana".equals(sort)) {
      if (sort_type != null
        && ALEipConstants.LIST_SORT_TYPE_DESC.equals(sort_type)) {
        query.orderDesending(crt_key);
        // add start
        query.orderDesending(EipMAddressbook.UPDATE_DATE_PROPERTY);
        // add end
      } else {
        query.orderAscending(crt_key);
        // add start
        query.orderAscending(EipMAddressbook.UPDATE_DATE_PROPERTY);
        // add end
        sort_type = ALEipConstants.LIST_SORT_TYPE_ASC;
      }
    } else if ("company_name".equals(sort)) {

      String crt_sub_key = map.getValue("name_kana");
      // add start
      // String first_name_key = map.getValue("first_name_kana");
      // add end
      if (sort_type != null
        && ALEipConstants.LIST_SORT_TYPE_DESC.equals(sort_type)) {
        query.orderDesending(crt_key);
        query.orderDesending(crt_sub_key);
        // add start
        // query.orderDesending(first_name_key);
        // add end
        // add start
        query.orderDesending(EipMAddressbook.UPDATE_DATE_PROPERTY);
        // add end
      } else {
        query.orderAscending(crt_key);
        query.orderAscending(crt_sub_key);
        // add start
        // query.orderAscending(first_name_key);
        // add end
        // add start
        query.orderAscending(EipMAddressbook.UPDATE_DATE_PROPERTY);
        // add end
        sort_type = ALEipConstants.LIST_SORT_TYPE_ASC;
      }
    }

    current_sort = sort;
    current_sort_type = sort_type;
    return query;
  }

  /**
   * 絞込み条件をクリア
   * 
   * @param rundata
   * @param context
   */
  public void clearAddressBookCorpSession(RunData rundata, Context context) {
    ALEipUtils.removeTemp(rundata, context, LIST_SORT_STR);
    ALEipUtils.removeTemp(rundata, context, LIST_SORT_TYPE_STR);
    ALEipUtils.removeTemp(rundata, context, LIST_FILTER_STR);
    ALEipUtils.removeTemp(rundata, context, LIST_FILTER_TYPE_STR);
  }
  // add end
}
