// -*-Mode: C++;-*-
//
//  Smooth spline interpolator class
//

#include <common.h>
#include <modules/molvis/molvis.hpp>

#include "SmoothSpline.hpp"

using namespace molvis;

SmoothSpline::SmoothSpline()
     : m_pCoeff0(NULL), m_pCoeff1(NULL), m_pCoeff2(NULL), m_pCoeff3(NULL)
{
}

SmoothSpline::~SmoothSpline()
{
  cleanup();
}

/** re-initialization */
void SmoothSpline::cleanup()
{
  if (m_pCoeff0!=NULL)
    delete [] m_pCoeff0;
  if (m_pCoeff1!=NULL)
    delete [] m_pCoeff1;
  if (m_pCoeff2!=NULL)
    delete [] m_pCoeff2;
  if (m_pCoeff3!=NULL)
    delete [] m_pCoeff3;
  m_pCoeff0 = m_pCoeff1 = m_pCoeff2 = m_pCoeff3 = NULL;

  m_nPoints = 0;
}

/** generate spline coeffs */
bool SmoothSpline::generate()
{
  int i;
  m_nPoints = m_veclist.size();

  if (m_nPoints==0)
    return false; // there is no points to interpolate!!

  // allocate coefficient table
  MB_ASSERT(m_pCoeff0==NULL);
  m_pCoeff0 = MB_NEW Vector4D[m_nPoints];
  MB_ASSERT(m_pCoeff1==NULL);
  m_pCoeff1 = MB_NEW Vector4D[m_nPoints];
  MB_ASSERT(m_pCoeff2==NULL);
  m_pCoeff2 = MB_NEW Vector4D[m_nPoints];
  MB_ASSERT(m_pCoeff3==NULL);
  m_pCoeff3 = MB_NEW Vector4D[m_nPoints];

  if (m_nPoints==1) {
    // Degenerated case (point)
    // cannot calculate spline, return constant polynom
    const Vector4D &p0 = m_veclist[0];

    m_pCoeff3[0] = Vector4D();
    m_pCoeff2[0] = Vector4D();
    m_pCoeff1[0] = Vector4D();
    m_pCoeff0[0] = p0;
    return true;
  }
  else if (m_nPoints==2) {
    // Degenerated case (line)
    const Vector4D &p0 = m_veclist[0];
    const Vector4D &p1 = m_veclist[1];

    m_pCoeff3[0] = Vector4D();
    m_pCoeff2[0] = Vector4D();
    m_pCoeff1[0] = p1-p0;
    m_pCoeff0[0] = p0;
    return true;
  }

  ///////////////////////////////////////////////////
  // calculate natural spline coeffs

  const int nsize = m_nPoints;
  alglib::real_1d_array t, x, y, z;

  t.setlength(nsize);
  x.setlength(nsize);
  y.setlength(nsize);
  z.setlength(nsize);

  for (int i=0; i<nsize; ++i) {
    t[i] = i; // tvec[i];
    const Vector4D &p = m_veclist[i];
    x[i] = p.x();
    y[i] = p.y();
    z[i] = p.z();
  }

  ae_int_t info;
  double v;
  double rho;
  alglib::spline1dinterpolant sx, sy, sz;
  alglib::spline1dfitreport rep;

  //
  // Fit with VERY small amount of smoothing (rho = -5.0)
  // and large number of basis functions (M=50).
  //
  // With such small regularization penalized spline almost fully reproduces function values
  //
  //rho = -5.0;
  rho = 3.0;
  int mseg = nsize+2-1;

  alglib::spline1dfitpenalized(t, x, mseg, rho, info, sx, rep);
  //fprintf(stderr, "X info: %d\n", int(info)); // EXPECTED: 1

  alglib::spline1dfitpenalized(t, y, mseg, rho, info, sy, rep);
  //fprintf(stderr, "Y info: %d\n", int(info)); // EXPECTED: 1

  alglib::spline1dfitpenalized(t, z, mseg, rho, info, sz, rep);
  //fprintf(stderr, "Z info: %d\n", int(info)); // EXPECTED: 1

  /*
  const double dt = 0.05;
  int nseq = floor(tmin+0.5);
  for (double t = tmin; t<=tmax; t+=dt) {
    double x = spline1dcalc(sx, t);
    double y = spline1dcalc(sy, t);
    double z = spline1dcalc(sz, t);

    //ATOM     10  CA  ALA A  15     -75.125  53.864  47.016  1.00131.78           C
    printf("ATOM  %5d  CA  ALA A%4d    %8.3f%8.3f%8.3f  1.00 30.00           C\n",
	   int(nseq), int(nseq), x, y, z);
    ++nseq;
  }
  return 0;
   */

  return true;
}

/** perform interpolation */
bool SmoothSpline::interpolate(double par, Vector4D *vec,
                              Vector4D *dvec /*= NULL*/,
                              Vector4D *ddvec /*= NULL*/)
{
  // check parameter value f
  int ncoeff = (int)::floor(par);
  if (ncoeff<0)
    ncoeff = 0;
  if (ncoeff>=(m_nPoints-1))
    ncoeff = m_nPoints-2;

  const Vector4D &coeff0 = m_pCoeff0[ncoeff];
  const Vector4D &coeff1 = m_pCoeff1[ncoeff];
  const Vector4D &coeff2 = m_pCoeff2[ncoeff];
  const Vector4D &coeff3 = m_pCoeff3[ncoeff];

  double f = par - (double)ncoeff;

  Vector4D tmp;
  tmp = coeff3.scale(f) + coeff2;
  tmp = tmp.scale(f) + coeff1;
  tmp = tmp.scale(f) + coeff0;
  *vec = tmp;

  if (dvec != NULL) {
    // calculate tangential vector
    tmp = coeff3.scale(3.0*f) + coeff2.scale(2.0);
    tmp = tmp.scale(f) + coeff1;
    *dvec = tmp;
  }

  if (ddvec != NULL) {
    // calculate curvature vector
    tmp = coeff3.scale(6.0*f) + coeff2.scale(2.0);
    *ddvec = tmp;
  }
  return true;
}



