//
// Text rendering routine using HTML5 Canvas 2D
//

#include <common.h>

#include "Canvas2DTextRender2.hpp"

#include <qlib/Utils.hpp>

using qlib::LString;

//////////////////////////////

Canvas2DTextRender2::Canvas2DTextRender2()
{
  m_dFontSize = 0.0;
}

Canvas2DTextRender2::~Canvas2DTextRender2()
{
}

//////////////////////////////

typedef boost::const_multi_array_ref<QUE_BYTE, 3> ConstArrayRef;
typedef boost::multi_array_ref<QUE_BYTE, 3> ArrayRef;

// make HTML5 CSS font name string
bool Canvas2DTextRender2::setupFont(double fontsize, const LString &fontname,
                                    const LString &font_style,
                                    const LString &font_wgt)
{
  LString cssfont;

  // Font style
  if (font_style.equalsIgnoreCase("italic")) {
    cssfont += "italic ";
  }
  else if (font_style.equalsIgnoreCase("oblique")) {
    cssfont += "oblique ";
  }
  //else if (font_style.equalsIgnoreCase("normal"))
  // normal may be default

  // Font weight
  if (font_wgt.equalsIgnoreCase("bold")) {
    cssfont += "bold ";
  }
  //if (font_wgt.equalsIgnoreCase("normal")) {
  // default is normal

  // Font size (always in pixel)
  cssfont += LString::format("%dpx ", int(fontsize));
  cssfont += fontname;

  m_dFontSize = fontsize;
  m_strFontName = fontname;
  m_strCSSFont = cssfont;

  return true;
}


/// Render text to the pixel buffer
bool Canvas2DTextRender2::renderText(const qlib::LString &text, gfx::PixelBuffer &buf)
{
  LString fontname = m_strCSSFont;
  nsCAutoString nsstr;
  nsAutoString text_ucs16, fontname_ucs16;
  nsresult rv;
  //uint32_t width, height;
  int width, height;

  // conv rendering text
  nsstr = nsCAutoString(text.c_str());
  ::CopyUTF8toUTF16(nsstr, text_ucs16);
  
  // conv font name
  nsstr = nsCAutoString(m_strCSSFont.c_str());
  ::CopyUTF8toUTF16(nsstr, fontname_ucs16);
  
  nsIDOMHTMLCanvasElement *pCanvas = getCanvas();
  if (pCanvas==NULL) {
    // TO DO: throw expt
    return false;
  }

  // measure text size
  {
    float fwidth;

    // Get canvas context 2d
    nsCOMPtr<nsISupports> ptmp;
    pCanvas->GetContext(NS_LITERAL_STRING("2d"), JSVAL_VOID, getter_AddRefs(ptmp));
    nsCOMPtr<nsIDOMCanvasRenderingContext2D> pCtx = do_QueryInterface(ptmp);

    pCtx->SetFont(fontname_ucs16);

    nsCOMPtr<nsIDOMTextMetrics> metrics;
    rv = pCtx->MeasureText(text_ucs16, getter_AddRefs(metrics));
    if ( NS_SUCCEEDED(rv) )
      metrics->GetWidth(&fwidth);
    else
      fwidth = 100;
    
    MB_DPRINTLN("Canvas2D> text width: %f", fwidth);
    
    int width = (int) ceil(fwidth);
    if (width%4!=0)
      width += (4-width%4);
    
    //int height = (int) ceil(m_dFontSize*3.0);
    int height = (int) ceil(m_dFontSize);
    
    //m_nImgWidth = width;
    //m_nImgHeight = height;
  }

  // allocate enough size of canvas
  {
    PRUint32 ncurw, ncurh;
    rv = pCanvas->GetWidth(&ncurw);
    NS_ENSURE_SUCCESS(rv, false);
    if (ncurw<width) {
      rv = pCanvas->SetWidth(width);
      NS_ENSURE_SUCCESS(rv, false);
    }

    rv = pCanvas->GetHeight(&ncurh);
    NS_ENSURE_SUCCESS(rv, false);
    if (ncurh<height) {
      rv = pCanvas->SetHeight(height);
      NS_ENSURE_SUCCESS(rv, false);
    }
  }

  // draw text to the canvas
  {
    // Get canvas context 2d
    nsCOMPtr<nsISupports> ptmp;
    pCanvas->GetContext(NS_LITERAL_STRING("2d"), JSVAL_VOID, getter_AddRefs(ptmp));
    nsCOMPtr<nsIDOMCanvasRenderingContext2D> pCtx = do_QueryInterface(ptmp);

    //mCanvasCtx->GetFont(ucs16);
    //::CopyUTF16toUTF8(ucs16, nsstr);
    //const char *pstr = ToNewCString(nsstr);
    //MB_DPRINTLN("font: %s", pstr);

    rv = pCtx->SetFont(fontname_ucs16);
    NS_ENSURE_SUCCESS(rv, false);
    rv = pCtx->FillText(text_ucs16, 0, float(m_dFontSize), 0);
    NS_ENSURE_SUCCESS(rv, false);

    // Convert Canvas ImageData to qlib PixelBuffer object
    const int nsize = width*height;
    QUE_BYTE *psf = new QUE_BYTE [nsize*4];
    pCtx->GetImageData_explicit(0, 0, width, height, psf, nsize*4);
    
    if (buf.size()<nsize)
      buf.resize(nsize);
    
    ConstArrayRef source(psf, boost::extents[height][width][4]);
    ArrayRef dest(buf.data(), boost::extents[height][width][1]);
    int i, j;
    for (j=0; j<height; j++) {
      for (i=0; i<width; i++) {
        //MB_DPRINTLN("(x=%d,y=%d,c=%d) = %d", i,j,0,xxx);
        dest[height-j-1][i][0] = source[j][i][0];
      }
    }
    
    buf.setDepth(8);
    buf.setWidth(width);
    buf.setHeight(height);
    delete [] psf;
  }

  // convertToPixelBuffer(buf, width, height);
  return true;
}  

/*
/// Convert Canvas ImageData to qlib PixelBuffer object
void Canvas2DTextRender2::convertToPixelBuffer(gfx::PixelBuffer &bytes, int w, int h)
{
  int i, j;

  mCanvasCtx->GetImageData_explicit(0, 0, m_nImgWidth, m_nImgHeight);

  // ImageData format is always in 8bit RGBA per pixel
  QUE_BYTE *psf = pImgSurf->Data();

  int nsize = w*h;
  if (bytes.size()<nsize)
    bytes.resize(nsize);

  ConstArrayRef source(psf, boost::extents[h][w][4]);
  ArrayRef dest(bytes.data(), boost::extents[h][w][1]);
  for (j=0; j<h; j++) {
    for (i=0; i<w; i++) {
      //MB_DPRINTLN("(x=%d,y=%d,c=%d) = %d", i,j,0,xxx);
      dest[h-j-1][i][0] = source[j][i][0];
    }
  }

  bytes.setDepth(8);
  bytes.setWidth(w);
  bytes.setHeight(h);
}
*/


////////////////////////////////////////////////////////////////////////////////////////
/*
gfx::TextRenderImpl *createTextRender()
{
  Canvas2DTextRender2 *pTTR = new Canvas2DTextRender2;
  pTTR->setupFont(12.0, "sans-serif", "normal", "normal");
  //pTTR->setupFont(20.0, "Times New Roman", FONT_STYLE_NORMAL, FONT_WEIGHT_NORMAL);

  return pTTR;
}

void destroyTextRender(void *pTR)
{
  Canvas2DTextRender2 *pTTR = static_cast<Canvas2DTextRender2 *>(pTR);
  delete pTTR;
}
*/
