// -*-Mode: C++;-*-
//
// Style sheet implementation in StyleMgr database
//
// $Id: StyleMgrStyleImpl.cpp,v 1.3 2011/04/17 06:16:17 rishitani Exp $

#include <common.h>

#include "StyleMgr.hpp"
#include "StyleSet.hpp"
#include "StyleSheet.hpp"
#include "StyleFile.hpp"
#include "StyleEditInfo.hpp"

#include <qlib/FileStream.hpp>
#include <qlib/LDOM2Stream.hpp>
#include <qlib/ObjectManager.hpp>
#include <gfx/NamedColor.hpp>
#include <qsys/SceneManager.hpp>

using namespace qsys;
using qlib::LDom2Node;

namespace {
  bool splitKeyName(const LString &keyname, LString &subkey, LString &trailkey)
  {
    int dpos = keyname.indexOf(".");

    if (dpos<0) {
      subkey = keyname;
      trailkey = "";
      return false;
    }
    else {
      subkey = keyname.substr(0, dpos);
      trailkey = keyname.substr(dpos+1);
      return true;
    }
  }
}

LDom2Node *StyleMgr::getStyleNode2(const LString &stylename,
                                   const LString &prop_names,
                                   qlib::uid_t ctxt)
{
  return getStyleNode2(LString(), stylename, prop_names, ctxt, false);
}

LDom2Node *StyleMgr::getStyleNode2(const LString &aSetID,
                                   const LString &stylename,
                                   const LString &prop_names,
                                   qlib::uid_t ctxt,
                                   bool bCreate)
{
  StyleList *pSList = getCreateStyleList(ctxt);
  LDom2Node *pStyle=NULL, *pNode=NULL;
  MB_ASSERT(pSList!=NULL);

  // Search style set list from top to bottom
  BOOST_FOREACH (StyleSetPtr pStySet, *pSList) {
    const LString &setid = pStySet->getName();
    
    // Empty aSetID matches all style sets
    if (!aSetID.isEmpty() && !aSetID.equals(setid))
      continue;
    
    pStyle = pStySet->getData(StyleSet::makeStyleKey(stylename));
    if (pStyle==NULL) {
      if (!bCreate) {
        // The styleset pStySet does not contain style named "stylename".
        // --> search the next style sheet
        //MB_DPRINTLN("Style %s is not found in stylesheet %s",
        //stylename.c_str(), setid.c_str());
        continue;
      }
      else {
        // create a new empty style node named stylename
        pStyle = MB_NEW LDom2Node();
        pStyle->setTagName("style");
        pStySet->putData(StyleSet::makeStyleKey(stylename), pStyle);
        MB_DPRINTLN("New style %s is created in sheet %s",
                    stylename.c_str(), setid.c_str());
      }
    }

    pNode = findStyleNodeByName(pStyle, prop_names, bCreate);
    if (pNode==NULL) {
      // the style pStyle does not contain style definition for "prop_names"
      // --> search next style set
      continue;
    }

    // style definition node for "stylename,prop_names" is found
    //MB_DPRINTLN("StyleNode for %s:%s is found in stylesheet %s",
    //stylename.c_str(), prop_names.c_str(), pStySet->getName().c_str());
    return pNode;
  }


  // fall-back to the global context search
  if (ctxt!=qlib::invalid_uid)
    return getStyleNode2(aSetID, stylename, prop_names, qlib::invalid_uid, bCreate);

  // not found!!
  return NULL;
}

/// Find style's node from the style root node pSty
LDom2Node *StyleMgr::findStyleNodeByName(LDom2Node *pSty, const LString &keyname, bool bCreate)
{
  LString subkey;
  LString trailkey;

  splitKeyName(keyname, subkey, trailkey);
  
  // MB_DPRINTLN("find %s -> %s, %s", keyname.c_str(), subkey.c_str(), trailkey.c_str());

  LDom2Node *pChNode = pSty->findChild(subkey);
  if (pChNode==NULL) {
    // style node is not found.
    if (!bCreate) {
      //MB_DPRINTLN("subkey %s is not found.", subkey.c_str());
      return NULL;
    }
    else {
      pChNode = MB_NEW LDom2Node();
      pChNode->setTagName(subkey);
      pSty->appendChild(pChNode);
      MB_DPRINTLN("New Style Node created for %s", pChNode->getTagName().c_str());
    }
  }
  
  if (trailkey.isEmpty())
    // pChNode is style (leaf) node for keyname.
    return pChNode;

  // recursively search in the child nodes
  return findStyleNodeByName(pChNode, trailkey, bCreate);
}

//////////
// value

LString StyleMgr::getStyleValue(qlib::uid_t ctxt,
                                const LString &setid,
                                const LString &dotname)
{
  LString style_name;
  LString prop_names;
  if (!splitKeyName(dotname, style_name, prop_names)) {
    LString msg = LString::format("StyleMgr.getStyleNode2> invalid key name: %s",dotname.c_str());
    MB_THROW(qlib::RuntimeException, msg);
    return LString();
  }

  LDom2Node *pNode = getStyleNode2(setid, style_name, prop_names, ctxt, false);
  if (pNode==NULL)
    return LString();
  return pNode->getValue();
}
    
void StyleMgr::setStyleValue(qlib::uid_t ctxt,
                             const LString &setid,
                             const LString &dotname,
                             const LString &value)
{
  LString style_name;
  LString prop_names;
  if (!splitKeyName(dotname, style_name, prop_names)) {
    LString msg = LString::format("StyleMgr.getStyleNode2> invalid key name: %s",dotname.c_str());
    MB_THROW(qlib::RuntimeException, msg);
    return;
  }

  LDom2Node *pNode = getStyleNode2(setid, style_name, prop_names, ctxt, true);
  if (pNode==NULL) {
    MB_ASSERT(false);
    return;
  }
  pNode->setValue(value);

  // set to the pending event list
  m_pendEvts.insert(PendEventSet::value_type(ctxt, style_name));
}


//////////////////////////////////////////////////////////////////////////////
// Style manupilation methods for UI

qlib::uid_t StyleMgr::hasStyleSet(const LString &id, qlib::uid_t ctxt)
{
  StyleList *pSL = getCreateStyleList(ctxt);
  MB_ASSERT(pSL!=NULL);

  StyleSetPtr pSet = pSL->findSet(id);
  if (pSet.isnull())
    return qlib::invalid_uid;
  
  return pSet->getUID();
}

StyleSetPtr StyleMgr::createStyleSet(const LString &id, qlib::uid_t ctxt)
{
  if (hasStyleSet(id, ctxt)!=qlib::invalid_uid)
    return StyleSetPtr();
  
  StyleList *pSL = getCreateStyleList(ctxt);
  MB_ASSERT(pSL!=NULL);

  StyleSetPtr pSet = StyleSetPtr( MB_NEW StyleSet );
  pSet->setContextID(ctxt);
  pSet->setSource("");
  pSet->setName(id);

  // Register new style to the end of the style list
  pSL->push_back(pSet);

  ScenePtr pScene = SceneManager::getSceneS(ctxt);

  // Fire the SCE_STYLE_ADDED Event
  if (!pScene.isnull()) {
    MB_DPRINTLN("StyleMgr> Firing SCE_STYLE_ADDED event...");
    SceneEvent ev;
    ev.setType(SceneEvent::SCE_STYLE_ADDED);
    ev.setSource(ctxt);
    ev.setTarget(pSet->getUID());
    pScene->fireSceneEvent(ev);
  }

  // Record undo/redo info
  UndoUtil uu(ctxt);
  if (uu.isOK()) {
    StyleCreateEditInfo *pInfo = MB_NEW StyleCreateEditInfo();
    pInfo->setupCreate(ctxt, pSet, -1);
    uu.add(pInfo);
  }

  return pSet;
}

qlib::uid_t StyleMgr::createStyleSetScr(const LString &id, qlib::uid_t ctxt)
{
  StyleSetPtr pSet = createStyleSet(id, ctxt);
  if (pSet.isnull()) return qlib::invalid_uid;
  return pSet->getUID();
}

bool StyleMgr::registerStyleSet(StyleSetPtr pSet, int nbefore, qlib::uid_t ctxt)
{
  if (hasStyleSet(pSet->getName(), ctxt)!=qlib::invalid_uid)
    return false;
  
  StyleList *pSL = getCreateStyleList(ctxt);
  MB_ASSERT(pSL!=NULL);

  pSet->setContextID(ctxt);

  if (nbefore >= pSL->size() || nbefore<0)
    pSL->push_back(pSet);
  else {
    StyleList::iterator iter = pSL->begin();
    for (int i=0; i<nbefore; ++i)
      ++iter;
    pSL->insert(iter, pSet);
  }

  // Fire the SCE_STYLE_ADDED Event
  ScenePtr pScene = SceneManager::getSceneS(ctxt);
  if (!pScene.isnull()) {
    MB_DPRINTLN("StyleMgr.register> Firing SCE_STYLE_ADDED event...");
    SceneEvent ev;
    ev.setType(SceneEvent::SCE_STYLE_ADDED);
    ev.setSource(ctxt);
    ev.setTarget(pSet->getUID());
    pScene->fireSceneEvent(ev);
  }

  return true;
}

bool StyleMgr::destroyStyleSet(qlib::uid_t nScopeID, qlib::uid_t nStyleSetID)
{
  StyleList *pSL = getCreateStyleList(nScopeID);
  MB_ASSERT(pSL!=NULL);

  StyleList::iterator iter = pSL->begin();
  StyleList::iterator eiter = pSL->end();
  StyleSetPtr pSet = StyleSetPtr();
  int nbefore = 0;
  for (; iter!=eiter; ++iter, ++nbefore) {
    if (nStyleSetID==(*iter)->getUID()) {
      pSet = *iter;
      break;
    }
  }
  if (iter==eiter)
    return false;

  // Fire the SCE_STYLE_REMOVING Event, before removing the style
  ScenePtr pScene = SceneManager::getSceneS(nScopeID);
  if (!pScene.isnull()) {
    MB_DPRINTLN("StyleMgr> Firing SCE_STYLE_REMOVING event...");
    SceneEvent ev;
    ev.setType(SceneEvent::SCE_STYLE_REMOVING);
    ev.setSource(nScopeID);
    ev.setTarget(pSet->getUID());
    pScene->fireSceneEvent(ev);
  }

  pSL->erase(iter);
  
  // setup event
  m_pendEvts.insert(PendEventSet::value_type(nScopeID, ""));

  // Record undo/redo info
  UndoUtil uu(nScopeID);
  if (uu.isOK()) {
    StyleCreateEditInfo *pInfo = MB_NEW StyleCreateEditInfo();
    pInfo->setupDestroy(nScopeID, pSet, nbefore);
    uu.add(pInfo);
  }

  // delete pSet;

  return true;
}

bool StyleMgr::saveStyleSetToFile(qlib::uid_t nScopeID, qlib::uid_t nStyleSetID, const LString &path)
{
  StyleList *pSL = getCreateStyleList(nScopeID);
  MB_ASSERT(pSL!=NULL);

  StyleSetPtr pStyleSet = StyleSetPtr();
  BOOST_FOREACH(StyleList::value_type pSet, *pSL) {
    if (pSet->getUID()==nStyleSetID) {
      pStyleSet = pSet;
      break;
    }
  }

  if (pStyleSet.isnull()) {
    LOG_DPRINT("SaveStyle> styleset (%d) not found\n", int(nStyleSetID));
    return false;
  }

  qlib::FileOutStream fos;
  try {
    fos.open(path);
  }
  catch (qlib::LException &e) {
    LOG_DPRINT("SaveStyle> cannot write file %s\n",path.c_str());
    LOG_DPRINT("SaveStyle>   (reason: %s)\n", e.getMsg().c_str());
    return false;
  }
    
  try {
    qlib::LDom2OutStream oos(fos);
    qlib::LDom2Tree tree("styles");
    
    qlib::LDom2Node *pNode = tree.top();
    pStyleSet->writeToDataNode(pNode);

    oos.write(&tree);
    //oos.close();
    fos.close();

    //tree.detach();
    //delete pIdNode;
  }
  catch (qlib::LException &e) {
    LOG_DPRINT("SaveStyle> cannot write file %s\n",path.c_str());
    LOG_DPRINT("SaveStyle>   (reason: %s)\n", e.getMsg().c_str());
    return false;
  }

  LString before = pStyleSet->getSource();

  if (before.equals(path))
    return true; // no change --> do nothing

  // convert to file-linked style
  pStyleSet->setSource(path);
  
  // Record undo/redo info
  UndoUtil uu(nScopeID);
  if (uu.isOK()) {
    StyleSrcEditInfo *pInfo = MB_NEW StyleSrcEditInfo();
    pInfo->setup(pStyleSet, before, path);
    uu.add(pInfo);
  }

  // Reset modified flag
  //  (because the content of this StyleSet is now synchronized with those in the file.)
  pStyleSet->setModified(false);

  return true;
}

qlib::uid_t StyleMgr::loadStyleSetFromFile(qlib::uid_t nScopeID, const LString &path)
{
  StyleFile sfile;
  qlib::uid_t uid = sfile.loadFile(path, nScopeID);
  m_pendEvts.insert(PendEventSet::value_type(nScopeID, ""));
  return uid;
}

LString StyleMgr::getStyleSetSource(qlib::uid_t nStyleSetID) const
{
  StyleSet *pRaw = qlib::ObjectManager::sGetObj<StyleSet>(nStyleSetID);

  /// create sp from raw ptr
  StyleSetPtr pSet = StyleSetPtr(pRaw);

  if (pSet.isnull()) {
    LString msg = LString::format("getStyleSetSource> invalid styleset ID: %d", int(nStyleSetID));
    MB_THROW(qlib::RuntimeException, msg);
    return LString();
  }

  return pSet->getSource();
}


////////////////////////////////////////////////////////////////
// event impl

void StyleMgr::firePendingEvents()
{
  BOOST_FOREACH (const PendEventSet::value_type &elem, m_pendEvts) {
    fireEventImpl(elem.first, elem.second);
  }

  clearPendingEvents();
}

void StyleMgr::fireEventImpl(qlib::uid_t uid, const LString &setname)
{
  if (setname.isEmpty() || setname.equals(">color")) {
    // color definition is (possibly) changed --> invalidate all named color cache data
    gfx::NamedColor::getResolver()->invalidateCache();
  }
  
  if (m_pLsnrs->isEmpty())
    return;
  
  int i, nsize = m_pLsnrs->getSize();
  std::vector<StyleEventListener *> cblist(nsize);
    
  if (m_pLsnrs->isLocked()) {
    MB_ASSERT(false);
    return;
  }

  {
    // make copy of listener list
    qlib::AutoEventCastLock lock(m_pLsnrs);
    StyleEventCaster::const_iterator iter = m_pLsnrs->begin();
    StyleEventCaster::const_iterator iend = m_pLsnrs->end();
    for (i=0 ; iter!=iend; iter++, i++)
      cblist[i] = iter->second;
  }

  StyleEvent ev;
  
  for (i=0; i<nsize; ++i) {
    StyleEventListener *pLsnr = cblist[i];
    qlib::uid_t nCtxtID = pLsnr->getStyleCtxtID();

    if (uid!=0 && uid!=nCtxtID)
      continue;

    // ">xxx" means non-style related event (color, sel, etc.)
    //   --> always fires styleChanged event!!
    if (!setname.isEmpty() && !setname.startsWith(">")) {
      StyleSheet *pSheet = pLsnr->getStyleSheet();
      if (!pSheet->contains(setname))
        continue;
    }

    pLsnr->styleChanged(ev);
  }

}

void StyleMgr::clearPendingEvents()
{
  m_pendEvts.clear();
}

bool StyleMgr::isModified(qlib::uid_t nSceneID) const
{
  StyleMgr *pthis = const_cast<StyleMgr *>(this);
  StyleList *pSList = pthis->getCreateStyleList(nSceneID);

  BOOST_FOREACH (StyleSetPtr pSet, *pSList) {
    if (pSet->isModified())
      return true;
  }

  return false;
}

//////////////////////////////////////////////////
// style event class

StyleEvent::~StyleEvent()
{
}

qlib::LCloneableObject *StyleEvent::clone() const
{
  return MB_NEW StyleEvent(*this);
}

