// -*-Mode: C++;-*-
//
//  Pov-Ray Display context implementation class
//

#ifndef POV_INTERNAL_DATA_HPP_INCLUDED
#define POV_INTERNAL_DATA_HPP_INCLUDED

#include "render.hpp"

#include <gfx/DisplayContext.hpp>
#include <gfx/ColorTable.hpp>
#include <qlib/LString.hpp>
#include <qlib/LStream.hpp>

namespace qlib {
  class PrintStream;
}

namespace render {

  using qlib::LString;
  using qlib::Vector4D;
  using qlib::OutStream;
  using gfx::ColorTable;
  using gfx::ColorPtr;
  using qlib::PrintStream;

  class FileDisplayContext;

  /// Internal data structure for PovContext
  class RENDER_API RendIntData
  {
    //private:
  public:

    typedef ColorTable::elem_t PovIntColor;

    /// output pov file
    OutStream *m_pPovOut;

    /// output inc file
    OutStream *m_pIncOut;

    /// target name
    LString m_name;

    /// Output texture type
    bool m_fUseTexBlend;

    /// Color table
    ColorTable m_clut;

    /// Clipping plane (negative value: no clipping)
    double m_dClipZ;

    /// default alpha (opacity) for this rendering
    double m_defAlpha;

    /// style name list of Renderer used for this rendering
    LString m_styleNames;

    /// Parent display context
    FileDisplayContext *m_pdc;

    //////////

    struct Line {
      Vector4D v1, v2;
      PovIntColor col;
      double w;
    };

    std::deque<Line *> m_lines;

    //////////

    struct Cyl {
      Vector4D v1, v2;
      PovIntColor col;
      double w1, w2;
      int ndetail;
      Matrix4D *pTransf;

      ~Cyl() {
        if (pTransf!=NULL) delete pTransf;
      }
    };

    typedef std::deque<Cyl *> CylList;

    CylList m_cylinders;

    //////////

    struct Sph {
      Vector4D v1;
      PovIntColor col;
      double r;
      int ndetail;
    };

    typedef std::deque<Sph *> SphList;

    /// sphere data
    SphList m_spheres;

    /// point data
    SphList m_dots;

    //////////

    struct MeshElem {
      Vector4D v, n;
      PovIntColor c;
    };

    struct Mesh
    {
      bool m_bLighting;

      RendIntData *m_pPar;
      std::deque<MeshElem *> m_verts;
      std::deque<int> m_faces;

      ~Mesh()
      {
        clear();
      }

      int getVertexSize() const { return m_verts.size(); }
      int getFaceSize() const { return m_faces.size()/3; }

      int addVertex(const Vector4D &v, const Vector4D &n, const PovIntColor &c)
      {
        MeshElem *p = MB_NEW MeshElem;
        p->v = v;
        p->n = n;
        p->c = c;

        int rval = m_verts.size();
        m_verts.push_back(p);
        return rval;
      }

      int addVertex(const Vector4D &v, const Vector4D &n, const PovIntColor &c, const Matrix4D &xfm)
      {
        Vector4D xv = v;
        xv.w() = 1.0;
        xfm.xform4D(xv);

        Vector4D xn = n;
        xn.w() = 0.0;
        xfm.xform4D(xn);

        return addVertex(xv, xn, c);
      }

      int addVertex(const Vector4D &v, const Vector4D &n, const ColorPtr &c)
      {
        return addVertex(v, n, m_pPar->convCol(c));
      }

      int addVertex(MeshElem *p)
      {
        int rval = m_verts.size();
        m_verts.push_back(p);
        return rval;
      }

      int copyVertex(MeshElem *pOrig)
      {
        return addVertex(pOrig->v, pOrig->n, pOrig->c);
      }

      void addFace(int iv0, int iv1, int iv2)
      {
        m_faces.push_back(iv0);
        m_faces.push_back(iv1);
        m_faces.push_back(iv2);
      }

      void clear()
      {
        std::for_each(m_verts.begin(), m_verts.end(), qlib::delete_ptr<MeshElem *>());
        m_verts.clear();
        m_faces.clear();
      }
    };

    Mesh m_mesh;

    // int m_nMeshMode;

    /// Mesh pivot
    int m_nMeshPivot;

    /// Material's @COLOR@ replaced flag table
    std::set<LString> m_matColReplTab;

    inline bool getMatColRepl(const LString &matname) const
    {
      std::set<LString>::const_iterator iter = m_matColReplTab.find(matname);
      if (iter==m_matColReplTab.end())
        return false; // not found
      return true;
    }

  public:
    /** mesh modes */
    enum {
      POV_INTD_TRIGS,
      POV_INTD_TRIGSTRIP,
      POV_INTD_INTARY
    };


  public:
    RendIntData(FileDisplayContext *pdc);
    virtual ~RendIntData();

    void start(OutStream *fp, OutStream *ifp, const char *name);
    void end();

    /// Append line segment
    void line(const Vector4D &v1, const Vector4D &v2, double w, const ColorPtr &col = ColorPtr());

    /// Append point
    void dot(const Vector4D &v1, double w, const ColorPtr &col = ColorPtr());

    /// Append cylinder
    void cylinder(const Vector4D &v1, const Vector4D &v2,
                  double w1, double w2,
                  int ndet, const Matrix4D *ptrf,
                  const ColorPtr &col = ColorPtr());

    /// Append sphere
    void sphere(const Vector4D &v1, double w, int ndet, const ColorPtr &col = ColorPtr());

    //////////
    // Mesh drawing operations

    void meshStart();
    void meshEndTrigs();
    void meshEndTrigStrip();
    void meshEndFan();

    /// Mesh generation for trigs & trigstrip
    void meshVertex(const Vector4D &v1, const Vector4D &n1, const ColorPtr &col)
    {
      m_mesh.addVertex(v1, n1, col);
    }

    // / mesh generation for trigfan
    // void mesh_fan(const Vector4D &v1, const Vector4D &n1, const LColor &c1, bool bMakeTri);

    void mesh(const Matrix4D &mat, const gfx::Mesh &mesh);

    // void setMeshMode(int n);
    // int getMeshMode() const { return m_nMeshMode; }

    /// Convert color to internal representation
    PovIntColor convCol();

    /// Convert color to internal representation (2)
    PovIntColor convCol(const ColorPtr &col);

    /// convert line to cylinder
    void convLines();

    /// convert dot to sphere
    void convDots();

    /// convert spheres to mesh
    void convSpheres();

    /// convert cylinders to mesh
    void convCylinders();

  private:
    void convSphere(Sph *);
    void convCyl(Cyl *);
    int selectTrig(int j, int k, int j1, int k1);

    /////////////////////////////////////////////////

  public:

    /// dump CLUT to POV file
    void dumpClut(OutStream *fp);
    //void dumpClutTex(OutStream *fp);

    // Vector4D getClutRGBA(int ind) const;

    void writeColor(const PovIntColor &id);

    bool writeLines(PrintStream &ps, PrintStream &ips);
    bool writeMeshes(PrintStream &ps, PrintStream &ips);
    int convTexInd(MeshElem *);

    bool writeSpheres(PrintStream &ps, PrintStream &ips);
    bool writeCyls(PrintStream &ps, PrintStream &ips);

    /// Mesh generation for gfx::Mesh
    //int meshVertex2(const Vector4D &v1, const Vector4D &n1, const ColorPtr &c1, FileDisplayContext *pdc);

    //int cutEdge(const Matrix4D &mat, const gfx::Mesh &rmesh, FileDisplayContext *pdc, int id1, int id2);

    //int addMeshFace(int iv0, int iv1, int iv2);

    void writeTextureList();

    void writeGradTexture();

    RendIntData::Mesh *calcMeshClip();
    RendIntData::MeshElem *cutEdge(MeshElem *pv1, MeshElem *pv2);

    double getDefaultAlpha() const {
      return m_defAlpha;
    }

    const LString &getStyleNames() const {
      return m_styleNames;
    }
  };

}

#endif

