// -*-Mode: C++;-*-
//
//  Povray display context implementation
//
//  $Id: RendIntData.cpp,v 1.16 2011/04/17 10:56:39 rishitani Exp $

#include <common.h>

#include "RendIntData.hpp"
#include <qlib/PrintStream.hpp>
#include <gfx/SolidColor.hpp>
#include <gfx/Mesh.hpp>
#include <qsys/style/StyleMgr.hpp>
#include "FileDisplayContext.hpp"

using namespace render;
using gfx::DisplayContext;
using qsys::StyleMgr;

RendIntData::RendIntData(FileDisplayContext *pdc)
     : m_pPovOut(NULL), m_pIncOut(NULL)
{
  m_pdc = pdc;
  m_dClipZ = -1.0;
  m_mesh.m_pPar = this;
}

RendIntData::~RendIntData()
{
  std::for_each(m_lines.begin(), m_lines.end(), qlib::delete_ptr<Line *>());
  std::for_each(m_cylinders.begin(), m_cylinders.end(), qlib::delete_ptr<Cyl *>());
  std::for_each(m_spheres.begin(), m_spheres.end(), qlib::delete_ptr<Sph *>());
}

/////

void RendIntData::start(OutStream *fp, OutStream *ifp, const char *name)
{
  m_pPovOut = fp;
  m_pIncOut = ifp;
  m_name = name;
  //m_fUseTexBlend = false;
  m_fUseTexBlend = true;
}

void RendIntData::meshStart()
{
  m_nMeshPivot = m_mesh.getVertexSize();
}

void RendIntData::meshEndTrigs()
{
  int nVerts = m_mesh.getVertexSize() - m_nMeshPivot;
  if (nVerts%3!=0) {
    LOG_DPRINTLN("RendIntData> Trig mesh: nVerts%3!=0 !!");
  }
  int nFaces = nVerts/3;
  int i;
  for (i=0; i<nFaces; i++) {
    m_mesh.addFace(m_nMeshPivot + i*3+0,
                   m_nMeshPivot + i*3+1,
                   m_nMeshPivot + i*3+2);
  }
}

void RendIntData::meshEndTrigStrip()
{
  int nVerts = m_mesh.getVertexSize() - m_nMeshPivot;
  if (nVerts<3) {
    LOG_DPRINTLN("RendIntData> TrigStrip mesh: nVerts<3 !!");
  }
  int i;
  for (i=2; i<nVerts; i++) {
    if (i%2==0) {
      m_mesh.addFace(m_nMeshPivot + i-2,
                  m_nMeshPivot + i-1,
                  m_nMeshPivot + i);
    }
    else {
      m_mesh.addFace(m_nMeshPivot + i,
                  m_nMeshPivot + i-1,
                  m_nMeshPivot + i-2);
    }
  }
}

void RendIntData::meshEndFan()
{
  int nVerts = m_mesh.getVertexSize() - m_nMeshPivot;
  if (nVerts<3) {
    LOG_DPRINTLN("RendIntData> TrigFan mesh: nVerts<3 !!");
  }
  int i;
  for (i=2; i<nVerts; i++) {
    m_mesh.addFace(m_nMeshPivot + i-1,
                   m_nMeshPivot + i,
                   m_nMeshPivot);
  }
}

void RendIntData::mesh(const Matrix4D &mat, const gfx::Mesh &rmesh)
{
  int i;
  const int nverts = rmesh.getVertSize();
  const int nfaces = rmesh.getFaceSize();
  const int *pinds = rmesh.getFaces();

  const int ivstart = m_mesh.getVertexSize();
  const int nPolyMode = m_pdc->getPolygonMode();
  
  Vector4D v, n;
  ColorPtr col;
  const double lw = m_pdc->getLineWidth();

  if (nPolyMode == DisplayContext::POLY_FILL) {

    for (i=0; i<nverts; ++i) {
      v = rmesh.getVertex(i);
      mat.xform3D(v);
      n = rmesh.getNormal(i);
      mat.xform4D(n);

      if (!rmesh.getCol(col, i))
        col = m_pdc->getCurrentColor();

      /*
      LString colmtr = col->getMaterial();
      if (colmtr.isEmpty())
        colmtr = pdc->getMaterial();
      p->c = m_clut.newColor(col, colmtr);
       */

      m_mesh.addVertex(v, n, col);
    }

    for (i=0; i<nfaces; ++i) {
      m_mesh.addFace(pinds[i*3+0] + ivstart,
                     pinds[i*3+1] + ivstart,
                     pinds[i*3+2] + ivstart);
    }

  }
  else if (nPolyMode == DisplayContext::POLY_LINE) {
    // XXX: this impl draw each edge twice
    for (i=0; i<nfaces; ++i) {
      const int i0 = pinds[i*3+0];
      const int i1 = pinds[i*3+1];
      const int i2 = pinds[i*3+2];

      Vector4D v0 = rmesh.getVertex(i0);
      Vector4D v1 = rmesh.getVertex(i1);
      Vector4D v2 = rmesh.getVertex(i2);

      mat.xform3D(v0);
      mat.xform3D(v1);
      mat.xform3D(v2);

      if (!rmesh.getCol(col, i0))
        col = m_pdc->getCurrentColor();
      line(v0, v1, lw, col);

      if (!rmesh.getCol(col, i1))
        col = m_pdc->getCurrentColor();
      line(v1, v2, lw, col);

      if (!rmesh.getCol(col, i2))
        col = m_pdc->getCurrentColor();
      line(v2, v0, lw, col);
    }

  }
  else if (nPolyMode == DisplayContext::POLY_POINT) {

    for (i=0; i<nverts; ++i) {
      v = rmesh.getVertex(i);
      mat.xform3D(v);
      // n = rmesh.getNormal(i);
      // mat.xform4D(n);

      if (!rmesh.getCol(col, i))
        col = m_pdc->getCurrentColor();

      dot(v, lw, col);
      //sphere(v, lw*line_scale, 2, col);
      //m_mesh.addVertex(v, n, col);
    }

  }
}

#if 0
void RendIntData::setMeshMode(int n)
{
  if (m_nMeshMode==n)
    return;
  /*if (m_mesh.size()>0) {
    // TO DO: throw exception
    LOG_DPRINTLN("POVWriter> ERROR: mesh mode cannot be changed during rendering.");
  }*/
  
  m_nMeshMode = n;
}
#endif

/// Append line segment
void RendIntData::line(const Vector4D &v1, const Vector4D &v2, double width, const ColorPtr &col)
{
  Line *p = MB_NEW Line;
  p->v1 = v1;
  p->v2 = v2;
  if (col.isnull())
    p->col = convCol();
  else
    p->col = convCol(col);
  p->w = width;
  m_lines.push_back(p);
}

/// Append point
void RendIntData::dot(const Vector4D &v1, double w, const ColorPtr &col)
{
  Sph *p = MB_NEW Sph;
  p->v1 = v1;

  // p->col = convCol();
  if (col.isnull())
    p->col = convCol();
  else
    p->col = convCol(col);

  p->r = w;
  p->ndetail = 0;
  m_dots.push_back(p);
}

/// Append cylinder
void RendIntData::cylinder(const Vector4D &v1, const Vector4D &v2,
                          double w1, double w2,
                          int ndet, const Matrix4D *ptrf,
                          const ColorPtr &col)
{
  Cyl *p = MB_NEW Cyl;
  p->v1 = v1;
  p->v2 = v2;

  if (col.isnull())
    p->col = convCol();
  else
    p->col = convCol(col);

  p->w1 = w1;
  p->w2 = w2;
  p->ndetail = ndet;
  if (ptrf==NULL)
    p->pTransf = NULL;
  else
    p->pTransf = MB_NEW Matrix4D(*ptrf);
  m_cylinders.push_back(p);
}

void RendIntData::sphere(const Vector4D &v1, double w, int ndet, const ColorPtr &col)
{
  Sph *p = MB_NEW Sph;
  p->v1 = v1;

  // p->col = convCol();
  if (col.isnull())
    p->col = convCol();
  else
    p->col = convCol(col);

  p->r = w;
  p->ndetail = ndet;
  m_spheres.push_back(p);
}

/// convert LColor to CLUT entry
RendIntData::PovIntColor RendIntData::convCol()
{
  const ColorPtr &pcol = m_pdc->getCurrentColor();
  LString defmtr = m_pdc->getMaterial();
  return m_clut.newColor(pcol, defmtr);
}

/// Convert color to internal representation (2)
RendIntData::PovIntColor RendIntData::convCol(const ColorPtr &pcol)
{
  LString defmtr = m_pdc->getMaterial();
  return m_clut.newColor(pcol, defmtr);
}

RendIntData::Mesh *RendIntData::calcMeshClip()
{
  int i, j;
  int nverts = m_mesh.getVertexSize();
  int nfaces = m_mesh.getFaceSize();

  std::vector<int> vidmap(nverts);
  
  RendIntData::Mesh *pMesh2 = MB_NEW RendIntData::Mesh();
  pMesh2->m_pPar = this;

  std::vector<MeshElem *> verts(nverts);
  bool bNone = true;
  i=0;
  BOOST_FOREACH (MeshElem *p, m_mesh.m_verts) {
    verts[i] = p;
    if (p->v.z()>=m_dClipZ) {
      // clipped
      vidmap[i] = -1;
      bNone = false;
      ++i;
      continue;
    }

    vidmap[i] = 1;
    ++i;
  }  

  if (bNone) {
    delete pMesh2;
    return NULL;
  }

  i=0; j=0;
  BOOST_FOREACH (MeshElem *p, m_mesh.m_verts) {
    if (vidmap[i]<0) {
      ++i;
      continue;
    }
    vidmap[i] = j;
    pMesh2->copyVertex(p);
    ++j;
    ++i;
  }  

  std::deque<int>::iterator iter2 = m_mesh.m_faces.begin();
  std::deque<int>::iterator iend2 = m_mesh.m_faces.end();
  int iv[3];
  int jv[3];
  bool b[3];
  for (i=0; iter2!=iend2; iter2++, i++) {
    iv[0] = *iter2;
    iter2++; MB_ASSERT(iter2!=m_mesh.m_faces.end());
    iv[1] = *iter2;
    iter2++; MB_ASSERT(iter2!=m_mesh.m_faces.end());
    iv[2] = *iter2;

    for (j=0; j<3; ++j)
      jv[j] = vidmap[iv[j]];
    
    for (j=0; j<3; ++j)
      b[j] = (jv[j]<0);
    
    if (b[0] && b[1] && b[2]) {
      // outside clip
      continue;
    }
    else if (!b[0] && !b[1] && !b[2]) {
      // inside clip
      pMesh2->addFace(jv[0], jv[1], jv[2]);
      continue;
    }
    
    for (j=0; j<3; ++j) {

      MeshElem *pv1 = verts[ iv[j] ];
      MeshElem *pv2 = verts[ iv[(j+1)%3] ];
      MeshElem *pv3 = verts[ iv[(j+2)%3] ];
      
      // check single clipped triangles
      if (!b[j] && b[(j+1)%3] && b[(j+2)%3]) {
        int jin = jv[j];
        MeshElem *pv1 = verts[ iv[j] ];
        MeshElem *pv2 = verts[ iv[(j+1)%3] ];
        MeshElem *pv3 = verts[ iv[(j+2)%3] ];
        int inv2 = pMesh2->addVertex( cutEdge(pv1, pv2) );
        int inv3 = pMesh2->addVertex( cutEdge(pv1, pv3) );
        pMesh2->addFace(jin, inv2, inv3);
        break;
      }

      // check double clipped triangles
      // TO DO: select convex-hull triangles
      if (!b[j] && !b[(j+1)%3] && b[(j+2)%3]) {
        int jin0 = jv[j];
        int jin1 = jv[(j+1)%3];
        int ed1 = pMesh2->addVertex( cutEdge(pv3, pv1) );
        int ed2 = pMesh2->addVertex( cutEdge(pv3, pv2) );

        //if (select_trig(m_verts[nin0], m_verts[ed1], m_verts[nin1], m_verts[ed2])) {
        pMesh2->addFace(jin0, jin1, ed1);
        pMesh2->addFace(ed2, ed1, jin1);
        //}
        //else {
        //addNewFace(nin0, nin1, ed2);
        //addNewFace(ed2, ed1, nin0);
        //}
        break;
      }
      
    }
  }

  return pMesh2;
}

RendIntData::MeshElem *RendIntData::cutEdge(MeshElem *pv1, MeshElem *pv2)
{
  const Vector4D &v1 = pv1->v;
  const Vector4D &v2 = pv2->v;
  
  const Vector4D &n1 = pv1->n;
  const Vector4D &n2 = pv2->n;
  
  MeshElem *pNew = MB_NEW MeshElem;
  const double t = (m_dClipZ-v1.z())/(v2.z()-v1.z());
  pNew->v = v2.scale(t) + v1.scale(1.0-t);
  
  Vector4D norm = n2.scale(t) + n1.scale(1.0-t);
  pNew->n = norm.normalize();
  
  pNew->c = pv1->c;
  return pNew;
}

/// convert dot to sphere
void RendIntData::convDots()
{
  if (m_dots.size()<=0)
    return;

  const double line_scale = m_pdc->getLineScale();

  BOOST_FOREACH (Sph *p, m_dots) {
    p->r *= line_scale;
    p->ndetail = 4;
    m_spheres.push_back(p);
  }

  m_dots.erase(m_dots.begin(), m_dots.end());
}

/// convert line to cylinder
void RendIntData::convLines()
{
  if (m_lines.size()<=0)
    return;
  const double lsc = m_pdc->getLineScale();
  BOOST_FOREACH (RendIntData::Line *p, m_lines) {
    double w = p->w * lsc;

    //cylinder(v1, v2, w, w, 4, NULL, p->);
    Cyl *pc = MB_NEW Cyl;
    pc->v1 = p->v1;
    pc->v2 = p->v2;
    pc->col = p->col;
    pc->w1 = pc->w2 = w;
    pc->ndetail = 4;
    pc->pTransf = NULL;
    m_cylinders.push_back(pc);

    delete p;
  }

  m_lines.erase(m_lines.begin(), m_lines.end());
}

/// convert sphere to mesh
void RendIntData::convSpheres()
{
  if (m_spheres.size()<=0)
    return;

  BOOST_FOREACH (Sph *p, m_spheres) {
    convSphere(p);
    delete p;
  }

  m_spheres.erase(m_spheres.begin(), m_spheres.end());
  return;
}

// convert single sphere to mesh
void RendIntData::convSphere(Sph *pSph)
{
  const Vector4D v1 = pSph->v1;
  const double rad = pSph->r;
  PovIntColor col = pSph->col;
  const double dmax = (M_PI*rad)/double(pSph->ndetail+1);

  const int ivstart = m_mesh.getVertexSize();

  int i, j;
  int nLat = pSph->ndetail+1;

  // detail in longitude direction is automatically determined by stack radius
  int nLng;

  MB_DPRINTLN("RendIntData::convSphere v1=(%f,%f,%f) r=%f",
              pSph->v1.x(), pSph->v1.y(), pSph->v1.z(), pSph->r);
  MB_DPRINTLN("sphere R=%f, nLat=%d (%f)", rad, nLat, rad*M_PI/dmax);

  int **ppindx = new int *[nLat+1];

  // generate verteces
  for (i=0; i<=nLat; ++i) {
    int ind;
    //std::list<int> ilst;

    if (i==0) {
      ind = m_mesh.addVertex(Vector4D(0, 0, rad) + v1,
                             Vector4D(0, 0, 1),
                             col);
      //ind = putVert(Vector4D(0, 0, rad) + v1);

      ppindx[i] = new int[1];
      ppindx[i][0] = ind;
    }
    else if (i==nLat) {
      ind = m_mesh.addVertex(Vector4D(0, 0, -rad) + v1,
                             Vector4D(0, 0, -1),
                             col);
      //ind = putVert(Vector4D(0, 0, -rad) + v1);
      //ilst.push_back(ind);

      ppindx[i] = new int[1];
      ppindx[i][0] = ind;
    }
    else {
      Vector4D vec, norm;
      const double th = double(i)*M_PI/double(nLat);
      const double ri = rad*::sin(th);
      vec.z()  = rad*::cos(th);
      nLng = (int) ::ceil(ri*M_PI*2.0/dmax);
      ppindx[i] = new int[nLng+2];
      ppindx[i][0] = nLng;
      const double start_phi = double(i%2) * 3.0 / nLng;
      //MB_DPRINTLN("Lat: %d start phi=%f", i, start_phi);
      for (j=0; j<nLng; ++j) {
        double ph = double(j)*M_PI*2.0/double(nLng) + start_phi;
        vec.x() = ri*::cos(ph);
        vec.y() = ri*::sin(ph);
        norm = vec.normalize();
        ind = m_mesh.addVertex(vec + v1, norm, col);
        //ind = putVert(vec + v1);

        ppindx[i][j+1] = ind;
      }
      ppindx[i][j+1] = ppindx[i][1];
    }
  } // for (i)

  // build faces from verteces
  for (i=0; i<nLat; ++i) {
    if (i==0) {
      int ipiv = ppindx[0][0];
      int nLng = ppindx[1][0];
      for (j=0; j<nLng; ++j) {
        //putFace(ipiv, ppindx[1][j+1], ppindx[1][j+2], icol);
        m_mesh.addFace(ipiv, ppindx[1][j+1], ppindx[1][j+2]);
      }
    }
    else if (i==nLat-1) {
      int ipiv = ppindx[nLat][0];
      int nLng = ppindx[nLat-1][0];
      for (j=0; j<nLng; ++j) {
        //putFace(ppindx[nLat-1][j+2], ppindx[nLat-1][j+1], ipiv, icol);
        m_mesh.addFace(ppindx[nLat-1][j+2], ppindx[nLat-1][j+1], ipiv);
      }
    }
    else /*if (i==2)*/ {

      int j = 0, k = 0;
      int bJ;

      int jmax = ppindx[i][0];
      int *piJ = &(ppindx[i][1]);

      int kmax = ppindx[i+1][0];
      int *piK = &(ppindx[i+1][1]);

//      double am1, am2;
      while (j+1<=jmax || k+1<=kmax) {
        if (j+1>jmax) bJ = 0;
        else if (k+1>kmax) bJ = 1;
        else bJ = selectTrig(piJ[j], piK[k], piJ[j+1], piK[k+1]);

        if (bJ==1) {
          m_mesh.addFace(piJ[j], piK[k], piJ[j+1]);
          //putFace(piJ[j], piK[k], piJ[j+1], icol);
          ++j;
        }
        else /*if (bJ==0)*/ {
          m_mesh.addFace(piJ[j], piK[k], piK[k+1]);
          //putFace(piJ[j], piK[k], piK[k+1], icol);
          ++k;
        }
      } // while

    }
  } // for (i)

  for (i=0; i<=nLat; ++i)
    delete [] ppindx[i];
  delete [] ppindx;
}

static
inline Vector4D makenorm(const Vector4D &pos1,
                         const Vector4D &pos2,
                         const Vector4D &pos3)
{
  const Vector4D v12 = pos2 - pos1;
  const Vector4D v23 = pos3 - pos2;
  Vector4D vn = v12.cross(v23);
  const double dnorm = vn.length();
  // if (dnorm<dtol) {
  //   throw error!!
  // }
  vn /= dnorm;
  return vn;
}

int RendIntData::selectTrig(int j, int k, int j1, int k1)
{
  const Vector4D &vj =  m_mesh.m_verts[j]->v;
  const Vector4D &vk =  m_mesh.m_verts[k]->v;
  const Vector4D &vj1 = m_mesh.m_verts[j1]->v;
  const Vector4D &vk1 = m_mesh.m_verts[k1]->v;

  Vector4D nj1 = makenorm(vj, vk, vj1);
  Vector4D nk1 = makenorm(vj, vk, vk1);
  
  double detj = nj1.dot(vk1-vk);
  double detk = nk1.dot(vj1-vj);

  if (detj<0 && detk>=0)
    return 1; // select j1

  if (detj>=0 && detk<0)
    return 0; // select k1

  MB_DPRINTLN("SelectTrig warning; (%d,%d,%d,%d) detj=%f, detk=%f", j, k, j1, k1, detj, detk);
  return 2;
  /*
  double am1 = angmax(m_verts[j].v3d(), m_verts[k].v3d(), m_verts[j1].v3d());
  double am2 = angmax(m_verts[j].v3d(), m_verts[k].v3d(), m_verts[k1].v3d());
  return (am1<am2);
   */
}

/////////////////////////////

/// convert cylinders to mesh
void RendIntData::convCylinders()
{
  if (m_cylinders.size()<=0)
    return;

  BOOST_FOREACH (Cyl *p, m_cylinders) {
    convCyl(p);
    delete p;
  }

  m_cylinders.erase(m_cylinders.begin(), m_cylinders.end());
  return;
}

/// convert cylinder to mesh
void RendIntData::convCyl(Cyl *pCyl)
{
  Vector4D cylv1(pCyl->v1), cylv2(pCyl->v2);
  PovIntColor col = pCyl->col;

  MB_DPRINTLN("=== RendIntData::convCyl ===");

  Vector4D nn = cylv1 - cylv2;
  double len = nn.length();
  if (len<=F_EPS4) {
    // ignore a degenerated cylinder
    return;
  }
  
  nn = cylv1 - cylv2;
  len = nn.length();
  nn = nn.scale(1.0/len);
  
  MB_DPRINTLN("nn: (%f,%f,%f)", nn.x(), nn.y(), nn.z());
  MB_DPRINTLN("v1: (%f,%f,%f)", cylv1.x(), cylv1.y(), cylv1.z());
  MB_DPRINTLN("v2: (%f,%f,%f)", cylv2.x(), cylv2.y(), cylv2.z());

  const Vector4D ex(1,0,0), ey(0,1,0), ez(0,0,1);
  Vector4D n1, n2;
  if (qlib::abs(nn.dot(ex)) < 0.9) {
    n1 = nn.cross(ex);
  }
  else if (qlib::abs(nn.dot(ey)) < 0.9) {
    n1 = nn.cross(ey);
  }
  else if (qlib::abs(nn.dot(ez)) < 0.9) {
    n1 = nn.cross(ez);
  }
  else {
    LOG_DPRINTLN("ConvCYL fatal error !!");
    return;
  }
  n1 = n1.normalize();
  Matrix4D mat = Matrix4D::makeRotMat(nn, n1);

  //
  // generate verteces
  //

  int i, j;
  double th;
  const double w1 = pCyl->w1;
  const double w2 = pCyl->w2;
  
  const int NDIVR = 2*(pCyl->ndetail+1);
  const double dth = (M_PI*2.0)/NDIVR;
  
  //const int NDIVV = qlib::max(2, (int) ::floor(len/((pCyl->w1)*dth)));
  const int NDIVV = 2;
  const double dw = (w2-w1)/double(NDIVV-1);
  const double dlen = len/double(NDIVV-1);
  
  MB_DPRINTLN("cyl ndiv r,v =(%d, %d)", NDIVR, NDIVV);
  
  Matrix4D xfm;
  if (pCyl->pTransf!=NULL)
    xfm = *(pCyl->pTransf);
  
  xfm.matprod( Matrix4D::makeTransMat(cylv2) );
  xfm.matprod( mat );

  // putVert(Vector4D(0,0,0));
  const int ivbot = 
    m_mesh.addVertex(Vector4D(0, 0, 0),
                     Vector4D(0, 0, -1),
                     col, xfm);
  
  for (j=0; j<NDIVV; ++j) {
    const double ww = w1 + dw*double(j);
    const double zz = dlen*double(j);
    for (th=0.0, i=0; i<NDIVR; ++i, th += dth) {
      const double costh = ::cos(th);
      const double sinth = ::sin(th);
      const double xx = ww * costh;
      const double yy = ww * sinth;
      // putVert(Vector4D(xx, yy, zz));
      m_mesh.addVertex(Vector4D(xx, yy, zz),
                       Vector4D(costh, sinth, 0),
                       col, xfm);
    }
  }
  
  // putVert(Vector4D(0,0,len));
  const int ivtop = 
    m_mesh.addVertex(Vector4D(0,0,len),
                     Vector4D(0, 0, 1),
                     col, xfm);

  //MB_DPRINTLN("vbot: (%f,%f,%f)", m_verts[ivbot].x(), m_verts[ivbot].y(), m_verts[ivbot].z());
  //MB_DPRINTLN("vtop: (%f,%f,%f)", m_verts[ivtop].x(), m_verts[ivtop].y(), m_verts[ivtop].z());

  //
  // connect verteces & make faces
  //

  // bottom disk
  for (i=0; i<=NDIVR; ++i) {
    const int ii = i%NDIVR;
    const int jj = (i+1)%NDIVR;
    //putFace(ivbot,
    //ivbot + 1 + jj,
    //ivbot + 1 + ii,
    //icol);
    m_mesh.addFace(ivbot,
                   ivbot + 1 + jj,
                   ivbot + 1 + ii);
  }

  // cylinder body
  for (j=0; j<NDIVV-1; ++j) {
    const int u = 1 + j*NDIVR;
    const int v = 1 + (j+1)*NDIVR;
    for (i=0; i<NDIVR; ++i) {
      const int ii = i%NDIVR;
      const int jj = (i+1)%NDIVR;
      /*
      putFace(ivbot + u + ii,
              ivbot + u + jj,
              ivbot + v + jj,
              icol);

      putFace(ivbot + u + ii,
              ivbot + v + jj,
              ivbot + v + ii,
              icol);
       */
      m_mesh.addFace(ivbot + u + ii,
                     ivbot + u + jj,
                     ivbot + v + jj);
      m_mesh.addFace(ivbot + u + ii,
                     ivbot + v + jj,
                     ivbot + v + ii);
    }
  }


  // top disk
  for (i=0; i<=NDIVR; ++i) {
    const int ii = i%NDIVR;
    const int jj = (i+1)%NDIVR;

    //putFace(ivtop,
    //ivbot + 1 + (NDIVV-1)*NDIVR + ii,
    //ivbot + 1 + (NDIVV-1)*NDIVR + jj,
    //icol);
    m_mesh.addFace(ivtop,
                   ivbot + 1 + (NDIVV-1)*NDIVR + ii,
                   ivbot + 1 + (NDIVV-1)*NDIVR + jj);
  }
}


/////////////////////////////////////////////////////////////////////////////
// POV-Ray specific impl (should be moved to POVDisplayContext!!)

/// dump CLUT to POV file
void RendIntData::dumpClut(OutStream *fp)
{
  PrintStream ps(*fp);
  StyleMgr *pSM = StyleMgr::getInstance();
  double defalpha = m_pdc->getAlpha();

  bool bDefAlpha = false;
  if (!m_pdc->isPostBlend() &&
      !qlib::Util::isNear4(defalpha, 1.0))
    bDefAlpha =true;

  int i;
  for (i=0; i<m_clut.size(); i++) {
    PovIntColor cind;
    cind.cid1 = i;
    cind.cid2 = -1;

    // get color
    Vector4D vc;
    m_clut.getRGBAVecColor(cind, vc);
      
    // write color
    ps.format("#declare %s_col_%d = ", m_name.c_str(), i);
    if (!bDefAlpha && qlib::Util::isNear4(vc.w(), 1.0))
      ps.format("<%f,%f,%f>;\n", vc.x(), vc.y(), vc.z());
    else
      ps.format("<%f,%f,%f,%f>;\n", vc.x(), vc.y(), vc.z(), 1.0-(vc.w()*defalpha));

    LString colortext = LString::format("%s_col_%d", m_name.c_str(), i);

    // get material
    LString mat;
    m_clut.getMaterial(cind, mat);
    if (mat.isEmpty()) mat = "default";
    LString matdef = pSM->getMaterial(mat, "pov");
    matdef = matdef.trim(" \r\n\t");
      
    // write material
    ps.format("#declare %s_tex_%d = ", m_name.c_str(), i);
    if (!matdef.isEmpty()) {
      if (matdef.replace("@COLOR@", colortext)>0)
        m_matColReplTab.insert(mat);
      ps.println(matdef);
    }
    else
      ps.println("texture {pigment{color rgbt "+colortext+"}}");

  }
}

void RendIntData::writeColor(const PovIntColor &ic)
{
  PrintStream ips(*m_pIncOut);

  const char *nm = m_name.c_str();
  if (ic.cid2<0) {
    //
    // Non-gradient color
    //
    if (m_fUseTexBlend) {
      ips.format("texture{%s_tex_%d}", nm, ic.cid1);
    }
    else {
      ips.format("texture{%s_tex_%d ", nm, ic.cid1);
      ips.format("pigment {color rgbt %s_col_%d}}", nm, ic.cid1);
    }
  }
  else{
    //
    // Gradient color
    //
    if (m_fUseTexBlend) {

      PovIntColor cind;
      cind.cid2 = -1;

      LString mat1, mat2;
      cind.cid1 = ic.cid1;
      m_clut.getMaterial(cind, mat1);
      if (mat1.isEmpty()) mat1 = "default";
      cind.cid1 = ic.cid2;
      m_clut.getMaterial(cind, mat2);
      if (mat2.isEmpty()) mat2 = "default";

      bool bMatRepl1 = getMatColRepl(mat1);
      bool bMatRepl2 = getMatColRepl(mat2);

      if (mat1.equals(mat2) && bMatRepl1 && bMatRepl2) {
        ips.format("texture{%s_tex_%d ", nm, ic.cid1);
        ips.format(
          "pigment {color rgbt %s_col_%d*%f+%s_col_%d*%f}}",
          nm, ic.cid1, ic.getRhoF(), nm, ic.cid2, 1.0-ic.getRhoF());
      }
      else {
        ips.format(
          "texture{function{%.6f}texture_map{[0 %s_tex_%d][1 %s_tex_%d]}}",
          1.0f-ic.getRhoF(), nm, ic.cid1, nm, ic.cid2);
      }
    }
    else
    {
      ips.format("texture{%s_tex_%d ", nm, ic.cid1);
      ips.format(
        "pigment {color rgbt %s_col_%d*%f+%s_col_%d*%f}}",
        nm, ic.cid1, ic.getRhoF(), nm, ic.cid2, 1.0-ic.getRhoF());
    }
  }
}

bool RendIntData::writeLines(PrintStream &ps, PrintStream &ips)
{
  if (m_lines.size()<=0)
    return false;

  // write INC file
  BOOST_FOREACH(Line *p, m_lines) {

    Vector4D v1 = p->v1, v2 = p->v2;
    double w = p->w;

    // always keep v1.z < v2.z
    if (v1.z()>v2.z())
      std::swap(v1, v2);

    Vector4D nn = v2 - v1;
    double len = nn.length();
    if (len<=F_EPS4) {
      // ignore degenerated cylinder
      delete p;
      continue;
    }
    
    if (m_dClipZ<0) {
      ips.format("cylinder{<%f, %f, %f>, ", v1.x(), v1.y(), v1.z());
      ips.format("<%f, %f, %f>, ", v2.x(), v2.y(), v2.z());
      ips.format("%s_lw*%f ", m_name.c_str(), w);
      writeColor(p->col);
      ips.format("}\n");
    }
    else if (m_dClipZ > v1.z()) {

      if (m_dClipZ < v2.z())
        v2 = nn.scale((m_dClipZ-v1.z())/(nn.z())) + v1;

      ips.format("cylinder{<%f, %f, %f>, ", v1.x(), v1.y(), v1.z());
      ips.format("<%f, %f, %f>, ", v2.x(), v2.y(), v2.z());
      ips.format("%s_lw*%f ", m_name.c_str(), w);
      writeColor(p->col);

      ips.format("}\n");
    }
    delete p;
  }

  m_lines.erase(m_lines.begin(), m_lines.end());

  const double line_scale = m_pdc->getLineScale();
  // write POV file
  ps.format("#declare %s_lw = %.3f;\n", m_name.c_str(), line_scale);
  ps.format("#declare %s_tex = texture {\n", m_name.c_str());
  ps.format("  pigment {color rgbft <0,0,0,1,1>}\n");
  ps.format("  normal {granite 0.0 scale 1.0}\n");
  ps.format("  finish {\n");
  ps.format("   ambient 0.3\n");
  ps.format("   diffuse 1.0\n");
  ps.format("   specular 0.0\n");
  ps.format("  }\n");
  ps.format(" }\n");

  return true;
}

/// convert gradient-map index to the POV's mesh texture index
int RendIntData::convTexInd(MeshElem *p1)
{
  if ( p1->c.cid2 < 0 ) {
    return p1->c.cid1;
  }
  else {
    int gind = m_clut.getGradIndex(p1->c);
    if (gind<0) {
      LOG_DPRINTLN("FATAL ERROR: cannot convert gradient index!!");
      return 0;
    }

    return m_clut.size() + gind*256 + p1->c.getRhoI();
  }
}

void RendIntData::writeTextureList()
{
  PrintStream ips(*m_pIncOut);

  int txsize = m_clut.size() + m_clut.m_grads.size()*256;
  ips.format("texture_list{ %d", txsize);

  // write simple textures
  for (int i=0; i<m_clut.size(); i++) {
    if (i%2==0)
      ips.format(",\n");
    else
      ips.format(",");

    /*
    if (m_fUseTexBlend)
      ips.format("texture{%s_tex%d}", nm, i);
    else
      ips.format("texture{%s pigment{color rgbt %s_col%d}}",
                 txnm, nm, i);
     */
    PovIntColor pic;
    pic.cid1 = i;
    pic.cid2 = -1;
    writeColor(pic);
  }
}

void RendIntData::writeGradTexture()
{
  PrintStream ips(*m_pIncOut);

  m_clut.indexGradients();

  BOOST_FOREACH(ColorTable::grads_type::value_type &entry, m_clut.m_grads) {
    //ColorTable::grads_type::const_iterator gmi = m_clut.m_grads.begin();
    //for ( ; gmi!=m_clut.m_grads.end(); gmi++) {
    //const PovIntColor &gd = gmi->first;

    const PovIntColor &gd = entry.first;
    for (int j=0; j<256; j++) {
      double rho = double(j)/255.0f;
      ips.format(",\n");
      //if (m_fUseTexBlend)
      //ips.format(
      //"texture{function{%.6f}texture_map{[0 %s_tex%d][1 %s_tex%d]}}",
      //1.0f-rho, nm, gd.cid1, nm, gd.cid2);
      //else 
      //ips.format(
      //"texture{%s pigment {color rgbt %s_col%d*%f+%s_col%d*%f}}",
      //txnm, nm, gd.cid1, rho, nm, gd.cid2, 1.0f-rho);
      PovIntColor pic;
      pic.cid1 = gd.cid1;
      pic.cid2 = gd.cid2;
      pic.setRhoI(j);
      writeColor(pic);
    }
  }
  ips.format("}\n");
}

bool RendIntData::writeMeshes(PrintStream &ps, PrintStream &ips)
{
  int i, j;
  const char *nm = m_name.c_str();

  if (m_mesh.getVertexSize()<=0 || m_mesh.getFaceSize()<1)
    return false;
  
  Mesh *pMesh = &m_mesh;
  bool bdel = false;
  if (m_dClipZ>0.0) {
    Mesh *pRes = calcMeshClip();
    if (pRes!=NULL) {
      pMesh = pRes;
      bdel = true;
    }
  }

  int nverts = pMesh->getVertexSize();
  int nfaces = pMesh->getFaceSize();

  // convert vertex list to array
  MeshElem **pmary = MB_NEW MeshElem *[nverts];
  i=0;
  BOOST_FOREACH (MeshElem *pelem, pMesh->m_verts) {
    pmary[i] = pelem;
    ++i;
  }
  
  //
  // generate mesh2 statement
  //
  
  ips.format("mesh2{\n");
  
  // write vertex_vectors
  ips.format("vertex_vectors{ %d", nverts);
  for (i=0; i<nverts; i++) {
    MeshElem *p = pmary[i];
    if (i%6==0)
      ips.format(",\n");
    else
      ips.format(",");

    if (!qlib::isFinite(p->v.x()) ||
        !qlib::isFinite(p->v.y()) ||
        !qlib::isFinite(p->v.z())) {
      LOG_DPRINTLN("PovWriter> ERROR: invalid mesh vertex");
      ips.format("<0, 0, 0>");
    }
    else {
      ips.format("<%f, %f, %f>", p->v.x(), p->v.y(), p->v.z());
    }
  }
  ips.format("}\n");
  
  // write normal_vectors
  ips.format("normal_vectors{ %d", nverts);
  for (i=0; i<nverts; i++) {
    MeshElem *p = pmary[i];
    if (i%6==0)
      ips.format(",\n");
    else
      ips.format(",");

    if (!qlib::isFinite(p->n.x()) ||
        !qlib::isFinite(p->n.y()) ||
        !qlib::isFinite(p->n.z())) {
      LOG_DPRINTLN("PovWriter> ERROR: invalid mesh normal");
      ips.format("<1, 0, 0>");
    }
    else {
      ips.format("<%f, %f, %f>", p->n.x(), p->n.y(), p->n.z());
    }
  }
  ips.format("}\n");
  
  //
  // write texture_list
  //
  writeTextureList();

  //
  // write gradient textures
  //
  writeGradTexture();

  //
  // write face_indices
  //
  ips.format("face_indices{ %d", nfaces);

  std::deque<int>::iterator iter2 = pMesh->m_faces.begin();
  std::deque<int>::iterator iend2 = pMesh->m_faces.end();
  for (i=0; iter2!=iend2; iter2++, i++) {
    int i1 = *iter2;
    iter2++; MB_ASSERT(iter2!=pMesh->m_faces.end());
    int i2 = *iter2;
    iter2++; MB_ASSERT(iter2!=pMesh->m_faces.end());
    int i3 = *iter2;
    
    if (i%4==0)
      ips.format(",\n");
    else
      ips.format(",");
    
    ips.format("<%d,%d,%d>,%d,%d,%d",
               i1, i2, i3,
               convTexInd(pmary[i1]),
               convTexInd(pmary[i2]),
               convTexInd(pmary[i3]));
  }
  ips.format("}\n");
  
  ips.format("}\n");
  
  //
  // clean up
  //
  if (bdel)
    delete pMesh;
  delete [] pmary;

  m_mesh.clear();

  return true;
}


bool RendIntData::writeSpheres(PrintStream &ps, PrintStream &ips)
{
  if (m_spheres.size()<=0)
    return false;

  BOOST_FOREACH (Sph *p, m_spheres) {

    if (m_dClipZ<0) {
      // no clipping
      ips.format("sphere{<%f, %f, %f>, ", p->v1.x(), p->v1.y(), p->v1.z());
      ips.format("%f ", p->r);
      writeColor(p->col);
      ips.format("}\n");
    }
    else if (m_dClipZ > (p->v1.z() - p->r)) {
      ips.format("sphere{<%f, %f, %f>, ", p->v1.x(), p->v1.y(), p->v1.z());
      ips.format("%f ", p->r);
      writeColor(p->col);
      if (m_dClipZ < (p->v1.z() + p->r)) {
        ips.format("\n bounded_by {");
        ips.format("  plane {z, %f} ", m_dClipZ);
        ips.format("}");
        ips.format("clipped_by { bounded_by }");
      }
      ips.format("}\n");
    }
    
    delete p;
  }

  m_spheres.erase(m_spheres.begin(), m_spheres.end());
  return true;
}

bool RendIntData::writeCyls(PrintStream &ps, PrintStream &ips)
{
  if (m_cylinders.size()<=0)
    return false;

  BOOST_FOREACH (Cyl *p, m_cylinders) {

    Vector4D v1 = p->v1, v2 = p->v2;
    double w1 = p->w1, w2 = p->w2;
    Vector4D nn = v1 - v2;
    double len = nn.length();
    if (len<=F_EPS4) {
      // ignore the degenerated cylinder
      delete p;
      continue;
    }

    // elongate cyl. for connectivity
    nn = nn.scale(1.0/len);
    v1 += nn.scale(0.01);
    v2 -= nn.scale(0.01);

    // always keep v1.z < v2.z
    if (v1.z()>v2.z()) {
      std::swap(v1, v2);
      std::swap(w1, w2);
    }
    //const double dz = v2.z() - v1.z();
    //const double dh = qlib::abs(v2.y() - v1.y());
    //const double l = ::sqrt(dz*dz+dh*dh);
    const double delw = qlib::max(w1, w2);
    const double thr1 = v1.z() - delw;
    const double thr2 = v2.z() + delw;

    MB_ASSERT(v2.z()>v1.z());

    if (m_dClipZ<0) {
      // no clipping Z
      if (qlib::isNear4(w1, w2)) {
        // cyliner
        ips.format("cylinder{<%f,%f,%f>,", v1.x(), v1.y(), v1.z());
        ips.format("<%f,%f,%f>,", v2.x(), v2.y(), v2.z());
        ips.format("%s_lw*%f ", m_name.c_str(), w1);
      }
      else {
        // cone
        ips.format("cone{<%f,%f,%f>,", v1.x(), v1.y(), v1.z());
        ips.format("%s_lw*%f,", m_name.c_str(), w1);
        ips.format("<%f,%f,%f>,", v2.x(), v2.y(), v2.z());
        ips.format("%s_lw*%f ", m_name.c_str(), w2);
      }

      writeColor(p->col);
      ips.format("}\n");
    }
    else {
      if (m_dClipZ > thr1) {
        if (qlib::isNear4(w1, w2)) {
          // cyliner
          ips.format("cylinder{<%f,%f,%f>,", v1.x(), v1.y(), v1.z());
          ips.format("<%f,%f,%f>,", v2.x(), v2.y(), v2.z());
          ips.format("%s_lw*%f ", m_name.c_str(), w1);
        }
        else {
        // cone
          ips.format("cone{<%f,%f,%f>,", v1.x(), v1.y(), v1.z());
          ips.format("%s_lw*%f,", m_name.c_str(), w1);
          ips.format("<%f,%f,%f>,", v2.x(), v2.y(), v2.z());
          ips.format("%s_lw*%f ", m_name.c_str(), w2);
        }
        
        writeColor(p->col);
        
        if (p->pTransf!=NULL) {
          const Matrix4D &m = *p->pTransf;
          ips.format(" matrix <");
          ips.format("%f, %f, %f, ", m.aij(1,1), m.aij(2,1), m.aij(3,1) );
          ips.format("%f, %f, %f, ", m.aij(1,2), m.aij(2,2), m.aij(3,2) );
          ips.format("%f, %f, %f, ", m.aij(1,3), m.aij(2,3), m.aij(3,3) );
          ips.format("%f, %f, %f>" , m.aij(1,4), m.aij(2,4), m.aij(3,4) );
        }
        else if (m_dClipZ < thr2) {
          ips.format("\n bounded_by {");
          ips.format("  plane {z, %f} ", m_dClipZ);
          ips.format("}");
          ips.format("clipped_by { bounded_by }");
        }
        ips.format("}\n");
      }
    }

    delete p;
  }

  m_cylinders.erase(m_cylinders.begin(), m_cylinders.end());
  return true;
}

void RendIntData::end()
{
  m_defAlpha = m_pdc->getAlpha();
  m_styleNames = m_pdc->getStyleNames();
  
  if (m_pPovOut==NULL||m_pIncOut==NULL)
    return;

  // TO DO: move povray specific impl to PovDisplayContext!!
  bool bcyl = false, bsph = false, blin = false, bmes = false;

  convDots();
  
  if (m_cylinders.size()<=0 &&
      m_dots.size()<=0 &&
      m_spheres.size()<=0 &&
      m_mesh.getVertexSize()<=0 &&
      m_mesh.getFaceSize()<=0 &&
      m_lines.size()<=0 &&
      m_clut.size()<=0) {
    return;
  }

  PrintStream ps(*m_pPovOut);
  PrintStream ips(*m_pIncOut);

  ps.format("//\n");
  ps.format("// rendering properties for %s\n", m_name.c_str());
  ps.format("//\n");
  ps.format("#ifndef (_show%s)\n", m_name.c_str());
  ps.format("#declare _show%s = 1;\n", m_name.c_str());
  ps.format("#end\n");
  
  ips.format("\n#if (_show%s)\n", m_name.c_str());

  dumpClut(m_pIncOut);

  blin = writeLines(ps, ips);

  bcyl = writeCyls(ps, ips);

  bsph = writeSpheres(ps, ips);

  bmes = writeMeshes(ps, ips);
  
  /*
  if (bcyl || bsph || bmes) {
    ps.format("#declare %s_tex = texture {\n", m_name.c_str());
    //ps.format("  pigment {color rgbft <0,0,0,1,1>}\n");
    ps.format("  normal {granite 0.0 scale 1.0}\n");
    ps.format("  finish {\n");
    ps.format("   ambient 0.10\n");
    ps.format("   diffuse 0.70  brilliance 1.00\n");
    ps.format("   specular 0.30  roughness 0.0050\n");
    ps.format("  }\n");
    ps.format(" }\n");
    //ps.format("#declare %s = material {texture {%s}}\n",
    //mat_name.c_str(), tex_name.c_str());
  }
     */

  if (bcyl)
    ps.format("#declare %s_lw = 1.00;\n", m_name.c_str());
  
  ips.format("\n#end\n");
  ps.format("\n");
}


