// -*-Mode: C++;-*-
//
// Molecular morphing animation object class
//

#include <common.h>

#include "MorphMol.hpp"

#include <qlib/Utils.hpp>
#include <modules/molstr/MolArrayMap.hpp>
#include <modules/molstr/MolAtom.hpp>

using namespace anim;
using molstr::MolArrayMap;
using molstr::MolCoord;
using molstr::MolCoordPtr;
using molstr::MolAtomPtr;
using molstr::SelectionPtr;

using qlib::LDom2Node;

MorphMol::MorphMol()
     : super_t()
{
  m_nAtoms = -1;
  m_dframe = 0.0;
  m_bScaleDframe = false;
}

MorphMol::~MorphMol()
{
  std::for_each(m_frames.begin(), m_frames.end(), qlib::delete_ptr<FrameData*>());
}

/*
void MorphMol::readerDetached()
{
  // setup frame crds using m_frames data
  setupData();
}
*/

/////////////////////////////////////////////////////
// Frame data implementation

bool FrameData::isDataSrcWritable() const
{
  return true;
}

LString FrameData::getDataChunkReaderName() const
{
  return LString("qdfpdb");
}

void FrameData::setDataChunkName(const LString &name, qlib::LDom2Node *pNode)
{
  LString src_type = getDataChunkReaderName();

  // set props
  m_src = name;
  m_srctype = src_type;

  // update node values
  pNode->setStrAttr("srctype", src_type);
  pNode->setStrAttr("src", name);
}

void FrameData::writeDataChunkTo(qlib::LDom2OutStream &oos) const
{
  if (m_pMol.isnull())
    return;
  m_pMol->writeDataChunkTo(oos);
}

void FrameData::readFromStream(qlib::InStream &ins)
{
  m_pMol = MolCoordPtr(MB_NEW MolCoord);
  m_pMol->setSource(m_src);
  m_pMol->setAltSource(m_altsrc);
  m_pMol->setSourceType(m_srctype);
  m_pMol->readFromStream(ins);
  m_crds.resize(0);
}

/////////////////////////////////////////////////////
// specific operations

void MorphMol::writeDataChunkTo(qlib::LDom2OutStream &oos) const
{
  int nfrms = m_frames.size();
  int nthis = -1;
  for (int i=0; i<nfrms; ++i) {
    FrameData *pFrm = m_frames[i];

    if ( pFrm->m_srctype.equals("<this>") ) {
      nthis = i;
      break;
    }
  }

  int i;
  for (i=0; i<m_nAtoms; ++i) {
    int aid = m_id2aid[i];
    MolAtomPtr pAtom = getAtom(aid);
    if (pAtom.isnull()) {
      LOG_DPRINTLN("MorphMol::update mol mismatch at ID=%d (ignored)", i);
    }
    qlib::Vector4D pos(m_frames[nthis]->m_crds.at(i*3),
                       m_frames[nthis]->m_crds.at(i*3+1),
                       m_frames[nthis]->m_crds.at(i*3+2));
    pAtom->setPos(pos);
  }
  
  super_t::writeDataChunkTo(oos);
}

void MorphMol::forceEmbed()
{
  super_t::forceEmbed();

  int nfrms = m_frames.size();
  for (int i=0; i<nfrms; ++i) {
    FrameData *pFrm = m_frames[i];

    if (pFrm->m_srctype.equals("<this>"))
      continue;

    pFrm->m_src = "datachunk:";
    pFrm->m_altsrc = "";
    pFrm->m_srctype = "";
  }  
}

void MorphMol::readFromStream(qlib::InStream &ins)
{
  super_t::readFromStream(ins);

  // setup <this>
  MolCoordPtr pthis(this);
  MolArrayMap thisset;
  thisset.setup(pthis);
  m_nAtoms = thisset.size();
  m_id2aid.resize(m_nAtoms);
  thisset.convertID(m_id2aid);

  int nfrms = m_frames.size();
  for (int i=0; i<nfrms; ++i) {
    FrameData *pFrm = m_frames[i];

    if ( !pFrm->m_srctype.equals("<this>") )
      continue;

    // copy default (<this>) coordinates to m_crds array
    pFrm->m_crds.resize(m_nAtoms*3);
    thisset.convertf( pFrm->m_crds );
  }

}

/// setup frames data
void MorphMol::setupData()
{
  // <this> should have been setup here
  MolCoordPtr pthis(this);
  MolArrayMap thisset;
  thisset.setup(pthis);

  // m_nAtoms = thisset.size();
  // m_id2aid.resize(m_nAtoms);
  // thisset.convertID(m_id2aid);

  int nfrms = m_frames.size();
  for (int i=0; i<nfrms; ++i) {
    FrameData *pFrm = m_frames[i];

    if ( pFrm->m_srctype.equals("<this>") ) {
      //pFrm->m_crds.resize(m_nAtoms*3);
      //thisset.convertf( pFrm->m_crds );
      // <this> should have been setup here
      continue;
    }
    else {
      // copy <frame> coordinates
      MolCoordPtr pmol = pFrm->m_pMol;

      MolArrayMap aset;
      aset.setup(pmol);

      if (m_nAtoms!=aset.size()) {
        LOG_DPRINTLN("MorphMol Error: atom size mismatch (%d != %d)",
                    aset.size(), m_nAtoms);

        // copy default (<this>) coordinates to m_crds array
        pFrm->m_crds.resize(m_nAtoms*3);
        thisset.convertf( pFrm->m_crds );
        continue;
      }

      // copy coordinates to m_crds array
      pFrm->m_crds.resize(m_nAtoms*3);
      aset.convertf( pFrm->m_crds );
    }
  }
}

/*
MolCoordPtr MorphMol::readNewMol(const LString &src,
                                 const LString &altsrc,
                                 const LString &srctype)
{
  MolCoordPtr pMol = MolCoordPtr(MB_NEW MolCoord);
  pMol->setSourceType(srctype);
  pMol->setSource(src);
  //pMol->readFromPath(src, altsrc, getScene());
  pMol->readFromSrcOrAltSrc2(src, altsrc, getScene());
  return pMol;
}
*/

/// Append new coordinates frame
bool MorphMol::appendMol(molstr::MolCoordPtr pmol)
{
#if 0
  if (m_frames.empty()) {
    setupData();
  }

  if (pmol->getAtomSize()!=m_nAtoms) {
    MB_THROW(qlib::IllegalArgumentException, "appendMol: atom size mismatch");
    return false;
  }

  MolArrayMap aset;
  aset.setup(pmol);
  if (aset.size()!=m_nAtoms) {
    LString msg = LString::format("appendMol: atom size mismatch (%d != %d)",
                                  aset.size(), m_nAtoms);
    MB_THROW(qlib::IllegalArgumentException, msg);
    return false;
  }
  
  PosArray *pNewAtoms = new PosArray(m_nAtoms*3);
  aset.convertf(*pNewAtoms);

  m_frames.push_back(pNewAtoms);
  m_names.push_back(pmol->getName());
  m_uids.push_back(pmol->getUID());
#endif

  return true;
}

void MorphMol::update(double dframe)
{
  int i;
  const int nframes = m_frames.size();

  // check data loaded
  for (i=0; i<nframes; ++i) {
    if (m_frames[i]->m_crds.size()>0)
      continue;
    setupData();
    break;
  }

  if (m_nAtoms<0) return;

  double xx;
  if (m_bScaleDframe) {
    // dframe changes between 0.0 and nfrm-1
    xx = qlib::trunc(dframe, 0.0, double(nframes-1));
  }
  else {
    // dframe changes between 0.0 and 1.0
    xx = qlib::trunc(dframe, 0.0, 1.0) * double(nframes-1);
  }

  int ifrm = int( ::floor(xx) );
  double rho = xx - double(ifrm);

  int ncrd = m_nAtoms*3;
  PosArray curtmp(ncrd);

  if (ifrm==m_frames.size()-1 || qlib::isNear4(rho, 0.0) ) {
    // ifrm is the last frame
    // rho is almost zero
    //   --> use ifrm (and not interpolate between ifrm~ifrm+1)
    for (i=0; i<ncrd; ++i) {
      curtmp[i] = m_frames[ifrm]->m_crds.at(i);
    }
  }
  else {
    // interpolate between ifrm~ifrm+1
    for (i=0; i<ncrd; ++i) {
      const double x0 = m_frames[ifrm]->m_crds.at(i);
      const double x1 = m_frames[ifrm+1]->m_crds.at(i);
      curtmp[i] = x0*(1.0-rho) + x1*rho;
    }
  }
  
  for (i=0; i<m_nAtoms; ++i) {
    int aid = m_id2aid[i];
    MolAtomPtr pAtom = getAtom(aid);
    if (pAtom.isnull()) {
      LOG_DPRINTLN("MorphMol::update mol mismatch at ID=%d (ignored)", i);
    }
    qlib::Vector4D pos(curtmp[i*3],
                       curtmp[i*3+1],
                       curtmp[i*3+2]);
    pAtom->setPos(pos);
  }

  // broadcast modification event
  fireAtomsMoved();
}

void MorphMol::writeTo2(LDom2Node *pNode) const
{
  super_t::writeTo2(pNode);

  LDom2Node *pFSNode = pNode->appendChild("frames");

  int nfrms = m_frames.size();
  for (int i=0; i<nfrms; ++i) {
    FrameData *pFrm = m_frames[i];
    if (pFrm->m_srctype.equals("<this>") ) {
      pFSNode->appendChild("this");
    }
    else {
      LDom2Node *pCCNode = pFSNode->appendChild("mol");

      const LString &src_str = pFrm->m_src;

      if (!src_str.startsWith("datachunk:")) {
        // External data source:
        // Convert to relative path from basedir, if possible.
        
        // AltSrc should be always absolute path and readable (normalization)
        LString alt_src_str = pFrm->m_altsrc;
        convSrcPath(src_str, alt_src_str, pCCNode, false);
      }
      else {
        //  ( embedded --> no path name conv, reader opts, and altpath aren't required. )
        pCCNode->setStrAttr("src", src_str);
        pCCNode->requestDataEmbed(pFrm);
      }

      pCCNode->setStrAttr("srctype", pFrm->m_srctype);
    }
  }
}

void MorphMol::readFrom2(LDom2Node *pNode)
{
  super_t::readFrom2(pNode);

  LDom2Node *pFSNode = pNode->findChild("frames");
  if (pFSNode==NULL)
    return;

  for (pFSNode->firstChild(); pFSNode->hasMoreChild(); pFSNode->nextChild()) {
    LDom2Node *pChNode = pFSNode->getCurChild();
    LString tag = pChNode->getTagName();

    FrameData *pFrm = NULL;

    for (;;) {
      if (tag.equals("mol")) {
        LString src = pChNode->getStrAttr("src");
        LString altsrc = pChNode->getStrAttr("alt_src");
        LString srctype = pChNode->getStrAttr("srctype");
        
        MB_DPRINTLN("MorphMol: src=%s, alt_src=%s, stctype=%s",
                    src.c_str(), altsrc.c_str(), srctype.c_str());
        // check validity of src/srctype
        if (srctype.isEmpty()) {
          // no srctype --> ignore entry
          break;
        }

        pFrm = MB_NEW FrameData;
        pFrm->m_src = src;
        pFrm->m_altsrc = altsrc;
        pFrm->m_srctype = srctype;

        // request data loading for morphframes
        pChNode->requestDataLoad(src, altsrc, srctype, pFrm);
      }
      else if (tag.equals("this")) {
        pFrm = MB_NEW FrameData;
        pFrm->m_src = "";
        pFrm->m_altsrc = "";
        pFrm->m_srctype = "<this>";
      }

      // OK
      break;
    }
    
    if (pFrm!=NULL)
      m_frames.push_back(pFrm);
  }
}

