//
// APBS electrostatic potential calculation tool
//
// $Id: apbs-calcpot.js,v 1.8 2011/04/29 17:38:47 rishitani Exp $
//

( function () { try {

  ///////////////////////////
  // Initialization
  
  const pref = require("preferences-service");
  const util = require("util");
  const jpkfile = require("file");
  const apbs_exe_key = "cuemol2.ui.apbs-exe-path";
  const pdb2pqr_py_key = "cuemol2.ui.pdb2pqr-py-path";
  //

  var dlg = window.gDlgObj = new Object();
  dlg.mTgtSceID = window.arguments[0];
  dd("APBSDlg> TargetScene="+dlg.mTgtSceID);
  dlg.mPlfName = util.getPlatformString();
  
  dlg.mObjBox = new cuemolui.ObjMenuList(
    "mol-select-box", window,
    function (elem) {
      if (elem.type=="MolCoord") return true;
      return false;
    },
    cuemol.evtMgr.SEM_OBJECT);
  dlg.mObjBox._tgtSceID = dlg.mTgtSceID;
  
  window.addEventListener("load", function(){
    try {dlg.onLoad();} catch (e) {debug.exception(e);}
  }, false);
  
  dlg.mMolSel = null;

  var default_path = "";

  // default apbs.exe path
  if (dlg.mPlfName=="Windows_NT")
    default_path = util.createDefaultPath("CurProcD", "apbs", "apbs.exe");
  else
    default_path = util.createDefaultPath("CurProcD", "apbs", "apbs");

  if (pref.has(apbs_exe_key))
    dlg.mApbsExePath = pref.get(apbs_exe_key);
  else
    dlg.mApbsExePath = default_path ;

  //
  
  // default pdb2pqr.py path
  default_path = util.createDefaultPath("CurProcD", "pdb2pqr.py");

  if (pref.has(pdb2pqr_py_key))
    dlg.mPdb2pqrPath = pref.get(pdb2pqr_py_key);
  else
    dlg.mPdb2pqrPath = default_path ;

  //

  dlg.mPqrFile = null;
  dlg.mApbsInFile = null;
  dlg.mPotFile = null;

  dlg.mPotPath = null;

  ///////////////////////////
  // Event Methods

  dlg.onLoad = function ()
  {
    var that = this;
    
    this.mApbsExePathBox = document.getElementById("apbs-exe-path");
    this.mApbsExePathBox.value = this.mApbsExePath;
    this.mSelBox = document.getElementById('mol-selection-list');
    this.mSelBox.targetSceID = this.mTgtSceID;
    this.mElepotName = document.getElementById('elepot-obj-name');
    this.mElepotName.disabled=false;

    this.mObjBox.addSelChanged(function(aEvent) {
      try { that.onObjBoxChanged(aEvent);}
      catch (e) { debug.exception(e); }
    });

    var nobjs = this.mObjBox.getItemCount();

    //alert("item count="+nobjs);
    if (nobjs==0) {
      document.getElementById("selection-check").disabled = true;
      this.mSelBox.disabled = true;
      this.mElepotName.disabled = true;
    }
    else {
      var mol = this.mObjBox.getSelectedObj();
      if (mol) {
	this.mSelBox.molID = mol.uid;
	this.mElepotName.value = this.makeSugName(mol.name);
      }
    }
    this.mSelBox.buildBox();

    this.onChgMthSel("use-internal-pqr");

    this.mPdb2pqrPathBox = document.getElementById("pdb2pqr-py-path");
    this.mPdb2pqrPathBox.value = this.mPdb2pqrPath;

  }
  
  dlg.makeSugName = function (name)
  {
    var newname = "pot_"+name;
    var scene = cuemol.getScene(this.mTgtSceID);
    if (scene==null||scene==undefined)
      return newname;

    if (scene.getObjectByName(newname)!=null) {
      newname = util.makeUniqName2(
	function (a) {return newname+"("+a+")"},
	function (a) {return scene.getObjectByName(a);} );
    }

    return newname;
  }

  dlg.onObjBoxChanged = function (aEvent)
  {
    dd("APBSDlg> ObjSelChg: "+aEvent.target.id);
    var mol = this.mObjBox.getSelectedObj();
    if (mol) {
      this.mSelBox.molID = mol.uid;
      this.mElepotName.value = this.makeSugName(mol.name);
    }      
  }


  dlg.onSelChk = function (aEvent)
  {
    if (aEvent.target.checked)
      this.mSelBox.disabled = false;
    else
      this.mSelBox.disabled = true;
  }
  
  dlg.onApbsExePath = function ()
  {
    const nsIFilePicker = Ci.nsIFilePicker;
    var fp = Cc["@mozilla.org/filepicker;1"].createInstance(Ci.nsIFilePicker);

    fp.init(window, "Select APBS executable file", nsIFilePicker.modeOpen);

    if (this.mPlfName=="Windows_NT") {
      fp.appendFilters(nsIFilePicker.filterApps);
    }
    else {
      fp.appendFilters(nsIFilePicker.filterAll);
    }

    var res = fp.show();
    if (res!=nsIFilePicker.returnOK) {
      return;
    }

    var path = fp.file.path;
    this.mApbsExePathBox.value = path;
    pref.set(apbs_exe_key, path);
  }

  dlg.onChgMthSel = function (id)
  {
    var distgts = document.getElementsByClassName("intchg-group");
    //alert("disable: "+distgts.length);
    var intChgGrp = Array.prototype.slice.call(distgts,0);
    distgts = document.getElementsByClassName("pdb2pqr-group");
    var p2pGrp = Array.prototype.slice.call(distgts, 0);

    if (id=="use-pdb2pqr") {

      intChgGrp.forEach( function (elem, ind, ary) {
	//alert("disable: "+elem.localName);
	elem.setAttribute("disabled", "true");
      });
      p2pGrp.forEach( function (elem, ind, ary) {
	elem.removeAttribute("disabled");
      });

    }
    else {

      intChgGrp.forEach( function (elem, ind, ary) {
	elem.removeAttribute("disabled");
      });
      p2pGrp.forEach( function (elem, ind, ary) {
	elem.setAttribute("disabled", "true");
      });

    }
  };

  dlg.onPdb2PqrPath = function ()
  {
    const nsIFilePicker = Ci.nsIFilePicker;
    var fp = Cc["@mozilla.org/filepicker;1"].createInstance(Ci.nsIFilePicker);

    fp.init(window, "Select pdb2pqr.py file", nsIFilePicker.modeOpen);

    fp.appendFilters(nsIFilePicker.filterAll);

    var res = fp.show();
    if (res!=nsIFilePicker.returnOK)
      return;

    var path = fp.file.path;
    this.mPdb2pqrPathBox.value = path;
    pref.set(pdb2pqr_py_key, path);
  };

  ///////////////////////////////////////////

  dlg.onDialogAccept = function (event)
  {
    this.mTgtMol = this.mObjBox.getSelectedObj();
    if (this.mTgtMol==null)
      return;

    try {
      this.makePqrFile();
      if (!this.calcGridDim())
	return;
      this.makeAPBSIn();
      this.runAPBS();
      this.loadPotDxFile();
    }
    catch (e) {
      debug.exception(e);
    }
    finally {
      if (this.mPqrFile) {
	try { this.mPqrFile.remove(false); } catch (e) {}
	this.mPqrFile = null;
      }
      if (this.mApbsInFile) {
	try { this.mApbsInFile.remove(false); } catch (e) {}
	this.mApbsInFile = null;
      }
      if (this.mPotFile) {
	try { this.mPotFile.remove(false); } catch (e) {}
	this.mPotFile = null;
      }
    }

  }

  ////////////////

  dlg.makePqrFile = function ()
  {
    var tgtmol = this.mTgtMol;

    // setup seleciton
    var molsel = null;
    if (!this.mSelBox.disabled)
      molsel = this.mSelBox.selectedSel;
    this.mMolSel = molsel;

    // select charge method
    if (document.getElementById("use-pdb2pqr").selected)
      this.runPdb2Pqr();
    else
      this.makeIntPqrFile();
  }
  
  dlg.makeIntPqrFile = function ()
  {
    var tgtmol = this.mTgtMol;
    var molsel = this.mMolSel;

    // create exporter obj (pqr)
    var strMgr = cuemol.getService("StreamManager");
    var exporter = strMgr.createHandler("pqr", 1);
    if (typeof exporter==="undefined" || exporter===null) {
      util.alert(window, "Save: get exporter Failed.");
      throw "cannot create exporter for pqr";
    }

    // check hydrogen flag
    var bUseH = document.getElementById("use-hydrogen").checked;

    // make pqr tmp file
    var file = util.createMozTmpFile("apbs_tmp.pqr");

    dd("tmp pqr file: "+file.path);

    this.mPqrFile = file;
    var pqrFileName = file.path;

    // write pqr files
    try {
      if (molsel && molsel.toString()!=="")
	exporter.sel = molsel;
      dd("write: " + pqrFileName);
      exporter.use_H = bUseH;
      //exporter.ns = "amber";
      exporter.setPath(pqrFileName);
      exporter.attach(tgtmol);
      exporter.write();
      exporter.detach();
    }
    catch (e) {
      debug.exception(e);
      throw e;
    }
    finally {
      // exporter is expected to be removed by GC...
      delete exporter;
    }
  }

  dlg.runPdb2Pqr = function ()
  {
    var tgtmol = this.mTgtMol;
    var molsel = this.mMolSel;

    // force field name
    var ffname = "charmm";
    var elem = document.getElementById("pdb2pqr-ff-list");
    if (elem.selectedItem.value)
      ffname = elem.selectedItem.value;

    // create exporter obj (pdb)
    var strMgr = cuemol.getService("StreamManager");
    var exporter = strMgr.createHandler("pdb", 1);
    if (typeof exporter==="undefined" || exporter===null) {
      util.alert(window, "Save: get exporter Failed.");
      throw "cannot create exporter for pdb";
    }

    // make pqr tmp file
    var file = util.createMozTmpFile("apbs_tmp.pqr");
    dd("tmp pqr file: "+file.path);
    this.mPqrFile = file;
    var pqrFileName = file.path;

    // write PDB files
    var pdb_tmpfile = null;
    try {
      // make pqr tmp file
      pdb_tmpfile = util.createMozTmpFile("apbs_tmp.pdb");
      dd("tmp pdb file: "+pdb_tmpfile.path);
      exporter.setPath(pdb_tmpfile.path);
      exporter.attach(tgtmol);
      if (molsel && molsel.toString()!=="")
	exporter.sel = molsel;
      exporter.write();
      exporter.detach();

      // run PDB2PQR
      var args = ["-v","--chain", "--nodebump", "--noopt", "--ff", ffname, pdb_tmpfile.path, pqrFileName];
      dd("APBSDlg> pdb2pqr.py args= "+args.join(","));
      util.run_proc(this.mPdb2pqrPathBox.value, pdb2pqr_py_key, args);
    }
    catch (e) {
      debug.exception(e);
      throw e;
    }
    finally {
      // exporter is expected to be removed by GC...
      delete exporter;

      // remove pdb file
      if (pdb_tmpfile) {
	try { pdb_tmpfile.remove(false); } catch (e) {}
	delete pdb_tmpfile;
      }
    }
    
  }

  dlg.calcGridDim = function ()
  {
    var tgtmol = this.mTgtMol;
    var oldsel = tgtmol.sel;
    var vmin,vmax;
    if (this.mMolSel) {
      tgtmol.sel = this.mMolSel;
      vmin = tgtmol.getBoundBoxMin(true);
      vmax = tgtmol.getBoundBoxMax(true);
      tgtmol.sel = oldsel;
    }
    else {
      vmin = tgtmol.getBoundBoxMin(false);
      vmax = tgtmol.getBoundBoxMax(false);
    }
    var vdim = vmax.sub(vmin);
    var vcen = vmax.add(vmin).divide(2.0);
    dd("APBSDlg> dim = "+vdim);
    dd("APBSDlg> cen = "+vcen);
    if (vdim.isZero()) {
      dd("APBSDlg> invalid dim");
      return false;
    }

    const cfac = 1.7;
    var cdim = vdim.scale(cfac);
    dd("APBSDlg> cdim = "+cdim);

    const fadd = 20.0;
    var fdim = cuemol.createObj("Vector");
    fdim.x = Math.min(cdim.x, vdim.x+fadd);
    fdim.y = Math.min(cdim.y, vdim.y+fadd);
    fdim.z = Math.min(cdim.z, vdim.z+fadd);
    dd("APBSDlg> fdim = "+fdim);

    //const nlev = 4;
    const mult_fac = 32; //Math.pow(2, nlev+1);
    
    ////
    // density value

    var nden = parseFloat(document.getElementById("grid-size").value);
    if (nden==NaN || nden<0.0)
      nden = 1.0;

    var dpts = fdim.divide(nden);

    var cs = cuemol.createObj("Vector");
    cs.x = Math.ceil(dpts.x/mult_fac);
    cs.y = Math.ceil(dpts.y/mult_fac);
    cs.z = Math.ceil(dpts.z/mult_fac);
    dd("APBSDlg> cs = "+cs);

    var fpts = cuemol.createObj("Vector");
    fpts.x = mult_fac*cs.x + 1.0;
    fpts.y = mult_fac*cs.y + 1.0;
    fpts.z = mult_fac*cs.z + 1.0;
    dd("APBSDlg> final grid points = "+fpts);

    this.mGridCse = cdim;
    this.mGridFin = fdim;
    this.mGridPts = fpts;

    return true;
  }

  /// Make apbs.in input file
  dlg.makeAPBSIn = function ()
  {
    // make output pot filename
    var cpos = util.splitFileName(this.mPqrFile.path, "*.pqr");

    var potdx_out = this.mPqrFile.path.substr(0, cpos);
    this.mPotPath = potdx_out+".dx";

    dd("tmp pot dx file: "+this.mPotPath);

    var dat = "";
    dat += "read\n";
    dat += "  mol pqr "+this.mPqrFile.path+"\n";
    dat += "end\n";
    dat += "elec\n";
    dat += "  mg-auto\n";
    dat += "  dime "+this.mGridPts.x.toFixed()+
      " "+this.mGridPts.y.toFixed()+
	" "+this.mGridPts.z.toFixed()+" "+"\n";

    dat += "  cglen "+this.mGridCse.x+
      " "+this.mGridCse.y+
	" "+this.mGridCse.z+" "+"\n";

    dat += "  fglen "+this.mGridFin.x+
      " "+this.mGridFin.y+
	" "+this.mGridFin.z+" "+"\n";

    dat += "  mol 1\n";
    dat += "  cgcent mol 1\n";
    dat += "  fgcent mol 1\n";

    var bUseNPBE = document.getElementById("use-npbe").checked;
    if (bUseNPBE)
      dat += "  npbe\n";
    else
      dat += "  lpbe\n";

    dat += "  bcfl sdh\n";
    //dat += "  ion 1 0.1 2.0\n";
    //dat += "  ion -1 0.1 2.0\n";
    var pdie = parseFloat(document.getElementById("prot-dielec").value);
    if (pdie==NaN || pdie<=0.0)
      pdie = 2.0;
    dat += "  pdie "+pdie+"\n";

    var sdie = parseFloat(document.getElementById("water-dielec").value);
    if (sdie==NaN || sdie<=0.0)
      sdie = 78.54;
    dat += "  sdie "+sdie+"\n";
    
    var temp = parseFloat(document.getElementById("calc-temp").value);
    if (temp==NaN || temp<=0.0)
      temp = 298.15;
    dat += "  temp "+temp+"\n";

    dat += "  chgm spl2\n";
    dat += "  srad 1.4\n";
    dat += "  swin 0.3\n";
    dat += "  sdens 10.0\n";
    dat += "  srfm smol\n";
    dat += "  calcenergy no\n";
    dat += "  calcforce no\n";
    dat += "  write pot dx "+potdx_out+"\n";
    dat += "end\n";
    dat += "quit\n";

    // make apbs input file
    var file = util.createMozTmpFile("apbs_tmp.in");
    var fout = jpkfile.open(file.path, "w");
    fout.write(dat);
    fout.close();
    delete fout;
    this.mApbsInFile = file;
  }

  dlg.runAPBS = function ()
  {
    var str_apbsexe = this.mApbsExePathBox.value;

    var apbsexe;
    try {
      apbsexe = util.createMozFile(str_apbsexe);
      if (!apbsexe.isFile()) {
	throw "cannot open apbs exe file: "+str_apbsexe;
      }    
      pref.set(apbs_exe_key, apbsexe.path);
    }
    catch (e) {
      util.alert(window, "Cannot open APBS executable file: "+str_apbsexe);
      throw e;
      return;
    }
    
    var proc = this.mProc = Cc["@mozilla.org/process/util;1"].createInstance(Ci.nsIProcess);
    proc.init(apbsexe);
    
    ////

    var args = [this.mApbsInFile.path];
    
    dd("APBSDlg> args= "+args.join(","));
    /*
    var that = this;
    this.mTimer = require("timer").setInterval(function() {
      try { that.onTimer(); }
      catch (e) {
	dd("Error: "+e);
	debug.exception(e);
      }
    }, 1000);
     */
    
    // block running of APBS process
    proc.run(true, args, args.length);

    try {
      this.mPotFile = util.createMozFile(this.mPotPath);
      if (!this.mPotFile.isFile()) {
	throw "cannot open apbs output dx file: "+this.mPotPath;
      }    
    }
    catch (e) {
      util.alert(window, "Run APBS is failed (no output file)");
      throw e;
      return;
    }
  }

  dlg.loadPotDxFile = function ()
  {
    var scene = cuemol.getScene(this.mTgtSceID);

    var strMgr = cuemol.getService("StreamManager");
    var reader = strMgr.createHandler("apbs", 0);
    reader.setPath(this.mPotFile.path);

    // make default names

    var tgtmol = this.mTgtMol;
    var newname = this.mElepotName.value; //"pot_"+tgtmol.name;
    /*if (scene.getObjectByName(newname)!=null) {
      newname = util.makeUniqName2(
	function (a) {return newname+"("+a+")"},
	function (a) {return scene.getObjectByName(a);} );
    }*/

    // EDIT TXN START //
    scene.startUndoTxn("Open APBS pot file");

    try {
      var newobj = reader.createDefaultObj();
      reader.attach(newobj);
      reader.read();
      reader.detach();

      newobj.name = newname;
      scene.addObject(newobj);
      newobj.forceEmbed();

      // create default renderer
      var rend = newobj.createRenderer("*unitcell");
      rend.name = "unitcell";
    }
    catch (e) {
      dd("File Open Error: "+e);
      debug.exception(e);
      
      util.alert(window, "Failed to open APBS pot file: "+path);
      reader = null;
      scene.rollbackUndoTxn();
      return;
    }

    reader = null;
    
    scene.commitUndoTxn();
    // EDIT TXN END //
  };

} catch (e) {debug.exception(e);} } )();

