//
// CueMol main service for Mozilla/XPCOM interface
//
// $Id: XPCCueMol.cpp,v 1.38 2011/03/10 13:11:55 rishitani Exp $
//

#include <common.h>

#include <qlib/ClassRegistry.hpp>
#include <qlib/EventManager.hpp>
#include <qsys/qsys.hpp>
#include <qsys/SceneManager.hpp>
#include <jsbr/jsbr.hpp>

#include "XPCCueMol.hpp"
#include "XPCObjWrapper.hpp"

#include "xpcom.hpp"
#include <nsIObserverService.h>

#ifdef XP_WIN
#  undef NEW_H
#  define NEW_H "new.h"
#  include "XPCNativeWidgetWin.hpp"
#endif

// #define _num_to_str(num) #num
// #define num_to_str(num) _num_to_str(num)
// #pragma message ("NEW_H: " num_to_str(new))

#ifdef XP_MACOSX
#  include "XPCNativeWidgetCocoa.hpp"
#endif

namespace molstr {
  extern bool init();
  extern void fini();
}

namespace molvis {
  extern bool init();
  extern void fini();
}

namespace xtal {
  extern bool init();
  extern void fini();
}

namespace molanl {
  extern bool init();
  extern void fini();
}

namespace surface {
  extern bool init();
  extern void fini();
}

namespace symm {
  extern bool init();
  extern void fini();
}

using namespace xpcom;

NS_IMPL_ISUPPORTS3(XPCCueMol, qICueMol, nsITimerCallback, nsIObserver)

XPCCueMol::XPCCueMol()
  : m_bInit(false)
{
}

XPCCueMol::~XPCCueMol()
{
  if (m_bInit)
    Fini();
}

//////////
// Idle timer implementation

NS_IMETHODIMP
XPCCueMol::Notify(nsITimer *timer)
{
  //MB_DPRINTLN("Timer: notified");

  qsys::SceneManager *pSM = qsys::SceneManager::getInstance();
  pSM->checkAndUpdateScenes();

  qlib::EventManager *pEM = qlib::EventManager::getInstance();
  pEM->messageLoop();

  return NS_OK;
}

// Quit-app observer

NS_IMETHODIMP
XPCCueMol::Observe(nsISupports* aSubject, const char* aTopic,
                   const PRUnichar* aData)
{
  //Fini();
  return NS_OK;
}

//////////

NS_IMETHODIMP XPCCueMol::Init(const char *confpath, PRBool *_retval)
{
  // XXX
  //AddRef();
  
  nsresult rv = NS_OK;
  
  if (m_bInit) {
    LOG_DPRINTLN("XPCCueMol> ERROR: CueMol already initialized.");
    return NS_ERROR_ALREADY_INITIALIZED;
  }

  // CueMol2 Application initialization
  qsys::init(confpath);
  MB_DPRINTLN("---------- qsys::init(confpath) OK");

  // load internal JS module
  jsbr::init();
  MB_DPRINTLN("---------- jsbr::init() OK");

  // load other modules
  molstr::init();
  molvis::init();
  xtal::init();
  symm::init();
  surface::init();
  molanl::init();

  initTextRender();

  // setup timer
  m_timer = do_CreateInstance("@mozilla.org/timer;1");
  if (m_timer) {
    rv = m_timer->InitWithCallback(this, 30, nsITimer::TYPE_REPEATING_SLACK);
    if (NS_FAILED(rv)) {
      m_timer = nsnull;
    }
  }

  // setup quit-app observer
  nsCOMPtr<nsIObserverService> obs = do_GetService("@mozilla.org/observer-service;1", &rv);
  NS_ENSURE_SUCCESS(rv, rv);

  rv = obs->AddObserver(this, "xpcom-shutdown", PR_FALSE);
  // rv = obs->AddObserver(this, "quit-application", PR_FALSE);
  NS_ENSURE_SUCCESS(rv, rv);

  MB_DPRINTLN("XPCCueMol> CueMol initialized.");
  m_bInit = true;
  *_retval = PR_TRUE;

  return NS_OK;
}

NS_IMETHODIMP XPCCueMol::Fini()
{
  int i;

  finiTextRender();

  MB_DPRINTLN("=== Cleaning up the unreleased wrappers... ===");
  for (i=0; i<m_pool.size(); ++i) {
    if (m_pool[i].bUsed) {
      XPCObjWrapper *pwr = m_pool[i].ptr;
      MB_DPRINTLN("Unreleased wrapper: %d %p", i, pwr);
      pwr->detach();
    }
  }
  MB_DPRINTLN("=== Done ===");

  if (m_timer) {
    m_timer->Cancel();
    m_timer = nsnull;
  }

  if (!m_bInit) {
    LOG_DPRINTLN("XPCCueMol> ERROR: CueMol not initialized.");
    return NS_ERROR_NOT_INITIALIZED;
  }

  molanl::fini();
  surface::fini();
  symm::fini();
  xtal::fini();
  molvis::fini();
  molstr::fini();

  jsbr::fini();

  // CueMol-App finalization
  qsys::fini();

  MB_DPRINTLN("XPCCueMol> CueMol finalized.");
  m_bInit = false;
  return NS_OK;
}

/* boolean isInitialized (); */
NS_IMETHODIMP XPCCueMol::IsInitialized(PRBool *_retval)
{
  *_retval = m_bInit;
  return NS_OK;
  //return NS_ERROR_NOT_IMPLEMENTED;
}


using qlib::ClassRegistry;

NS_IMETHODIMP XPCCueMol::GetService(const char *svcname, qIObjWrapper **_retval)
{
  ClassRegistry *pMgr = ClassRegistry::getInstance();
  if (pMgr==NULL) {
    LOG_DPRINTLN("XPCCueMol> ERROR: CueMol not initialized.");
    return NS_ERROR_NOT_INITIALIZED;
  }

  qlib::LDynamic *pobj;
  try {
    pobj = pMgr->getSingletonObj(svcname);
  }
  catch (const qlib::LException &e) {
    LOG_DPRINTLN("Caught exception <%s>", typeid(e).name());
    LOG_DPRINTLN("Reason: %s", e.getMsg().c_str());
    return NS_ERROR_NOT_IMPLEMENTED;
  }
  catch (...) {
    LOG_DPRINTLN("Caught unknown exception");
    return NS_ERROR_NOT_IMPLEMENTED;
  }

  qlib::LScriptable *pscr = dynamic_cast<qlib::LScriptable *>(pobj);
  if (pscr==NULL) {
    LOG_DPRINTLN("Fatal error dyncast failed!!");
    return NS_ERROR_NOT_IMPLEMENTED;
  }

  XPCObjWrapper *pWrap = createWrapper();
  pWrap->setWrappedObj(pscr);

  *_retval = pWrap;
  NS_ADDREF((*_retval));
  MB_DPRINTLN("getService(%s) OK: %p", svcname, pscr);
  return NS_OK;
}

NS_IMETHODIMP XPCCueMol::CreateObj(const char *clsname, qIObjWrapper **_retval)
{
  ClassRegistry *pMgr = ClassRegistry::getInstance();
  if (pMgr==NULL) {
    LOG_DPRINTLN("XPCCueMol> ERROR: CueMol not initialized.");
    return NS_ERROR_NOT_INITIALIZED;
  }

  qlib::LDynamic *pobj;
  try {
    qlib::LClass *pcls = pMgr->getClassObj(clsname);
    if (pcls==NULL)
      MB_THROW(qlib::NullPointerException, "null");
    pobj = pcls->createScrObj();
  }
  catch (const qlib::LException &e) {
    LOG_DPRINTLN("XPCCueMol> Caught exception <%s>", typeid(e).name());
    LOG_DPRINTLN("Reason: %s", e.getMsg().c_str());
    return NS_ERROR_NOT_IMPLEMENTED;
  }
  catch (...) {
    LOG_DPRINTLN("XPCCueMol> Caught unknown exception");
    return NS_ERROR_NOT_IMPLEMENTED;
  }

  qlib::LScriptable *pscr = dynamic_cast<qlib::LScriptable *>(pobj);
  if (pscr==NULL) {
    LOG_DPRINTLN("XPCCueMol> Fatal error dyncast failed!!");
    return NS_ERROR_NOT_IMPLEMENTED;
  }

  XPCObjWrapper *pWrap = createWrapper();
  pWrap->setWrappedObj(pscr);

  *_retval = pWrap;
  NS_ADDREF((*_retval));
  MB_DPRINTLN("XPCCueMol> createObj(%s) OK: %p", clsname, pscr);
  return NS_OK;
}

/* qIObjWrapper getObjById (in PRInt32 uid); */
NS_IMETHODIMP XPCCueMol::GetObjById(PRInt32 uid, qIObjWrapper **_retval)
{
    return NS_ERROR_NOT_IMPLEMENTED;
}

XPCObjWrapper *XPCCueMol::createWrapper()
{
  int nind;
  if (m_freeind.size()>0) {
    // reuse allocated entry
    nind = m_freeind.back();
    m_freeind.pop_back();
    // MB_ASSERT(m_pool[nind].bUsed==false);
    // return m_pool[nind].ptr;
  }
  else {
    // append to the last entry
    nind = m_pool.size();
    m_pool.push_back(Cell());
  }

  MB_ASSERT(m_pool[nind].ptr==NULL);
  MB_ASSERT(m_pool[nind].bUsed==false);

  XPCObjWrapper *pWr = new XPCObjWrapper(this, nind);
  m_pool[nind].ptr = pWr;
  m_pool[nind].bUsed = true;

  return pWr;
}

void XPCCueMol::notifyDestr(int nind)
{
  m_pool[nind].bUsed = false;
  m_pool[nind].ptr = NULL;
  m_freeind.push_back(nind);
}

/* void getErrMsg (out string confpath); */
NS_IMETHODIMP XPCCueMol::GetErrMsg(char **_retval NS_OUTPARAM)
{
  nsCAutoString nsstr(m_errMsg.c_str());
  *_retval = ToNewCString(nsstr);

  return NS_OK;
}

///////////////////////

/* qINativeWidget createNativeWidget (); */
NS_IMETHODIMP XPCCueMol::CreateNativeWidget(qINativeWidget **_retval NS_OUTPARAM)
{
  XPCNativeWidget *pWgt = NULL;

#ifdef XP_MACOSX
  pWgt = new XPCNativeWidgetCocoa();
#endif

#ifdef XP_WIN
  pWgt = new XPCNativeWidgetWin();
#endif

  if (pWgt==NULL) {
    LOG_DPRINTLN("XPCCueMol> FATAL ERROR: cannot create native widget");
    return NS_ERROR_NOT_IMPLEMENTED;
  }

  *_retval = pWgt;
  NS_ADDREF((*_retval));
  MB_DPRINTLN("XPCCueMol> createNativeWidget OK: %p", pWgt);
  return NS_OK;
}

///////////////////////

NS_IMETHODIMP XPCCueMol::Test(nsISupports *arg)
{
  *((int *) 0) = 100;
  return NS_OK;
}

