// -*-Mode: C++;-*-
//
//  WGL dependent molecular viewer implementation
//
//  $Id: WglView.cpp,v 1.29 2011/02/20 09:34:26 rishitani Exp $

#include <common.h>

#include <GL/gl.h>
#include <GL/glu.h>
#include <qlib/Utils.hpp>

#include "WglView.hpp"
#include "WglDisplayContext.hpp"

// #include "UpdateEvent.hpp"

// #define HITBUF_SIZE (64*1024)
using qsys::InDevEvent;

using namespace sysdep;

WglView::WglView()
{
  m_bCursorIn = false;
  // m_pHitBuf = new GLuint[HITBUF_SIZE];
  m_bInitOK = false;

  m_pCtxt = NULL;

  m_hDC = NULL;
  m_hGL = NULL;
  m_hWnd = NULL;

  m_nDragStart = DRAG_NONE;

  m_bHasQuadBuffer = false;
/*
  m_hStdCursor = ::LoadCursor(NULL, IDC_ARROW);
  m_hWaitCursor = ::LoadCursor(NULL, IDC_WAIT);
  m_hHandCursor = ::LoadCursor(NULL, IDC_SIZEALL);
  m_hCrossCursor = ::LoadCursor(NULL, IDC_CROSS);
 */
}

WglView::~WglView()
{
  MB_DPRINTLN("WglView (ctxt=%p) destructing.", m_pCtxt);
}

LString WglView::toString() const
{
  return LString::format("WGL/OpenGL View(%p)", this);
}

void WglView::swapBuffers()
{
  if (m_hDC!=NULL)
    ::SwapBuffers(m_hDC);
}

DisplayContext *WglView::getDisplayContext()
{
  return m_pCtxt;
}

////////////////////////////////////////////

bool WglView::attach(HWND hWnd, HDC hDC)
{
  MB_ASSERT(hDC != NULL);
  MB_ASSERT(hWnd != NULL);

  // Save the old WND/DC
  HWND hOldWnd = NULL;
  HDC hOldDC = NULL;
  HGLRC hOldGL = NULL;
  if (m_hWnd!=NULL) {
    hOldWnd = m_hWnd;
    hOldDC = m_hDC;
    hOldGL = m_hGL;
  }

  m_hWnd = hWnd;
  m_hDC = hDC;

  MB_DPRINTLN("HWND==%p", m_hWnd);
  MB_DPRINTLN("HDC==%p", m_hDC);

  m_hGL = setupWglContext();

  if (hOldGL==NULL)
    setupShareList();
  else {
    ::wglShareLists(hOldGL, m_hGL);
    ::wglMakeCurrent( NULL, NULL );
    ::wglDeleteContext(hOldGL);
  }

  // create display context object for OpenGL
  if (m_pCtxt==NULL)
    m_pCtxt = new WglDisplayContext(getSceneID(), this);

  if (!m_pCtxt->attach(m_hDC, m_hGL)) {
    // NOTE: This cannot be happen!!
    //LOG_DPRINTLN("Fatal error Cannot create WglDisplayContext!!");
    //delete pCtxt;
    return false;
  }

  m_pCtxt->setCurrent();

  // perform OpenGL-common initialization tasks
  OglView::setup();

  m_bInitOK = true;
  MB_DPRINTLN("WglView::setup() OK.");

  return true;
}

void WglView::unloading()
{
  /*if (m_hDC!=NULL && m_hWnd!=NULL) {
    // HDC will be destroyed at delete m_pDspCtxt!!
    ::ReleaseDC(m_hWnd, m_hDC);
    m_hDC = NULL;
  }*/
  
  if (m_pCtxt!=NULL) {
    delete m_pCtxt;
    m_pCtxt = NULL;
  }

  ::wglMakeCurrent( NULL, NULL );
  ::wglDeleteContext( m_hGL );
}

////////////////////////////////////////////

/// Setup OpenGL (stage 1)
HGLRC WglView::setupWglContext()
{
  setupPixelFormat();
  
  // create and enable the render context (RC)
  return ::wglCreateContext( m_hDC );
}

/// Setup OpenGL (stage 2)
bool WglView::setupShareList()
{
  DisplayContext *pShare = getSiblingCtxt();
  WglDisplayContext *pwshcx = dynamic_cast<WglDisplayContext *>(pShare);
  if (pwshcx==NULL) {
    MB_DPRINTLN("WGL> No sibling context.");
    return true;
  }
  HGLRC shcx = pwshcx->getHGLRC();
  ::wglShareLists(shcx, m_hGL);
  return true;
}

bool WglView::setupPixelFormat()
{
  int ipx;
  /*
  int ipx = ::GetPixelFormat(m_hDC);
  if (ipx>0) {
    ::DescribePixelFormat(m_hDC, ipx,
                          sizeof(PIXELFORMATDESCRIPTOR), &m_pfd);
    return true;
  }
   */

  ::memset(&m_pfd, 0, sizeof(PIXELFORMATDESCRIPTOR));
  m_pfd.nSize = (sizeof(PIXELFORMATDESCRIPTOR));
  m_pfd.nVersion = 1;

  m_pfd.dwFlags =
    PFD_DRAW_TO_WINDOW | // support window
      PFD_SUPPORT_OPENGL |          // support OpenGL
        PFD_DOUBLEBUFFER;           // double buffered
  m_pfd.iPixelType = PFD_TYPE_RGBA; // RGBA type
  m_pfd.cColorBits = 24; // 24-bit color depth
  m_pfd.cDepthBits = 32;
  m_pfd.iLayerType = PFD_MAIN_PLANE; // main layer
  
  m_pfd.cRedBits = 8;
  m_pfd.cGreenBits = 8;
  m_pfd.cBlueBits = 8;

  // valid color bit size
  int bufsz[] = {32, 24, 16, 0};
  int i;

  // Check Quad-buffered stereo capability
  m_bHasQuadBuffer = false;
  for (i=0; bufsz[i]>0; ++i) {
    ipx = choosePixFmt(bufsz[i], true);
    if (ipx>0) {
      LOG_DPRINTLN("WglView.PixFmt> cbits=%d (hardware stereo) is accepted.", bufsz[i]);
      m_bHasQuadBuffer = true;
      break;
    }
  }

  if (m_bHasQuadBuffer) {
    LOG_DPRINTLN("WglView.PixFmt> Quadbuffer stereo capable videoboard is detected.");
    if (getStereoMode()==qsys::Camera::CSM_HW_QBUF) {
      setPixFmt(ipx);
      return true; // ==> Use the found quadbuffer stereo pixel format
    }
  }
  else {
    MB_DPRINTLN("Cannot use quad-buffered stereo in this environment.");
  }
  
  // Check non-stereo OpenGL pixel format
  for (i=0; bufsz[i]>0; ++i) {
    ipx = choosePixFmt(bufsz[i], false);
    if (ipx>0) {
      LOG_DPRINTLN("WglView.PixFmt> cbits=%d (no stereo) is accepted.", bufsz[i]);
      setPixFmt(ipx);
      return true;
    }
  }

  LOG_DPRINTLN("WglView.PixFmt> FATAL ERROR, No suitable OpenGL pixel format was found!!");
  return false;
}

int WglView::choosePixFmt(int nColorBits, bool bStereo)
{
  int pixelformat;
  m_pfd.cColorBits = nColorBits;

  if (bStereo)
    m_pfd.dwFlags |= PFD_STEREO;
  else
    m_pfd.dwFlags &= ~PFD_STEREO;

  if ( (pixelformat = ::ChoosePixelFormat(m_hDC, &m_pfd)) == 0 ) {
    MB_DPRINTLN("ChoosePixFmt(cbit:%d, stereo:%d) failed", nColorBits, bStereo);
    return 0;
  }

  ::DescribePixelFormat(m_hDC, pixelformat,
                        sizeof(PIXELFORMATDESCRIPTOR), &m_pfd);
    
  // check the selected pixel format
  if (bStereo)
    if (!(m_pfd.dwFlags & PFD_STEREO))
      return 0;

  if (nColorBits>m_pfd.cColorBits)
    return 0;

  return pixelformat;
}

bool WglView::setPixFmt(int ipx)
{
  if (!::SetPixelFormat(m_hDC, ipx, &m_pfd))
    return false;

  return true;
}

/////////////////////////////////////////////////////////////////////////////

void WglView::setUpMouseEvent(UINT nFlags, POINTS point, InDevEvent &ev)
{
  int modif = 0;

  if (nFlags & MK_CONTROL)
    modif |= InDevEvent::INDEV_CTRL;
  if (nFlags & MK_SHIFT)
    modif |= InDevEvent::INDEV_SHIFT;
  if (nFlags & MK_LBUTTON)
    modif |= InDevEvent::INDEV_LBTN;
  if (nFlags & MK_MBUTTON)
    modif |= InDevEvent::INDEV_MBTN;
  if (nFlags & MK_RBUTTON)
    modif |= InDevEvent::INDEV_RBTN;

  ev.setSource(this);
  ev.setModifier(modif);
  ev.setX(point.x);
  ev.setY(point.y);

  POINT ptroot = {point.x, point.y};
  ::ClientToScreen(m_hWnd, &ptroot);

  ev.setRootX(ptroot.x);
  ev.setRootY(ptroot.y);
}

LRESULT WglView::handleEvent(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
  switch (msg) {
  case WM_PAINT: {
    // draw a frame and display the string
    PAINTSTRUCT ps;
    RECT rc;

    HDC hdc = BeginPaint(hWnd, &ps);
    GetClientRect(hWnd, &rc);
    forceRedraw();
    EndPaint(hWnd, &ps);
    return 0;
    //break;
  }

  case WM_SIZE: {
    int cx = LOWORD(lParam);
    int cy = HIWORD(lParam);
    sizeChanged(cx, cy);
    return 0;
    //break;
  }

  case WM_LBUTTONDOWN: 
  case WM_RBUTTONDOWN: {
    POINTS pt = MAKEPOINTS(lParam);
    m_startPt = m_prevPt = pt;
    //m_fDragStart = false;
    m_nDragStart = DRAG_CHECK;

    UINT nFlags = wParam;
    /*
    if (nFlags & MK_LBUTTON)
      MB_DPRINT("[L]");
    if (nFlags & MK_MBUTTON)
      MB_DPRINT("[M]");
    if (nFlags & MK_RBUTTON)
      MB_DPRINT("[R]");
    MB_DPRINTLN("");
     */

    {
      InDevEvent ev;
      setUpMouseEvent(nFlags, pt, ev);
      ev.setType(InDevEvent::INDEV_MOUSE_DOWN);
      fireInDevEvent(ev);
    }

    return 0;
    //break;
  }

  case WM_LBUTTONUP:
  case WM_RBUTTONUP: {
    POINTS point = MAKEPOINTS(lParam);
    InDevEvent ev;
    UINT nFlags = wParam;

    nFlags &= ~(MK_LBUTTON|MK_RBUTTON|MK_MBUTTON);
    if (msg==WM_LBUTTONUP)
      nFlags |= MK_LBUTTON;
    else
      nFlags |= MK_RBUTTON;

    setUpMouseEvent(nFlags, point, ev);
    
    if (m_nDragStart==DRAG_DRAG) {
      // mouse drag is initiated
      // case of the end of drag
      ev.setType(InDevEvent::INDEV_DRAG_END);
      fireInDevEvent(ev);
      m_nDragStart = DRAG_NONE;
      return 0;
      //break;
    }
    if (m_nDragStart==DRAG_CHECK) {
      // click event case
      if (msg==WM_LBUTTONUP)
        ev.setType(InDevEvent::INDEV_LBTN_CLICK);
      else
        ev.setType(InDevEvent::INDEV_RBTN_CLICK);
      fireInDevEvent(ev);
      //LOG_DPRINTLN("clicked!!");
      m_nDragStart = DRAG_NONE;
      return 0;
    }

    // ???
    m_nDragStart = DRAG_NONE;
    break;
  }

  case WM_MOUSELEAVE: {
    // MB_DPRINTLN("****** WM_MOUSELEAVE");
    m_bCursorIn = false;
    TRACKMOUSEEVENT tme;
    tme.cbSize = sizeof tme;
    tme.dwFlags = TME_CANCEL|TME_LEAVE;
    tme.hwndTrack = m_hWnd;
    ::TrackMouseEvent(&tme);

    InDevEvent ev;
    ev.setType(InDevEvent::INDEV_MOUSE_ENTER);
    ev.setModifier(1); // leave flag
    ev.setSource(this);
    fireInDevEvent(ev);
    break;
  }

  case WM_MOUSEMOVE: {
    //::SetCursor(m_hWndCursor);
    if (!m_bCursorIn) {
      // MB_DPRINTLN("****** Mouse Enter ??? ");
      TRACKMOUSEEVENT tme;
      tme.cbSize = sizeof tme;
      tme.dwFlags = TME_LEAVE;
      tme.hwndTrack = m_hWnd;
      ::TrackMouseEvent(&tme);
      m_bCursorIn = true;

      InDevEvent ev;
      ev.setType(InDevEvent::INDEV_MOUSE_ENTER);
      ev.setModifier(0);
      ev.setSource(this);
      fireInDevEvent(ev);
    }

    //MB_DPRINTLN("SetCursor!!");
    if (m_nDragStart == DRAG_NONE) return 0;

    POINTS point = MAKEPOINTS(lParam);
    InDevEvent ev;

    if (!(wParam & MK_LBUTTON) &&
        !(wParam & MK_MBUTTON) &&
        !(wParam & MK_RBUTTON)) {
      return 0;
      //break;
    }

    // check drag start
    if (m_nDragStart==DRAG_CHECK) {
      // TO DO : make the "Drag Start Range" configureable
      if (qlib::abs<int>(point.x-m_prevPt.x)<2 &&
          qlib::abs<int>(point.y-m_prevPt.y)<2 ) {
        return 0;
        //break;
      }

      m_nDragStart = DRAG_DRAG;
      ev.setType(InDevEvent::INDEV_DRAG_START);
      setUpMouseEvent(wParam, m_prevPt, ev);
      fireInDevEvent(ev);
    }
    
    ev.setType(InDevEvent::INDEV_DRAG_MOVE);
    setUpMouseEvent(wParam, point, ev);
    ev.setDeltaX(point.x - m_prevPt.x);
    ev.setDeltaY(point.y - m_prevPt.y);
    ev.setMoveX(point.x - m_startPt.x);
    ev.setMoveY(point.y - m_startPt.y);
    
    fireInDevEvent(ev);
    m_prevPt = point;
    return 0;
    //break;
  }
    
  case WM_LBUTTONDBLCLK: {
    POINTS point = MAKEPOINTS(lParam);
    UINT nFlags = wParam;
    // windows doesn't set this flag !!
    nFlags |= MK_LBUTTON;

    InDevEvent ev;
    setUpMouseEvent(nFlags, point, ev);

    ev.setType(InDevEvent::INDEV_LBTN_DBLCLICK);
    fireInDevEvent(ev);
    return 0;
    //break;
  }

  case WM_RBUTTONDBLCLK: {

    POINTS point = MAKEPOINTS(lParam);
    UINT nFlags = wParam;
    // windows doesn't set this flag !!
    nFlags |= MK_RBUTTON;

    InDevEvent ev;
    setUpMouseEvent(nFlags, point, ev);

    ev.setType(InDevEvent::INDEV_RBTN_DBLCLICK);
    fireInDevEvent(ev);
    return 0;
    //break;
  }

  case WM_MOUSEWHEEL: {
    UINT nFlags = GET_KEYSTATE_WPARAM(wParam);
    short zDelta = GET_WHEEL_DELTA_WPARAM(wParam);
    POINTS point = MAKEPOINTS(lParam);

    InDevEvent ev;
    setUpMouseEvent(nFlags, point, ev);

    ev.setType(InDevEvent::INDEV_WHEEL);
    ev.setDeltaX(zDelta);
    fireInDevEvent(ev);
    return 0;
    //break;
  }


  case WM_DESTROY: {
    break;
  }

  case WM_ERASEBKGND: {
    return 1;
  }

  case WM_SHOWWINDOW: {
    break;
  }

  default:
    break;
  }

  return -1; //::DefWindowProc(hWnd, msg, wParam, lParam);
}

/*
void WglView::setCursor(int nCursorID)
{
  switch (nCursorID) {
  default:
  case 0:
    m_hWndCursor = m_hStdCursor;
    break;
    
  case 1:
    m_hWndCursor = m_hWaitCursor;
    break;
    
  case 2:
    m_hWndCursor = m_hHandCursor;
    break;
    
  case 3:
    m_hWndCursor = m_hCrossCursor;
    break;
  }
  
  //::SetClassLong(m_hWnd, GCL_HCURSOR, (LONG)m_hWndCursor);
}
*/

/// Query hardware stereo capability
bool WglView::hasHWStereo() const
{
  LOG_DPRINTLN("WglView> hasHWStereo: %d", m_bHasQuadBuffer);
  return m_bHasQuadBuffer;
}

////////////////////////////////////////////

namespace qsys {
  //static
  qsys::View *View::createView()
  {
    qsys::View *pret = new WglView();
    MB_DPRINTLN("WglView created (%p, ID=%d)", pret, pret->getUID());
    return pret;
//    return NULL;
  }
}
