// -*-Mode: C++;-*-
//
//  OpenGL display context implementation
//
//  $Id: OglDisplayContext.cpp,v 1.26 2011/04/10 10:48:09 rishitani Exp $

#include <common.h>

#ifdef HAVE_GL_GL_H
#  include <GL/gl.h>
#elif defined(HAVE_OPENGL_GL_H)
#  include <OpenGL/gl.h>
#else
#  error no gl.h
#endif

#ifdef HAVE_GL_GLU_H
#  include <GL/glu.h>
#elif defined(HAVE_OPENGL_GLU_H)
#  include <OpenGL/glu.h>
#else
#  error no glu.h
#endif

#include "OglDisplayContext.hpp"
#include "OglDisplayList.hpp"

#include <gfx/TextRenderManager.hpp>
#include <gfx/PixelBuffer.hpp>
#include <gfx/SolidColor.hpp>
#include <gfx/Mesh.hpp>

using namespace sysdep;
using gfx::DisplayContext;

OglDisplayContext::OglDisplayContext(int sceneid)
{
  m_nSceneID = sceneid;
  m_pGluData = NULL;
  m_color = Vector4D(1.0, 1.0, 1.0, 1.0);
  m_nDetail = 5;
}

OglDisplayContext::~OglDisplayContext()
{
  if (m_pGluData!=NULL)
    ::gluDeleteQuadric((GLUquadricObj *)m_pGluData);
}

void OglDisplayContext::vertex(const Vector4D &v)
{
  ::glVertex3d(v.x(), v.y(), v.z());

  /*if (m_fDebugMode) {
    std::pair<Vector4D, Vector4D> nv;
    nv.second = m_curVertex;
    nv.first  = Vector4D(x, y, z);
    m_normlist.push_back(nv);
    }*/
}

void OglDisplayContext::vertex(double x, double y, double z)
{
  ::glVertex3d(x, y, z);
}

void OglDisplayContext::normal(const Vector4D &rn)
{
  ::glNormal3d(rn.x(), rn.y(), rn.z());

/*
  Vector4D nn = rn;
  double len = nn.length();
  if (len>=F_EPS4) {
    nn = nn.divide(len);
  }
  
  glNormal3d(nn.x(), nn.y(), nn.z());

  if (m_fDebugMode) {
    m_curVertex = Vector4D(x/2.0, y/2.0, z/2.0);
  }
*/
}

void OglDisplayContext::normal(double x, double y, double z)
{
  ::glNormal3d(x,y,z);
}

void OglDisplayContext::setMaterial(const LString &name)
{
  super_t::setMaterial(name);
  //if (name.isEmpty())
  //::glEnable(GL_COLOR_MATERIAL);
}

void OglDisplayContext::setMaterImpl(const LString &name)
{
/*
  if (m_curMater.equals(name))
    return;
  m_curMater = name;

  gfx::StyleMgr *pSM = gfx::StyleMgr::getInstance();
  double dvalue;
  GLfloat vv[] = {0.0, 0.0, 0.0, 1.0};

  if (name.isEmpty()) {
    glEnable(GL_COLOR_MATERIAL);
    return;
  }

  glDisable(GL_COLOR_MATERIAL);

  dvalue = pSM->getMaterial(name, gfx::Material::MAT_AMBIENT);
  if (dvalue>=-0.1) {
    vv[0] = dvalue * m_color.x();
    vv[1] = dvalue * m_color.y();
    vv[2] = dvalue * m_color.z();
    vv[3] = m_color.w();
    ::glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, vv);
  }

  dvalue = pSM->getMaterial(name, gfx::Material::MAT_DIFFUSE);
  if (dvalue>=-0.1) {
    vv[0] = dvalue * m_color.x();
    vv[1] = dvalue * m_color.y();
    vv[2] = dvalue * m_color.z();
    vv[3] = m_color.w();
    ::glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, vv);
  }

  dvalue = pSM->getMaterial(name, gfx::Material::MAT_SPECULAR);
  if (dvalue>=-0.1) {
    vv[0] = dvalue * m_color.x();
    vv[1] = dvalue * m_color.y();
    vv[2] = dvalue * m_color.z();
    vv[3] = m_color.w();
    ::glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, vv);
  }

  dvalue = pSM->getMaterial(name, gfx::Material::MAT_SHININESS);
  if (dvalue>=-0.1) {
    vv[0] = vv[1] = vv[2] = dvalue;
    ::glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, vv);
  }

  dvalue = pSM->getMaterial(name, gfx::Material::MAT_EMISSION);
  if (dvalue>=-0.1) {
    vv[0] = dvalue * m_color.x();
    vv[1] = dvalue * m_color.y();
    vv[2] = dvalue * m_color.z();
    vv[3] = m_color.w();
    ::glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, vv);
  }
*/
}

void OglDisplayContext::color(const ColorPtr &c)
{
  //::glColor4ub(c->r(), c->g(), c->b(), c->a());

  m_color.x() = c->fr();
  m_color.y() = c->fg();
  m_color.z() = c->fb();
  m_color.w() = c->fa() * getAlpha();

/*
  LString strmat = c->getMaterial();
  LString defmat = getMaterial();
  if (strmat.isEmpty()) {
    setMaterImpl(defmat);
  }
  else {
    setMaterImpl(strmat);
  }
*/
  
  ::glColor4d(m_color.x(), m_color.y(), m_color.z(), m_color.w());
}

void OglDisplayContext::color(double r, double g, double b, double a)
{
  m_color.x() = r;
  m_color.y() = g;
  m_color.z() = b;
  m_color.w() = a * getAlpha();
  //::glColor4d(r, g, b, a);
  ::glColor4d(m_color.x(), m_color.y(), m_color.z(), m_color.w());
}

void OglDisplayContext::color(double r, double g, double b)
{
  m_color.x() = r;
  m_color.y() = g;
  m_color.z() = b;
  m_color.w() = getAlpha();
  ::glColor4d(r, g, b, getAlpha());
}

void OglDisplayContext::setLineWidth(double lw)
{
  glLineWidth((float)lw);
}

void OglDisplayContext::setLineStipple(unsigned short pattern)
{
  if (pattern==0xFFFF)
    glDisable(GL_LINE_STIPPLE);
  else {
    glEnable(GL_LINE_STIPPLE);
    glLineStipple(1,pattern);
  }
}

void OglDisplayContext::setPointSize(double size)
{
  ::glPointSize(size);
}

void OglDisplayContext::startPoints()
{
  glBegin(GL_POINTS);
}

void OglDisplayContext::startPolygon()
{
  glBegin(GL_POLYGON);
}

void OglDisplayContext::startLines()
{
  glBegin(GL_LINES);
}

void OglDisplayContext::startLineStrip()
{
  glBegin(GL_LINE_STRIP);
}

void OglDisplayContext::startTriangleStrip()
{
  glBegin(GL_TRIANGLE_STRIP);
}

void OglDisplayContext::startTriangleFan()
{
  glBegin(GL_TRIANGLE_FAN);
}

void OglDisplayContext::startTriangles()
{
  glBegin(GL_TRIANGLES);
}

void OglDisplayContext::startQuadStrip()
{
  glBegin(GL_QUAD_STRIP);
}

void OglDisplayContext::startQuads()
{
  glBegin(GL_QUADS);
}

void OglDisplayContext::end()
{
  glEnd();

  /*  if (m_fDebugMode && m_normlist.size()>0) {
    glBegin(GL_LINES);
    std::list<std::pair<Vector4D, Vector4D> >::const_iterator iter =
      m_normlist.begin();
    for ( ; iter!=m_normlist.end(); iter++) {
      Vector4D p1 = (*iter).first;
      Vector4D p2 = p1+(*iter).second;
      glVertex3d(p1.x, p1.y, p1.z);
      glVertex3d(p2.x, p2.y, p2.z);
    }
    glEnd();
    m_normlist.erase(m_normlist.begin(), m_normlist.end());
  }
  */
}

void OglDisplayContext::pushMatrix()
{
  glPushMatrix();
}

void OglDisplayContext::multMatrix(const Matrix4D &mat)
{
  GLdouble m[16];

  m[0]  = mat.aij(1,1);
  m[4]  = mat.aij(1,2);
  m[8]  = mat.aij(1,3);
  m[12] = mat.aij(1,4);

  m[1]  = mat.aij(2,1);
  m[5]  = mat.aij(2,2);
  m[9]  = mat.aij(2,3);
  m[13] = mat.aij(2,4);

  m[2]  = mat.aij(3,1);
  m[6]  = mat.aij(3,2);
  m[10] = mat.aij(3,3);
  m[14] = mat.aij(3,4);

  m[3]  = mat.aij(4,1);
  m[7]  = mat.aij(4,2);
  m[11] = mat.aij(4,3);
  m[15] = mat.aij(4,4);

  glMultMatrixd(m);
}

void OglDisplayContext::loadMatrix(const Matrix4D &mat)
{
  GLdouble m[16];

  m[0]  = mat.aij(1,1);
  m[4]  = mat.aij(1,2);
  m[8]  = mat.aij(1,3);
  m[12] = mat.aij(1,4);

  m[1]  = mat.aij(2,1);
  m[5]  = mat.aij(2,2);
  m[9]  = mat.aij(2,3);
  m[13] = mat.aij(2,4);

  m[2]  = mat.aij(3,1);
  m[6]  = mat.aij(3,2);
  m[10] = mat.aij(3,3);
  m[14] = mat.aij(3,4);

  m[3]  = mat.aij(4,1);
  m[7]  = mat.aij(4,2);
  m[11] = mat.aij(4,3);
  m[15] = mat.aij(4,4);

  glLoadMatrixd(m);
}

void OglDisplayContext::popMatrix()
{
  glPopMatrix();
}

void OglDisplayContext::enableDepthTest(bool f)
{
  if (f)
    ::glDepthMask(GL_TRUE);
  else
    ::glDepthMask(GL_FALSE);
}

void OglDisplayContext::startHit(qlib::uid_t rend_uid)
{
  glLoadName(rend_uid);
  glPushName(-1);
}

void OglDisplayContext::endHit()
{
  glPopName();
}

void OglDisplayContext::drawPointHit(int nid, const Vector4D &pos)
{
  glLoadName(nid);
  //glColor3d(1.0, 1.0, 1.0);

  startPoints();
  vertex(pos);
  end();
}


void OglDisplayContext::loadName(int nameid)
{
  glLoadName(nameid);
}

void OglDisplayContext::pushName(int nameid)
{
  glPushName(nameid);
}

void OglDisplayContext::popName()
{
  glPopName();
}



void OglDisplayContext::setLighting(bool f)
{
  if (f)
    glEnable(GL_LIGHTING);
  else
    glDisable(GL_LIGHTING);
}

void OglDisplayContext::setCullFace(bool f/*=true*/)
{
  if (f)
    glEnable(GL_CULL_FACE);
  else
    glDisable(GL_CULL_FACE);
}

void OglDisplayContext::drawPixels(const Vector4D &pos,
                                   const gfx::PixelBuffer &data,
                                   const AbstractColor &col)
{
  glRasterPos3d(pos.x(), pos.y(), pos.z());

  if (data.getDepth()==8) {
    glPixelTransferf(GL_RED_BIAS, (float) col.fr());
    glPixelTransferf(GL_GREEN_BIAS, (float) col.fg());
    glPixelTransferf(GL_BLUE_BIAS, (float) col.fb());
    glDrawPixels(data.getWidth(), data.getHeight(),
                 GL_ALPHA, GL_UNSIGNED_BYTE, data.data());
    glPixelTransferf(GL_RED_BIAS, 0);
    glPixelTransferf(GL_GREEN_BIAS, 0);
    glPixelTransferf(GL_BLUE_BIAS, 0);
  }
  else if (data.getDepth()==1) {
    glColor4ub(col.r(), col.g(), col.b(), col.a());
    glBitmap(data.getWidth(), data.getHeight(),
             0, 0, 0, 0, data.data());
  }
}

void OglDisplayContext::drawString(const Vector4D &pos, const qlib::LString &str)
{
  gfx::TextRenderManager *pTRM = gfx::TextRenderManager::getInstance();
  if (pTRM==NULL) return;

  gfx::PixelBuffer pixbuf;
  if (!pTRM->renderText(str, pixbuf))
    return;

  gfx::SolidColor col(m_color);
  drawPixels(pos, pixbuf, col);
/*
  glRasterPos3d(pos.x(), pos.y(), pos.z());

  glPixelTransferf(GL_RED_BIAS, (float) m_color.x());
  glPixelTransferf(GL_GREEN_BIAS, (float) m_color.y());
  glPixelTransferf(GL_BLUE_BIAS, (float) m_color.z());

  glDrawPixels(width, height, GL_ALPHA, GL_UNSIGNED_BYTE, pixbuf);

  //glDrawPixels(checkImageWidth, checkImageHeight, GL_ALPHA, GL_UNSIGNED_BYTE, checkImage);
  //glDrawPixels(width, height, GL_RGBA, GL_UNSIGNED_BYTE, pixbuf);
  //glDrawPixels(checkImageWidth, checkImageHeight, GL_COLOR_INDEX, GL_UNSIGNED_BYTE, checkImage);
  //glDrawPixels(checkImageWidth, checkImageHeight, GL_LUMINANCE, GL_UNSIGNED_BYTE, checkImage);
*/
}

#if 0
void OglDisplayContext::drawString(const Vector4D &pos, const qlib::LString &str)
{
  const int checkImageWidth  = 80;
  const int checkImageHeight = 15;
  GLubyte checkImage[checkImageWidth*checkImageHeight]; //[2];
  
  for (int i=0; i<checkImageWidth; ++i) {
    for (int j=0; j<checkImageHeight; ++j) {
      //int c = ((((i&0x8)==0)^((j&0x8))==0))*255;
      checkImage[i+j*checkImageWidth] = (GLubyte) i;
      //checkImage[i][j][1] = (GLubyte) j*4;
      //checkImage[i][j][2] = (GLubyte) (i*j/4);
      //checkImage[i][j][3] = (GLubyte) (i*4);
    }
  }
  
  glRasterPos3d(pos.x(), pos.y(), pos.z());
  //glDrawPixels(checkImageWidth, checkImageHeight, GL_COLOR_INDEX, GL_UNSIGNED_BYTE, checkImage);
  //glDrawPixels(checkImageWidth, checkImageHeight, GL_LUMINANCE, GL_UNSIGNED_BYTE, checkImage);

  glPixelTransferf(GL_RED_BIAS, (float) m_color.x());
  glPixelTransferf(GL_GREEN_BIAS, (float) m_color.y());
  glPixelTransferf(GL_BLUE_BIAS, (float) m_color.z());
  glDrawPixels(checkImageWidth, checkImageHeight, GL_ALPHA, GL_UNSIGNED_BYTE, checkImage);
}
#endif


void OglDisplayContext::setPolygonMode(int id)
{
  switch (id) {
  default:
  case DisplayContext::POLY_FILL:
    //glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
    glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
    break;
  
  case DisplayContext::POLY_LINE:
    glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
    break;
  
  case DisplayContext::POLY_POINT:
    glPolygonMode(GL_FRONT_AND_BACK, GL_POINT);
    break;
  }  
}

////////////////////////////////////////////////
// DisplayCommand overloads for GL perf.

void OglDisplayContext::translate(const Vector4D &v)
{
  glTranslated(v.x(), v.y(), v.z());
}

void OglDisplayContext::scale(const Vector4D &v)
{
  glScaled(v.x(), v.y(), v.z());
}

void OglDisplayContext::loadIdent()
{
  glLoadIdentity();
}

/*void OglDisplayContext::rotate(const qlib::LQuat &q)
{
  multMatrix(q.toRotMatrix());
}*/

//////////////////////////////////////////////////////////////////
// Texture impl

/*
#include "OglTexture.hpp"

void OglDisplayContext::useTexture(const LTexture &tex)
{
  OglTextureRep *pRep = dynamic_cast<OglTextureRep *>(tex.getRep());
  if (pRep==NULL) return;
  glBindTexture(GL_TEXTURE_2D, pRep->getTexID());
  glEnable(GL_TEXTURE_2D);
}

void OglDisplayContext::unuseTexture()
{
  glDisable(GL_TEXTURE_2D);
}

void OglDisplayContext::texCoord(float fx, float fy)
{
  glTexCoord2d(fx, fy);
}
*/

//////////////////////////////////////////////////////////////////
// Display list impl

DisplayContext *OglDisplayContext::createDisplayList()
{
  OglDisplayList *pdl = new OglDisplayList(m_nSceneID);
  pdl->setAlpha(getAlpha());
  pdl->setMaterial(getMaterial());
  return pdl;
}

bool OglDisplayContext::canCreateDL() const
{
  return true;
}

void OglDisplayContext::callDisplayList(DisplayContext *pdl)
{
  OglDisplayList *psrc = dynamic_cast<OglDisplayList *>(pdl);
  if (psrc==NULL || !psrc->isValid())
    return;

  GLuint id = psrc->getID();
  if (id==0)
    return;

  glCallList(id);
}

bool OglDisplayContext::isCompatibleDL(DisplayContext *pdl) const
{
  OglDisplayList *psrc = dynamic_cast<OglDisplayList *>(pdl);
  if (psrc==NULL)
    return false;
  return true;
}

bool OglDisplayContext::isDisplayList() const
{
  return false;
}

bool OglDisplayContext::recordStart()
{
  return false;
}

void OglDisplayContext::recordEnd()
{
}

//////////////////////////////////////////////////////////////////
// Quadric object drawing impl

namespace {
GLUquadricObj *createQuadricObj()
{
  GLUquadricObj *quadObj;

  quadObj = gluNewQuadric(); 
  gluQuadricDrawStyle(quadObj, GLU_FILL); 
  gluQuadricOrientation(quadObj, GLU_OUTSIDE); 
  gluQuadricNormals(quadObj, GLU_SMOOTH); 
  return quadObj;
}

void setTransRot(const Vector4D &pos, const Vector4D &vec)
{
  double a, f, m[4][4];

  a = sqrt(vec.x() * vec.x() + vec.y() * vec.y());
  if (a > 1.0e-6) {
    f = 1.0 / a;
    m[0][0] = vec.x() * vec.z() * f;
    m[0][1] = vec.y() * vec.z() * f;
    m[0][2] = - a;
    m[0][3] = 0.0;
    m[1][0] = - vec.y() * f;
    m[1][1] = vec.x() * f;
    m[1][2] = 0.0;
    m[1][3] = 0.0;
    m[2][0] = vec.x();
    m[2][1] = vec.y();
    m[2][2] = vec.z();
    m[2][3] = 0.0;
    m[3][0] = pos.x();
    m[3][1] = pos.y();
    m[3][2] = pos.z();
    m[3][3] = 1.0;
    ::glMultMatrixd(&m[0][0]);
  } else {
    ::glTranslated(pos.x(), pos.y(), pos.z());
    if (vec.z()<0.0)
      ::glRotated(180.0, 1.0, 0.0, 0.0);
  }
}

} // namespace

void OglDisplayContext::sphere()
{
  if (m_pGluData==NULL) {
    m_pGluData = createQuadricObj();
  }

  ::gluSphere((GLUquadricObj *)m_pGluData, 1, (m_nDetail+1)*2, m_nDetail+1); 
}

void OglDisplayContext::cone(double r1, double r2,
                             const Vector4D &pos1, const Vector4D &pos2,
                             bool bCap)
{
  if (m_pGluData==NULL) {
    m_pGluData = createQuadricObj();
  }

  pushMatrix();

  Vector4D dv = pos2 - pos1;
  double len = dv.length();
  dv /= len;
  setTransRot(pos1, dv);

  gluCylinder((GLUquadricObj *)m_pGluData, r1, r2, len, (m_nDetail+1)*2, 1); 
  // gluCylinder((GLUquadricObj *)m_pGluData, r, r, len, 10, 1); 

  if (bCap) {
    if (r1>1.0e-4) {
      glPushMatrix();
      glRotated(180, 1, 0, 0);
      gluDisk((GLUquadricObj *)m_pGluData, 0.0, r1,
              (m_nDetail+1)*2, 1);
      glPopMatrix();
    }
    
    if (r2>1.0e-4) {
      glTranslated(0, 0, len);
      gluDisk((GLUquadricObj *)m_pGluData, 0.0, r2,
              (m_nDetail+1)*2, 1);
    }
  }

  popMatrix();
}

void OglDisplayContext::sphere(double r, const Vector4D &vec)
{
  pushMatrix();
  translate(vec);
  scale(Vector4D(r,r,r));
  sphere();
  popMatrix();
}

/*
void OglDisplayContext::cylinder(double r, const Vector4D &pos1, const Vector4D &pos2)
{
  cone(r, r, pos1, pos2, false);
}

void OglDisplayContext::cylinderCap(double r, const Vector4D &pos1, const Vector4D &pos2)
{
  cone(r, r, pos1, pos2, true);
}
*/

void OglDisplayContext::setDetail(int n)
{
  m_nDetail = n;
}

int OglDisplayContext::getDetail() const
{
  return m_nDetail;
}

void OglDisplayContext::drawMesh(const gfx::Mesh &mesh)
{
  const int nverts = mesh.getVertSize();
  const int nfaces = mesh.getFaceSize();

  glEnableClientState(GL_VERTEX_ARRAY);
  glEnableClientState(GL_NORMAL_ARRAY);
  glEnableClientState(GL_COLOR_ARRAY);

  const float *pverts = mesh.getFloatVerts();
  glVertexPointer(3, GL_FLOAT, 0, pverts);

  const float *pnorms = mesh.getFloatNorms();
  glNormalPointer(GL_FLOAT, 0, pnorms);

  unsigned char *pcols = new unsigned char[nverts*4];
  mesh.convRGBAByteCols(pcols, nverts*4);
  glColorPointer(4, GL_UNSIGNED_BYTE, 0, pcols);

  const int *pinds = mesh.getFaces();

  glDrawElements(GL_TRIANGLES, 3*nfaces, GL_UNSIGNED_INT, pinds);

  delete [] pcols;
}


